import { useState, useEffect, useRef } from "react";

/* ═══════════════════════════════════════════
   TOAST SYSTEM
═══════════════════════════════════════════ */
const TOAST_DURATION = 4500;
function ToastItem({ toast, onRemove }) {
  const [visible, setVisible] = useState(false);
  const [leaving, setLeaving] = useState(false);
  useEffect(() => {
    const t1 = setTimeout(() => setVisible(true), 10);
    const t2 = setTimeout(() => setLeaving(true), TOAST_DURATION - 400);
    const t3 = setTimeout(() => onRemove(toast.id), TOAST_DURATION);
    return () => { clearTimeout(t1); clearTimeout(t2); clearTimeout(t3); };
  }, []);
  const handleClose = () => { setLeaving(true); setTimeout(() => onRemove(toast.id), 380); };
  const cfg = {
    success: { bg: "#16a34a", icon: "✅" },
    error:   { bg: "#dc2626", icon: "❌" },
    warning: { bg: "#d97706", icon: "⚠️" },
    info:    { bg: "#1d4ed8", icon: "ℹ️" },
  }[toast.type] || { bg: "#1d4ed8", icon: "ℹ️" };
  return (
    <div style={{
      background: cfg.bg, borderRadius: 14, overflow: "hidden",
      boxShadow: "0 8px 32px rgba(0,0,0,0.35)", marginBottom: 10,
      minWidth: 290, maxWidth: 360,
      transform: visible && !leaving ? "translateX(0)" : "translateX(120%)",
      opacity: visible && !leaving ? 1 : 0,
      transition: leaving ? "transform 0.38s cubic-bezier(0.4,0,1,1),opacity 0.38s" : "transform 0.42s cubic-bezier(0.16,1,0.3,1),opacity 0.42s",
    }}>
      <div style={{ display:"flex", alignItems:"flex-start", gap:10, padding:"13px 14px" }}>
        <span style={{ fontSize:18, flexShrink:0 }}>{cfg.icon}</span>
        <span style={{ flex:1, color:"#fff", fontSize:13, fontWeight:600, lineHeight:1.5 }}>{toast.msg}</span>
        <button onClick={handleClose} style={{ color:"rgba(255,255,255,0.7)", background:"rgba(255,255,255,0.15)", border:"none", borderRadius:"50%", width:22, height:22, cursor:"pointer", fontSize:11, display:"flex", alignItems:"center", justifyContent:"center" }}>✕</button>
      </div>
      <div style={{ height:3, background:"rgba(0,0,0,0.2)" }}>
        <div style={{ height:"100%", background:"rgba(255,255,255,0.5)", animation:`shrink ${TOAST_DURATION}ms linear forwards` }} />
      </div>
    </div>
  );
}
function ToastContainer({ toasts, removeToast }) {
  return (
    <div style={{ position:"fixed", top:76, right:16, zIndex:9999, display:"flex", flexDirection:"column", alignItems:"flex-end", pointerEvents:"none" }}>
      {toasts.map(t => <div key={t.id} style={{ pointerEvents:"all" }}><ToastItem toast={t} onRemove={removeToast} /></div>)}
    </div>
  );
}
function useToast() {
  const [toasts, setToasts] = useState([]);
  const add = (msg, type="info") => { const id = Date.now()+Math.random(); setToasts(p=>[...p,{id,msg,type}]); };
  const remove = (id) => setToasts(p=>p.filter(t=>t.id!==id));
  return { toasts, add, remove };
}

/* ═══════════════════════════════════════════
   DATA
═══════════════════════════════════════════ */
const SERVICES = [
  { id:"tow",      icon:"🚛", label:"Tow Truck",        color:"#ef4444", bg:"#fef2f2", border:"#fecaca", eta:"20-35 min", price:799,  desc:"Professional towing to nearest garage or preferred destination. Available 24/7.", features:["Up to 50 km included","Any vehicle type","GPS tracked truck","Insurance accepted"] },
  { id:"puncture", icon:"🔧", label:"Tyre Puncture",    color:"#f97316", bg:"#fff7ed", border:"#fed7aa", eta:"15-25 min", price:399,  desc:"On-site tyre repair or replacement. Carry spare tyre installation service.", features:["On-site repair","Spare tyre fitting","All tyre sizes","Tubeless repair"] },
  { id:"fuel",     icon:"⛽", label:"Fuel Delivery",    color:"#eab308", bg:"#fefce8", border:"#fef08a", eta:"20-30 min", price:299,  desc:"Emergency fuel delivered to your location. Petrol, diesel & CNG available.", features:["Up to 5L delivered","Petrol & Diesel","Safe container","Same price as pump"] },
  { id:"battery",  icon:"🔋", label:"Battery Jump",     color:"#22c55e", bg:"#f0fdf4", border:"#bbf7d0", eta:"10-20 min", price:349,  desc:"Jump-start your dead battery or get instant battery replacement on-site.", features:["Jump-start service","Battery replacement","All vehicle types","New battery warranty"] },
  { id:"mechanic", icon:"🛠️", label:"On-Site Mechanic", color:"#3b82f6", bg:"#eff6ff", border:"#bfdbfe", eta:"25-40 min", price:599,  desc:"Certified mechanic dispatched to diagnose and fix minor on-road issues.", features:["Engine diagnostics","Minor repairs","Spark plug/belt","OBD scan included"] },
  { id:"lockout",  icon:"🔑", label:"Car Lockout",      color:"#8b5cf6", bg:"#f5f3ff", border:"#ddd6fe", eta:"15-25 min", price:449,  desc:"Locked out of your vehicle? Our technician will unlock it without damage.", features:["All car models","No damage method","24/7 availability","Key cutting service"] },
];

const PROVIDERS = [
  { id:1, name:"Rajesh Auto Works",    rating:4.9, reviews:312, dist:"1.2 km", eta:"12 min", emoji:"🔧", services:["tow","mechanic","battery"], city:"Ahmedabad", verified:true,  active:true,  badge:"Top Rated",    jobs:1240 },
  { id:2, name:"Speedy Rescue",        rating:4.8, reviews:198, dist:"2.1 km", eta:"18 min", emoji:"🚛", services:["tow","fuel","puncture"],    city:"Surat",      verified:true,  active:true,  badge:"Fast Response", jobs:876  },
  { id:3, name:"RoadSafe Solutions",   rating:4.7, reviews:445, dist:"3.0 km", eta:"22 min", emoji:"🛡️", services:["mechanic","lockout","battery"],city:"Vadodara",  verified:true,  active:false, badge:"Verified Pro",  jobs:2100 },
  { id:4, name:"QuickFix Motors",      rating:4.6, reviews:134, dist:"3.8 km", eta:"28 min", emoji:"⚡", services:["battery","fuel","puncture"], city:"Rajkot",     verified:false, active:true,  badge:"Affordable",    jobs:543  },
];

const MY_REQUESTS = [
  { id:"RR-20240122-001", service:"Tow Truck",        date:"2024-01-22", status:"Completed",  provider:"Rajesh Auto Works",  amount:799,  location:"NH-48 near Sola, Ahmedabad",   rating:5 },
  { id:"RR-20240115-002", service:"Battery Jump",     date:"2024-01-15", status:"Completed",  provider:"QuickFix Motors",    amount:349,  location:"CG Road, Ahmedabad",           rating:4 },
  { id:"RR-20240110-003", service:"Tyre Puncture",    date:"2024-01-10", status:"Cancelled",  provider:"Speedy Rescue",      amount:0,    location:"Satellite, Ahmedabad",         rating:null },
  { id:"RR-20240201-004", service:"On-Site Mechanic", date:"2024-02-01", status:"In Progress",provider:"RoadSafe Solutions", amount:599,  location:"SG Highway, Ahmedabad",        rating:null },
];

const FAQS = [
  { q:"How quickly can I expect help?",            a:"Average response time is 15–35 minutes depending on your location and selected service. Urban areas receive faster response." },
  { q:"Do you operate 24/7?",                      a:"Yes, RoadRescue operates 24 hours a day, 7 days a week including holidays. Emergency services never stop." },
  { q:"How is payment handled?",                   a:"You can pay via UPI, credit/debit card, or cash. Payment is collected after service is completed." },
  { q:"What areas do you cover?",                  a:"We currently cover all major cities and highways across Gujarat. National coverage is expanding monthly." },
  { q:"Is my vehicle covered for damage?",         a:"All our providers are insured. Any accidental damage during service is covered under our service guarantee." },
  { q:"Can I track my service provider in real-time?",a:"Yes! Once your request is accepted, you get a live tracking link via SMS. Watch them arrive on the map." },
];

const STATS = [
  { v:"50,000+", l:"Rescues Completed", ic:"🚗" },
  { v:"< 20 min", l:"Avg Response Time", ic:"⚡" },
  { v:"98.7%",   l:"Satisfaction Rate",  ic:"⭐" },
  { v:"24/7",    l:"Always Available",   ic:"🕐" },
];

/* ═══════════════════════════════════════════
   SHARED UI COMPONENTS
═══════════════════════════════════════════ */
function Modal({ open, onClose, title, children, size="md" }) {
  useEffect(() => { document.body.style.overflow = open ? "hidden" : ""; return () => { document.body.style.overflow = ""; }; }, [open]);
  if (!open) return null;
  const w = { sm:"max-w-sm", md:"max-w-lg", lg:"max-w-2xl", xl:"max-w-4xl" }[size];
  return (
    <div className="fixed inset-0 z-[200] flex items-end sm:items-center justify-center p-0 sm:p-4" onClick={onClose}>
      <div className="absolute inset-0 bg-black/60 backdrop-blur-sm" />
      <div className={`relative bg-white w-full ${w} rounded-t-3xl sm:rounded-3xl shadow-2xl max-h-[92vh] overflow-y-auto`}
        onClick={e=>e.stopPropagation()} style={{ animation:"slideUp 0.32s cubic-bezier(0.16,1,0.3,1)" }}>
        <div className="sticky top-0 bg-white z-10 flex items-center justify-between px-5 sm:px-6 py-4 border-b border-gray-100 rounded-t-3xl">
          <h3 className="font-display text-lg sm:text-xl font-bold text-gray-900">{title}</h3>
          <button onClick={onClose} className="w-8 h-8 rounded-full bg-gray-100 hover:bg-red-50 hover:text-red-500 flex items-center justify-center text-gray-500 transition-colors text-sm">✕</button>
        </div>
        <div className="px-5 sm:px-6 py-5">{children}</div>
      </div>
    </div>
  );
}

function StatusBadge({ status }) {
  const cfg = {
    "Completed":   "bg-green-100 text-green-700",
    "In Progress": "bg-amber-100 text-amber-700",
    "Cancelled":   "bg-red-100 text-red-700",
    "Pending":     "bg-blue-100 text-blue-700",
    "Dispatched":  "bg-purple-100 text-purple-700",
  };
  const icons = { "Completed":"✅","In Progress":"🔄","Cancelled":"❌","Pending":"⏳","Dispatched":"🚀" };
  return <span className={`${cfg[status]||"bg-gray-100 text-gray-600"} text-xs font-bold px-3 py-1.5 rounded-full`}>{icons[status]} {status}</span>;
}

function StarRating({ rating, interactive=false, onRate }) {
  const [hover, setHover] = useState(0);
  return (
    <div className="flex gap-0.5">
      {[1,2,3,4,5].map(i=>(
        <span key={i}
          className={`${interactive?"cursor-pointer text-xl":"text-sm"} transition-colors`}
          style={{ color: i<=(interactive?hover||rating:rating) ? "#f59e0b" : "#e5e7eb" }}
          onMouseEnter={()=>interactive&&setHover(i)}
          onMouseLeave={()=>interactive&&setHover(0)}
          onClick={()=>interactive&&onRate&&onRate(i)}
        >★</span>
      ))}
    </div>
  );
}

/* ═══════════════════════════════════════════
   LIVE TRACKER COMPONENT
═══════════════════════════════════════════ */
function LiveTracker({ request, onClose }) {
  const [progress, setProgress] = useState(0);
  const steps = ["Request Received","Provider Assigned","On the Way","Arrived","Service Complete"];
  const currentStep = Math.min(Math.floor(progress/25), 4);

  useEffect(() => {
    const interval = setInterval(() => {
      setProgress(p => p < 100 ? p + 1 : 100);
    }, 180);
    return () => clearInterval(interval);
  }, []);

  return (
    <div className="space-y-5">
      {/* Map placeholder */}
      <div className="relative h-48 sm:h-56 rounded-2xl overflow-hidden bg-gradient-to-br from-slate-800 to-slate-900 border border-slate-700">
        <div className="absolute inset-0 opacity-30" style={{ backgroundImage:"repeating-linear-gradient(0deg,transparent,transparent 40px,rgba(255,255,255,0.05) 40px,rgba(255,255,255,0.05) 41px),repeating-linear-gradient(90deg,transparent,transparent 40px,rgba(255,255,255,0.05) 40px,rgba(255,255,255,0.05) 41px)" }} />
        {/* Road lines */}
        <div className="absolute inset-0 flex items-center">
          <div className="w-full h-1 bg-white/20" />
        </div>
        {/* Animated truck */}
        <div className="absolute top-1/2 -translate-y-1/2 text-3xl transition-all duration-500" style={{ left:`${Math.min(progress*0.5+5, 50)}%` }}>🚛</div>
        {/* Destination pin */}
        <div className="absolute top-1/2 right-8 -translate-y-1/2 flex flex-col items-center">
          <div className="text-2xl animate-bounce">📍</div>
          <div className="bg-red-500 text-white text-xs px-2 py-0.5 rounded-full font-bold mt-1">YOU</div>
        </div>
        {/* ETA overlay */}
        <div className="absolute top-3 left-3 bg-black/60 backdrop-blur-sm text-white rounded-xl px-3 py-2">
          <div className="text-xs text-gray-300">ETA</div>
          <div className="font-black text-lg text-green-400">{Math.max(1, Math.ceil((100-progress)/5))} min</div>
        </div>
        <div className="absolute top-3 right-3 bg-green-500 text-white rounded-xl px-3 py-2 text-xs font-bold flex items-center gap-1">
          <span className="w-2 h-2 bg-white rounded-full animate-pulse" />LIVE
        </div>
      </div>

      {/* Provider info */}
      <div className="bg-slate-50 border border-slate-200 rounded-2xl p-4 flex items-center gap-4">
        <div className="w-12 h-12 bg-red-100 rounded-2xl flex items-center justify-center text-2xl">🔧</div>
        <div className="flex-1">
          <div className="font-bold text-gray-900">{request.provider}</div>
          <div className="text-xs text-gray-500">⭐ 4.9 · 🚗 {request.vehicle || "KA-01-AB-1234"}</div>
        </div>
        <a href="tel:+919876543210" className="bg-green-500 hover:bg-green-600 text-white px-3 py-2 rounded-xl text-xs font-bold transition-colors flex items-center gap-1">
          📞 Call
        </a>
      </div>

      {/* Progress steps */}
      <div>
        <div className="flex justify-between mb-2">
          <span className="text-xs text-gray-500 font-medium">Progress</span>
          <span className="text-xs font-bold text-red-600">{progress}%</span>
        </div>
        <div className="w-full h-2 bg-gray-100 rounded-full overflow-hidden mb-4">
          <div className="h-full bg-gradient-to-r from-red-500 to-orange-500 rounded-full transition-all duration-300" style={{ width:`${progress}%` }} />
        </div>
        <div className="flex flex-col gap-2">
          {steps.map((s,i) => (
            <div key={s} className={`flex items-center gap-3 py-2 px-3 rounded-xl transition-all ${i===currentStep?"bg-red-50 border border-red-200":i<currentStep?"bg-green-50 border border-green-100":"bg-gray-50 border border-gray-100"}`}>
              <div className={`w-7 h-7 rounded-full flex items-center justify-center text-xs font-black flex-shrink-0 ${i<currentStep?"bg-green-500 text-white":i===currentStep?"bg-red-500 text-white":"bg-gray-200 text-gray-400"}`}>
                {i<currentStep?"✓":i+1}
              </div>
              <span className={`text-sm font-medium ${i===currentStep?"text-red-700 font-bold":i<currentStep?"text-green-700":"text-gray-400"}`}>{s}</span>
              {i===currentStep && <span className="ml-auto text-xs text-red-500 animate-pulse font-bold">●  Active</span>}
            </div>
          ))}
        </div>
      </div>

      {progress===100 && (
        <div className="bg-green-500 text-white rounded-2xl p-4 text-center">
          <div className="text-3xl mb-2">🎉</div>
          <div className="font-bold text-lg">Service Complete!</div>
          <div className="text-green-100 text-sm">Please rate your experience below</div>
        </div>
      )}
    </div>
  );
}

/* ═══════════════════════════════════════════
   SERVICE REQUEST WIZARD
═══════════════════════════════════════════ */
function RequestWizard({ preService, onClose, onSubmit }) {
  const [step, setStep] = useState(preService ? 2 : 1);
  const [selectedService, setSelectedService] = useState(preService || null);
  const [selectedProvider, setSelectedProvider] = useState(null);
  const [form, setForm] = useState({ name:"", phone:"", vehicle:"", plateNo:"", location:"", landmark:"", notes:"", emergency:false });
  const [errors, setErrors] = useState({});
  const [locating, setLocating] = useState(false);

  const inp = "w-full bg-gray-50 border-2 border-gray-200 rounded-xl px-4 py-3 text-sm focus:border-red-400 focus:bg-white outline-none transition-all";
  const lbl = "block text-xs font-bold text-gray-500 mb-1.5 uppercase tracking-wide";

  const validate = () => {
    const e = {};
    if (!form.name.trim()) e.name = "Full name required";
    if (!form.phone.match(/^\d{10}$/)) e.phone = "Valid 10-digit number required";
    if (!form.vehicle) e.vehicle = "Vehicle type required";
    if (!form.plateNo.trim()) e.plateNo = "Plate number required";
    if (!form.location.trim()) e.location = "Location required";
    setErrors(e);
    return !Object.keys(e).length;
  };

  const detectLocation = () => {
    setLocating(true);
    setTimeout(() => {
      setForm(f=>({...f, location:"NH-48, near Sarkhej Raska, Ahmedabad - 380055", landmark:"Near Zydus Hospital"}));
      setLocating(false);
    }, 1800);
  };

  const availableProviders = PROVIDERS.filter(p => p.active && p.services.includes(selectedService?.id));

  return (
    <div>
      {/* Progress bar */}
      <div className="flex items-center gap-2 mb-6">
        {[1,2,3,4].map((s,i)=>(
          <div key={s} className="flex items-center gap-2 flex-1">
            <div className={`w-8 h-8 rounded-full flex items-center justify-center text-xs font-black transition-all flex-shrink-0 ${step>s?"bg-green-500 text-white":step===s?"bg-red-500 text-white":"bg-gray-200 text-gray-400"}`}>
              {step>s?"✓":s}
            </div>
            {i<3 && <div className={`flex-1 h-1 rounded-full transition-all ${step>s?"bg-green-400":"bg-gray-200"}`} />}
          </div>
        ))}
      </div>
      <div className="flex justify-between text-xs text-gray-400 mb-6 -mt-3">
        {["Service","Details","Provider","Confirm"].map(l=><span key={l} className="font-medium">{l}</span>)}
      </div>

      {/* STEP 1 — Choose Service */}
      {step===1 && (
        <div>
          <h4 className="font-display text-lg font-bold mb-1">What do you need?</h4>
          <p className="text-gray-400 text-sm mb-5">Select the type of emergency assistance required.</p>
          <div className="grid grid-cols-2 sm:grid-cols-3 gap-3">
            {SERVICES.map(s=>(
              <button key={s.id} onClick={()=>setSelectedService(s)}
                className={`p-3 sm:p-4 rounded-2xl border-2 text-left transition-all hover:-translate-y-0.5 ${selectedService?.id===s.id?"shadow-lg":"border-gray-100 hover:border-gray-200 bg-gray-50"}`}
                style={selectedService?.id===s.id?{borderColor:s.color,background:s.bg}:{}} >
                <div className="text-2xl sm:text-3xl mb-2">{s.icon}</div>
                <div className="font-bold text-xs sm:text-sm leading-tight" style={selectedService?.id===s.id?{color:s.color}:{}}>{s.label}</div>
                <div className="text-xs text-gray-400 mt-1">⏱️ {s.eta}</div>
              </button>
            ))}
          </div>
          {selectedService && (
            <div className="mt-4 p-4 rounded-2xl border" style={{ background:selectedService.bg, borderColor:selectedService.border }}>
              <div className="flex items-start gap-3">
                <span className="text-2xl">{selectedService.icon}</span>
                <div>
                  <div className="font-bold text-sm">{selectedService.label}</div>
                  <div className="text-xs text-gray-500 mt-0.5">{selectedService.desc}</div>
                  <div className="flex flex-wrap gap-1 mt-2">
                    {selectedService.features.map(f=><span key={f} className="text-xs px-2 py-0.5 bg-white/70 rounded-full text-gray-600">✓ {f}</span>)}
                  </div>
                </div>
              </div>
            </div>
          )}
          <button onClick={()=>selectedService&&setStep(2)} disabled={!selectedService}
            className={`w-full mt-5 py-3.5 rounded-2xl font-black text-white transition-all ${selectedService?"bg-red-600 hover:bg-red-700 shadow-lg hover:-translate-y-0.5":"bg-gray-200 text-gray-400 cursor-not-allowed"}`}>
            Continue →
          </button>
        </div>
      )}

      {/* STEP 2 — Your Details */}
      {step===2 && (
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <h4 className="font-display text-lg font-bold">Your Details</h4>
            <button onClick={()=>setStep(1)} className="text-xs text-red-500 font-bold">← Change Service</button>
          </div>
          {selectedService && (
            <div className="flex items-center gap-3 p-3 rounded-xl border" style={{ background:selectedService.bg, borderColor:selectedService.border }}>
              <span className="text-xl">{selectedService.icon}</span>
              <div className="font-bold text-sm" style={{ color:selectedService.color }}>{selectedService.label}</div>
              <div className="ml-auto font-black" style={{ color:selectedService.color }}>₹{selectedService.price}</div>
            </div>
          )}
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div><label className={lbl}>Full Name *</label><input className={inp} placeholder="Your full name" value={form.name} onChange={e=>setForm({...form,name:e.target.value})} />{errors.name&&<p className="text-xs text-red-500 mt-1">{errors.name}</p>}</div>
            <div><label className={lbl}>Phone *</label><input className={inp} placeholder="10-digit mobile" value={form.phone} onChange={e=>setForm({...form,phone:e.target.value})} maxLength={10} />{errors.phone&&<p className="text-xs text-red-500 mt-1">{errors.phone}</p>}</div>
            <div>
              <label className={lbl}>Vehicle Type *</label>
              <select className={inp} value={form.vehicle} onChange={e=>setForm({...form,vehicle:e.target.value})}>
                <option value="">Select type</option>
                {["Car","Bike","Truck","Bus","SUV","Auto Rickshaw"].map(v=><option key={v}>{v}</option>)}
              </select>
              {errors.vehicle&&<p className="text-xs text-red-500 mt-1">{errors.vehicle}</p>}
            </div>
            <div><label className={lbl}>Plate Number *</label><input className={inp} placeholder="e.g. GJ-01-AB-1234" value={form.plateNo} onChange={e=>setForm({...form,plateNo:e.target.value.toUpperCase()})} />{errors.plateNo&&<p className="text-xs text-red-500 mt-1">{errors.plateNo}</p>}</div>
          </div>
          <div>
            <label className={lbl}>Your Current Location *</label>
            <div className="flex gap-2">
              <input className={`${inp} flex-1`} placeholder="Enter your location or use GPS" value={form.location} onChange={e=>setForm({...form,location:e.target.value})} />
              <button onClick={detectLocation} disabled={locating}
                className="bg-red-600 hover:bg-red-700 text-white px-3 py-2.5 rounded-xl font-bold text-xs transition-all flex-shrink-0 flex items-center gap-1">
                {locating?<span className="animate-spin">⌛</span>:"📍"} {locating?"...":"GPS"}
              </button>
            </div>
            {errors.location&&<p className="text-xs text-red-500 mt-1">{errors.location}</p>}
          </div>
          <div><label className={lbl}>Landmark (Optional)</label><input className={inp} placeholder="Near hospital, mall, etc." value={form.landmark} onChange={e=>setForm({...form,landmark:e.target.value})} /></div>
          <div><label className={lbl}>Additional Notes</label><textarea className={inp} rows={2} placeholder="Describe the issue in detail..." value={form.notes} onChange={e=>setForm({...form,notes:e.target.value})} /></div>
          <label className="flex items-center gap-3 cursor-pointer p-3 bg-red-50 border-2 border-red-200 rounded-xl">
            <input type="checkbox" checked={form.emergency} onChange={e=>setForm({...form,emergency:e.target.checked})} className="w-4 h-4 accent-red-600" />
            <div>
              <div className="font-bold text-sm text-red-700">🚨 Mark as Emergency</div>
              <div className="text-xs text-red-500">Priority response for accident or dangerous situation</div>
            </div>
          </label>
          <button onClick={()=>{ if(validate()) setStep(3); }} className="w-full py-3.5 rounded-2xl bg-red-600 hover:bg-red-700 text-white font-black shadow-lg transition-all hover:-translate-y-0.5">
            Choose Provider →
          </button>
        </div>
      )}

      {/* STEP 3 — Choose Provider */}
      {step===3 && (
        <div>
          <div className="flex items-center justify-between mb-5">
            <h4 className="font-display text-lg font-bold">Nearby Providers</h4>
            <button onClick={()=>setStep(2)} className="text-xs text-red-500 font-bold">← Edit Details</button>
          </div>
          {availableProviders.length===0?(
            <div className="text-center py-10">
              <div className="text-5xl mb-3">😕</div>
              <div className="font-bold text-gray-700">No providers available right now</div>
              <div className="text-gray-400 text-sm mt-1">Try again in a few minutes or call our helpline</div>
            </div>
          ):(
            <div className="space-y-3">
              {availableProviders.map(p=>(
                <button key={p.id} onClick={()=>setSelectedProvider(p)}
                  className={`w-full text-left p-4 rounded-2xl border-2 transition-all ${selectedProvider?.id===p.id?"border-red-400 bg-red-50 shadow-md":"border-gray-100 hover:border-gray-200 bg-gray-50"}`}>
                  <div className="flex items-start gap-3">
                    <div className="w-12 h-12 rounded-2xl bg-white border border-gray-200 flex items-center justify-center text-2xl flex-shrink-0">{p.emoji}</div>
                    <div className="flex-1 min-w-0">
                      <div className="flex items-start justify-between gap-2">
                        <div className="font-bold text-sm text-gray-900">{p.name}</div>
                        <div className={`text-xs font-bold px-2 py-0.5 rounded-full flex-shrink-0 ${selectedProvider?.id===p.id?"bg-red-500 text-white":"bg-gray-200 text-gray-600"}`}>{p.badge}</div>
                      </div>
                      <div className="flex items-center gap-3 mt-1 text-xs text-gray-400">
                        <span className="text-amber-400">{"★".repeat(Math.floor(p.rating))}</span>
                        <span>{p.rating} ({p.reviews})</span>
                        <span>·</span><span>📍 {p.dist}</span>
                        <span>·</span><span>⏱️ {p.eta}</span>
                      </div>
                    </div>
                  </div>
                </button>
              ))}
            </div>
          )}
          <button onClick={()=>selectedProvider&&setStep(4)} disabled={!selectedProvider}
            className={`w-full mt-5 py-3.5 rounded-2xl font-black text-white transition-all ${selectedProvider?"bg-red-600 hover:bg-red-700 shadow-lg hover:-translate-y-0.5":"bg-gray-200 text-gray-400 cursor-not-allowed"}`}>
            Review & Confirm →
          </button>
        </div>
      )}

      {/* STEP 4 — Confirm */}
      {step===4 && (
        <div>
          <div className="flex items-center justify-between mb-5">
            <h4 className="font-display text-lg font-bold">Confirm Request</h4>
            <button onClick={()=>setStep(3)} className="text-xs text-red-500 font-bold">← Change Provider</button>
          </div>
          {/* Service summary */}
          <div className="p-4 rounded-2xl border mb-4" style={{ background:selectedService?.bg, borderColor:selectedService?.border }}>
            <div className="flex items-center gap-3 mb-3">
              <span className="text-3xl">{selectedService?.icon}</span>
              <div>
                <div className="font-black text-base" style={{ color:selectedService?.color }}>{selectedService?.label}</div>
                <div className="text-xs text-gray-500">⏱️ ETA: {selectedService?.eta}</div>
              </div>
              <div className="ml-auto font-black text-xl" style={{ color:selectedService?.color }}>₹{selectedService?.price}</div>
            </div>
          </div>
          {/* Details grid */}
          <div className="grid grid-cols-2 gap-3 mb-4">
            {[
              { l:"Name",       v:form.name },
              { l:"Phone",      v:form.phone },
              { l:"Vehicle",    v:`${form.vehicle} · ${form.plateNo}` },
              { l:"Provider",   v:selectedProvider?.name },
              { l:"Location",   v:form.location },
              { l:"Est. Arrive",v:selectedProvider?.eta },
            ].map(r=>(
              <div key={r.l} className="bg-gray-50 rounded-xl p-3">
                <div className="text-xs text-gray-400">{r.l}</div>
                <div className="font-bold text-xs sm:text-sm text-gray-800 mt-0.5 break-words">{r.v||"—"}</div>
              </div>
            ))}
          </div>
          {form.landmark && <div className="bg-gray-50 rounded-xl p-3 mb-4"><div className="text-xs text-gray-400">Landmark</div><div className="font-bold text-sm text-gray-800">{form.landmark}</div></div>}
          {form.emergency && <div className="bg-red-50 border border-red-200 rounded-xl p-3 mb-4 flex items-center gap-2"><span>🚨</span><div className="text-red-700 font-bold text-sm">Marked as Emergency — Priority dispatch</div></div>}
          {form.notes && <div className="bg-yellow-50 border border-yellow-200 rounded-xl p-3 mb-4"><div className="text-xs text-yellow-600 font-bold">Notes</div><div className="text-sm text-gray-700 mt-0.5">{form.notes}</div></div>}
          <div className="bg-slate-900 text-white rounded-2xl p-4 mb-5 flex justify-between items-center">
            <div><div className="text-xs text-slate-400">Total Amount</div><div className="font-black text-2xl">₹{selectedService?.price}</div><div className="text-xs text-slate-400">Pay after service</div></div>
            <div className="text-4xl">{selectedService?.icon}</div>
          </div>
          <button onClick={()=>onSubmit({service:selectedService,provider:selectedProvider,form})}
            className="w-full py-4 rounded-2xl bg-red-600 hover:bg-red-700 text-white font-black text-base shadow-2xl transition-all hover:-translate-y-0.5">
            🚀 Send Rescue Request
          </button>
          <p className="text-center text-xs text-gray-400 mt-3">By confirming, you agree to our Terms of Service. Help is on the way!</p>
        </div>
      )}
    </div>
  );
}

/* ═══════════════════════════════════════════
   PAGES
═══════════════════════════════════════════ */

// HOME PAGE
function HomePage({ onNavigate, onEmergency }) {
  const [heroAnim, setHeroAnim] = useState(false);
  const [activeService, setActiveService] = useState(null);
  useEffect(()=>{ setTimeout(()=>setHeroAnim(true),80); },[]);

  return (
    <div>
      {/* HERO */}
      <section className="relative min-h-screen flex items-center overflow-hidden" style={{ background:"linear-gradient(135deg,#0a0a0a 0%,#1a0505 50%,#0d0d0d 100%)" }}>
        {/* Grid texture */}
        <div className="absolute inset-0 opacity-10" style={{ backgroundImage:"linear-gradient(rgba(255,255,255,0.1) 1px,transparent 1px),linear-gradient(90deg,rgba(255,255,255,0.1) 1px,transparent 1px)", backgroundSize:"60px 60px" }} />
        {/* Red glow */}
        <div className="absolute top-0 right-0 w-96 h-96 rounded-full opacity-20 blur-3xl" style={{ background:"radial-gradient(circle,#ef4444,transparent)" }} />
        <div className="absolute bottom-0 left-0 w-64 h-64 rounded-full opacity-15 blur-3xl" style={{ background:"radial-gradient(circle,#f97316,transparent)" }} />

        {/* Floating icons */}
        {["🚗","🚛","🔧","⛽","🔋","🔑"].map((e,i)=>(
          <div key={i} className="absolute select-none opacity-10 text-4xl" style={{ left:`${5+i*16}%`, top:`${15+(i%3)*25}%`, animation:`floatY ${3+i*0.5}s ease-in-out infinite`, animationDelay:`${i*0.6}s` }}>{e}</div>
        ))}

        <div className="relative z-10 max-w-6xl mx-auto px-4 sm:px-6 pt-24 pb-20 sm:pt-32 grid md:grid-cols-2 gap-10 items-center">
          <div className={`transition-all duration-700 ${heroAnim?"opacity-100 translate-y-0":"opacity-0 translate-y-8"}`}>
            <div className="inline-flex items-center gap-2 bg-red-500/20 text-red-400 border border-red-500/30 px-4 py-2 rounded-full text-xs font-bold mb-6 uppercase tracking-widest">
              <span className="w-2 h-2 bg-red-500 rounded-full animate-pulse" /> 24/7 Emergency Assistance
            </div>
            <h1 className="font-display text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-black text-white leading-tight mb-5">
              Stranded?<br />
              <span className="text-transparent bg-clip-text" style={{ backgroundImage:"linear-gradient(90deg,#ef4444,#f97316)" }}>We're Coming.</span>
            </h1>
            <p className="text-gray-400 text-base sm:text-lg leading-relaxed mb-8 max-w-md">
              Instant roadside rescue for breakdowns, punctures, fuel emergencies, battery failures and more. Help dispatched in minutes.
            </p>
            <div className="flex flex-col sm:flex-row gap-3">
              <button onClick={onEmergency} className="flex items-center justify-center gap-2 px-8 py-4 rounded-2xl font-black text-white text-base shadow-2xl transition-all hover:-translate-y-1 hover:shadow-red-500/40 animate-pulse-slow" style={{ background:"linear-gradient(135deg,#dc2626,#ef4444)" }}>
                🚨 Request Help NOW
              </button>
              <button onClick={()=>onNavigate("services")} className="px-8 py-4 rounded-2xl font-bold text-white border border-white/20 bg-white/5 hover:bg-white/10 transition-all hover:-translate-y-1">
                View All Services →
              </button>
            </div>
            <div className="mt-8 flex items-center gap-4 text-xs text-gray-500">
              <span className="flex items-center gap-1"><span className="text-green-400">●</span> {PROVIDERS.filter(p=>p.active).length} providers online near you</span>
              <span>·</span>
              <span>Avg response: 18 min</span>
            </div>
          </div>

          <div className={`hidden md:flex justify-center items-center transition-all duration-700 delay-200 ${heroAnim?"opacity-100 translate-x-0":"opacity-0 translate-x-10"}`}>
            <div className="relative">
              <div className="w-72 h-72 rounded-full border border-red-500/20 flex items-center justify-center" style={{ background:"radial-gradient(circle,rgba(239,68,68,0.15),transparent)" }}>
                <div className="text-[140px]" style={{ animation:"floatY 3s ease-in-out infinite" }}>🚛</div>
              </div>
              {[{e:"⚡",pos:"top-4 right-4"},{e:"🔧",pos:"bottom-8 left-4"},{e:"⛽",pos:"top-1/2 -right-6"}].map((f,i)=>(
                <div key={i} className={`absolute ${f.pos} w-14 h-14 bg-white/10 backdrop-blur-sm border border-white/20 rounded-2xl flex items-center justify-center text-2xl`} style={{ animation:`floatY ${2.5+i}s ease-in-out infinite`, animationDelay:`${i*0.7}s` }}>{f.e}</div>
              ))}
              <div className="absolute -bottom-4 right-0 bg-green-500 text-white rounded-2xl px-4 py-2.5 shadow-xl">
                <div className="text-xs">Response Time</div>
                <div className="font-black text-xl">~18 min ⚡</div>
              </div>
            </div>
          </div>
        </div>

        <div className="absolute bottom-0 left-0 right-0">
          <svg viewBox="0 0 1440 60" fill="#f9fafb" className="w-full"><path d="M0,40 Q360,0 720,40 Q1080,80 1440,40 L1440 60 L0 60Z"/></svg>
        </div>
      </section>

      {/* STATS */}
      <section className="bg-gray-50 py-10 border-b border-gray-100">
        <div className="max-w-5xl mx-auto px-4 sm:px-6 grid grid-cols-2 md:grid-cols-4 gap-6 text-center">
          {STATS.map(s=>(
            <div key={s.l} className="p-3">
              <div className="text-3xl mb-2">{s.ic}</div>
              <div className="font-display text-2xl sm:text-3xl font-black text-red-600">{s.v}</div>
              <div className="text-gray-500 text-xs mt-1">{s.l}</div>
            </div>
          ))}
        </div>
      </section>

      {/* SERVICES PREVIEW */}
      <section className="py-16 sm:py-24 bg-white">
        <div className="max-w-6xl mx-auto px-4 sm:px-6">
          <div className="text-center mb-10 sm:mb-14">
            <span className="text-red-600 font-bold text-xs uppercase tracking-widest">Emergency Services</span>
            <h2 className="font-display text-3xl sm:text-4xl md:text-5xl font-black mt-2 mb-3">How Can We Help?</h2>
            <p className="text-gray-500 max-w-xl mx-auto text-sm sm:text-base">Select your emergency below. Certified help dispatched within minutes, 24/7.</p>
          </div>
          <div className="grid grid-cols-2 sm:grid-cols-3 gap-4 sm:gap-5">
            {SERVICES.map(s=>(
              <button key={s.id} onClick={()=>onEmergency(s)}
                className="text-left p-4 sm:p-6 rounded-3xl border-2 transition-all hover:-translate-y-1 hover:shadow-xl group"
                style={{ borderColor:activeService===s.id?s.color:"#f1f5f9", background:activeService===s.id?s.bg:"white" }}
                onMouseEnter={()=>setActiveService(s.id)} onMouseLeave={()=>setActiveService(null)}>
                <div className="text-4xl sm:text-5xl mb-3 sm:mb-4 group-hover:scale-110 transition-transform">{s.icon}</div>
                <h3 className="font-display font-bold text-sm sm:text-base mb-1" style={{ color:s.color }}>{s.label}</h3>
                <p className="text-gray-400 text-xs leading-relaxed mb-3 hidden sm:block line-clamp-2">{s.desc}</p>
                <div className="flex items-center justify-between">
                  <span className="text-xs font-bold text-gray-400">⏱️ {s.eta}</span>
                  <span className="font-black text-sm" style={{ color:s.color }}>₹{s.price}</span>
                </div>
              </button>
            ))}
          </div>
        </div>
      </section>

      {/* HOW IT WORKS */}
      <section className="py-16 sm:py-24 bg-gray-50">
        <div className="max-w-5xl mx-auto px-4 sm:px-6">
          <div className="text-center mb-10 sm:mb-14">
            <span className="text-red-600 font-bold text-xs uppercase tracking-widest">Simple Process</span>
            <h2 className="font-display text-3xl sm:text-4xl font-black mt-2">Help in 4 Steps</h2>
          </div>
          <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-6">
            {[
              { n:"01", ic:"📍", t:"Share Location",  d:"Open app & share your GPS location or type your address." },
              { n:"02", ic:"🔧", t:"Select Service",  d:"Choose the type of emergency help you need." },
              { n:"03", ic:"🚛", t:"Provider Dispatched", d:"Nearest certified provider is dispatched immediately." },
              { n:"04", ic:"✅", t:"Get Rescued",      d:"Problem fixed on-site. Pay only after completion." },
            ].map(step=>(
              <div key={step.n} className="text-center group">
                <div className="w-16 h-16 sm:w-20 sm:h-20 mx-auto mb-4 rounded-3xl bg-white border-2 border-gray-200 flex items-center justify-center text-3xl sm:text-4xl group-hover:border-red-300 group-hover:bg-red-50 transition-all group-hover:-translate-y-1">
                  {step.ic}
                </div>
                <div className="text-red-500 font-black text-xs uppercase tracking-widest mb-2">{step.n}</div>
                <h3 className="font-bold text-sm sm:text-base mb-2">{step.t}</h3>
                <p className="text-gray-400 text-xs sm:text-sm leading-relaxed">{step.d}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* PROVIDERS PREVIEW */}
      <section className="py-16 sm:py-24 bg-white">
        <div className="max-w-6xl mx-auto px-4 sm:px-6">
          <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4 mb-8">
            <div>
              <span className="text-red-600 font-bold text-xs uppercase tracking-widest">Available Now</span>
              <h2 className="font-display text-3xl sm:text-4xl font-black mt-1">Nearby Providers</h2>
            </div>
            <button onClick={()=>onNavigate("providers")} className="text-red-600 font-bold text-sm hover:text-red-700">View All →</button>
          </div>
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-5">
            {PROVIDERS.filter(p=>p.active).slice(0,2).map(p=><ProviderCard key={p.id} provider={p} onBook={()=>onEmergency(null)} />)}
          </div>
        </div>
      </section>

      {/* EMERGENCY BANNER */}
      <section className="py-14 sm:py-20" style={{ background:"linear-gradient(135deg,#dc2626,#b91c1c)" }}>
        <div className="max-w-4xl mx-auto px-4 sm:px-6 text-center text-white">
          <div className="text-5xl sm:text-6xl mb-5 animate-bounce">🚨</div>
          <h2 className="font-display text-3xl sm:text-4xl md:text-5xl font-black mb-4">In an Emergency?</h2>
          <p className="text-red-200 mb-8 text-sm sm:text-base">Don't wait. Request help immediately. Our team is on standby 24/7 across Gujarat.</p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button onClick={onEmergency} className="bg-white text-red-600 px-10 py-4 rounded-2xl font-black text-base sm:text-lg hover:bg-red-50 shadow-2xl transition-all hover:-translate-y-1">
              🚨 Request Rescue Now
            </button>
            <a href="tel:+918000123456" className="border-2 border-white/40 text-white px-10 py-4 rounded-2xl font-bold hover:bg-white/10 transition-all hover:-translate-y-1 flex items-center justify-center gap-2">
              📞 Call Helpline
            </a>
          </div>
        </div>
      </section>
    </div>
  );
}

// PROVIDER CARD
function ProviderCard({ provider, onBook }) {
  return (
    <div className="bg-white rounded-3xl border border-gray-100 shadow-sm hover:-translate-y-1 hover:shadow-xl transition-all duration-300 p-5 sm:p-6">
      <div className="flex items-start gap-4 mb-4">
        <div className="w-14 h-14 bg-gray-100 rounded-2xl flex items-center justify-center text-3xl flex-shrink-0">{provider.emoji}</div>
        <div className="flex-1 min-w-0">
          <div className="flex items-start justify-between gap-2">
            <h3 className="font-bold text-base text-gray-900">{provider.name}</h3>
            <span className={`text-xs font-bold px-2.5 py-1 rounded-full flex-shrink-0 ${provider.active?"bg-green-100 text-green-700":"bg-gray-100 text-gray-500"}`}>
              {provider.active?"● Online":"○ Offline"}
            </span>
          </div>
          <div className="flex items-center gap-2 mt-1 text-xs text-gray-400">
            <span className="text-amber-400">{"★".repeat(Math.floor(provider.rating))}</span>
            <span>{provider.rating} ({provider.reviews} reviews)</span>
          </div>
        </div>
      </div>
      <div className="flex flex-wrap gap-2 mb-4">
        <span className="bg-blue-50 text-blue-700 text-xs font-semibold px-2.5 py-1 rounded-full">📍 {provider.dist}</span>
        <span className="bg-amber-50 text-amber-700 text-xs font-semibold px-2.5 py-1 rounded-full">⏱️ ETA {provider.eta}</span>
        <span className="bg-gray-100 text-gray-600 text-xs font-semibold px-2.5 py-1 rounded-full">🛠️ {provider.jobs.toLocaleString()} jobs</span>
        {provider.verified && <span className="bg-green-100 text-green-700 text-xs font-semibold px-2.5 py-1 rounded-full">✓ Verified</span>}
      </div>
      <div className="flex flex-wrap gap-1 mb-4">
        {provider.services.map(sid=>{ const s=SERVICES.find(x=>x.id===sid); return s?<span key={sid} className="text-xs px-2 py-1 rounded-full font-medium" style={{ background:s.bg, color:s.color }}>{s.icon} {s.label}</span>:null; })}
      </div>
      <button onClick={onBook} disabled={!provider.active}
        className={`w-full py-2.5 rounded-xl font-bold text-xs text-white transition-all ${provider.active?"bg-red-600 hover:bg-red-700 hover:-translate-y-0.5 shadow-md":"bg-gray-200 text-gray-400 cursor-not-allowed"}`}>
        {provider.active?"📞 Request This Provider":"Currently Unavailable"}
      </button>
    </div>
  );
}

// SERVICES PAGE
function ServicesPage({ onRequest }) {
  const [sel, setSel] = useState(null);
  return (
    <div className="min-h-screen bg-gray-50 pt-20 sm:pt-24 pb-20">
      <div className="max-w-6xl mx-auto px-4 sm:px-6">
        <div className="py-8 sm:py-10">
          <span className="text-red-600 font-bold text-xs uppercase tracking-widest">All Services</span>
          <h1 className="font-display text-3xl sm:text-4xl font-black mt-1 mb-2">Emergency Services</h1>
          <p className="text-gray-500 text-sm sm:text-base">Tap any service to see details and book instantly.</p>
        </div>
        <div className="space-y-4">
          {SERVICES.map(s=>(
            <div key={s.id} className={`bg-white rounded-3xl border-2 overflow-hidden transition-all duration-300 shadow-sm hover:shadow-md ${sel===s.id?"":"border-gray-100 hover:border-gray-200"}`}
              style={sel===s.id?{borderColor:s.color}:{}}>
              <button className="w-full px-5 sm:px-7 py-5 flex items-center gap-4 sm:gap-5 text-left" onClick={()=>setSel(sel===s.id?null:s.id)}>
                <div className="w-14 h-14 sm:w-16 sm:h-16 rounded-2xl flex items-center justify-center text-3xl sm:text-4xl flex-shrink-0" style={{ background:s.bg, border:`2px solid ${s.border}` }}>{s.icon}</div>
                <div className="flex-1 min-w-0">
                  <div className="flex flex-wrap items-center gap-2 mb-1">
                    <h3 className="font-display text-lg sm:text-xl font-bold" style={{ color:s.color }}>{s.label}</h3>
                    <span className="text-xs px-2.5 py-1 rounded-full font-bold" style={{ background:s.bg, color:s.color }}>⏱️ {s.eta}</span>
                  </div>
                  <p className="text-gray-500 text-xs sm:text-sm line-clamp-1">{s.desc}</p>
                </div>
                <div className="text-right flex-shrink-0 ml-2">
                  <div className="font-black text-lg sm:text-xl" style={{ color:s.color }}>₹{s.price}</div>
                  <div className="text-gray-400 text-sm mt-1 transition-transform duration-300" style={{ transform:sel===s.id?"rotate(180deg)":"rotate(0)" }}>⌄</div>
                </div>
              </button>
              {sel===s.id && (
                <div className="px-5 sm:px-7 pb-6 border-t" style={{ borderColor:s.border, background:s.bg }}>
                  <p className="text-gray-600 text-sm mt-4 mb-4 leading-relaxed">{s.desc}</p>
                  <div className="grid grid-cols-2 sm:grid-cols-4 gap-3 mb-5">
                    {s.features.map(f=>(
                      <div key={f} className="bg-white rounded-xl p-3 flex items-start gap-2">
                        <span style={{ color:s.color }} className="font-black text-sm flex-shrink-0">✓</span>
                        <span className="text-xs text-gray-600">{f}</span>
                      </div>
                    ))}
                  </div>
                  <button onClick={()=>onRequest(s)} className="px-8 py-3 rounded-xl font-black text-white transition-all hover:-translate-y-0.5 shadow-lg" style={{ background:s.color }}>
                    Request {s.label} →
                  </button>
                </div>
              )}
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}

// PROVIDERS PAGE
function ProvidersPage({ onBook }) {
  const [filter, setFilter] = useState({ service:"all", available:false, city:"all" });
  const cities = [...new Set(PROVIDERS.map(p=>p.city))];
  let filtered = PROVIDERS
    .filter(p=>!filter.available||p.active)
    .filter(p=>filter.city==="all"||p.city===filter.city)
    .filter(p=>filter.service==="all"||p.services.includes(filter.service));
  return (
    <div className="min-h-screen bg-gray-50 pt-20 sm:pt-24 pb-20">
      <div className="max-w-6xl mx-auto px-4 sm:px-6">
        <div className="py-8 sm:py-10">
          <span className="text-red-600 font-bold text-xs uppercase tracking-widest">Service Network</span>
          <h1 className="font-display text-3xl sm:text-4xl font-black mt-1 mb-2">Our Providers</h1>
          <p className="text-gray-500 text-sm">{filtered.length} verified providers found</p>
        </div>
        {/* Filters */}
        <div className="bg-white rounded-2xl sm:rounded-3xl p-4 sm:p-5 mb-6 shadow-sm border border-gray-100 flex flex-wrap gap-3">
          <select className="bg-gray-50 border border-gray-200 rounded-xl px-4 py-2.5 text-sm focus:border-red-400 outline-none"
            value={filter.service} onChange={e=>setFilter(f=>({...f,service:e.target.value}))}>
            <option value="all">All Services</option>
            {SERVICES.map(s=><option key={s.id} value={s.id}>{s.icon} {s.label}</option>)}
          </select>
          <select className="bg-gray-50 border border-gray-200 rounded-xl px-4 py-2.5 text-sm focus:border-red-400 outline-none"
            value={filter.city} onChange={e=>setFilter(f=>({...f,city:e.target.value}))}>
            <option value="all">All Cities</option>
            {cities.map(c=><option key={c}>{c}</option>)}
          </select>
          <label className="flex items-center gap-2 cursor-pointer bg-gray-50 border border-gray-200 rounded-xl px-4 py-2.5 text-sm select-none">
            <input type="checkbox" checked={filter.available} onChange={e=>setFilter(f=>({...f,available:e.target.checked}))} className="accent-red-600 w-4 h-4" />
            <span className="font-medium">Online Now</span>
          </label>
        </div>
        <div className="grid grid-cols-1 sm:grid-cols-2 gap-5">
          {filtered.map(p=><ProviderCard key={p.id} provider={p} onBook={()=>onBook(null)} />)}
        </div>
      </div>
    </div>
  );
}

// MY REQUESTS PAGE
function RequestsPage({ addToast }) {
  const [requests, setRequests] = useState(MY_REQUESTS);
  const [trackModal, setTrackModal] = useState(null);
  const [rateModal, setRateModal] = useState(null);
  const [myRating, setMyRating] = useState(0);
  const [myReview, setMyReview] = useState("");

  const handleRate = (id) => {
    setRequests(r=>r.map(x=>x.id===id?{...x,rating:myRating}:x));
    addToast("Thank you for your feedback! ⭐","success");
    setRateModal(null); setMyRating(0); setMyReview("");
  };

  const statusColor = { "Completed":"border-green-200","In Progress":"border-amber-200","Cancelled":"border-red-200","Pending":"border-blue-200" };

  return (
    <div className="min-h-screen bg-gray-50 pt-20 sm:pt-24 pb-20">
      <div className="max-w-3xl mx-auto px-4 sm:px-6">
        <div className="py-8">
          <span className="text-red-600 font-bold text-xs uppercase tracking-widest">History</span>
          <h1 className="font-display text-3xl sm:text-4xl font-black mt-1 mb-2">My Requests</h1>
          <p className="text-gray-500 text-sm">{requests.length} service requests</p>
        </div>

        {/* Active request tracker banner */}
        {requests.find(r=>r.status==="In Progress") && (
          <div className="bg-gradient-to-r from-red-600 to-orange-500 text-white rounded-3xl p-5 mb-6 shadow-xl">
            <div className="flex items-center justify-between mb-3">
              <div>
                <div className="text-red-100 text-xs font-bold uppercase tracking-wider mb-1">Active Request</div>
                <div className="font-black text-lg">{requests.find(r=>r.status==="In Progress").id}</div>
              </div>
              <div className="bg-white/20 rounded-2xl px-3 py-2 text-sm font-bold flex items-center gap-1">
                <span className="w-2 h-2 bg-white rounded-full animate-pulse" />In Progress
              </div>
            </div>
            <div className="text-red-100 text-sm mb-3">{requests.find(r=>r.status==="In Progress").service} · {requests.find(r=>r.status==="In Progress").provider}</div>
            <button onClick={()=>setTrackModal(requests.find(r=>r.status==="In Progress"))} className="bg-white text-red-600 px-5 py-2 rounded-xl font-black text-sm hover:bg-red-50 transition-colors">
              🗺️ Track Live
            </button>
          </div>
        )}

        <div className="space-y-4">
          {requests.map(req=>(
            <div key={req.id} className={`bg-white rounded-2xl sm:rounded-3xl border-2 ${statusColor[req.status]||"border-gray-100"} shadow-sm overflow-hidden`}>
              <div className="p-4 sm:p-5">
                <div className="flex items-start justify-between gap-3 mb-3">
                  <div>
                    <div className="font-black text-sm sm:text-base text-gray-900">{req.id}</div>
                    <div className="text-xs text-gray-400 mt-0.5">📅 {req.date} · 👷 {req.provider}</div>
                  </div>
                  <StatusBadge status={req.status} />
                </div>
                <div className="flex items-center gap-3 mb-3">
                  <span className="text-xl">{SERVICES.find(s=>s.label===req.service)?.icon||"🔧"}</span>
                  <span className="font-bold text-sm text-gray-800">{req.service}</span>
                </div>
                <div className="text-xs text-gray-400 mb-3 flex items-center gap-1">📍 {req.location}</div>
                <div className="flex items-center justify-between">
                  <div>
                    {req.status==="Completed" && <div className="font-black text-base text-gray-900">₹{req.amount}</div>}
                    {req.rating && <StarRating rating={req.rating} />}
                  </div>
                  <div className="flex gap-2">
                    {req.status==="In Progress" && (
                      <button onClick={()=>setTrackModal(req)} className="text-xs font-bold bg-red-600 text-white px-3 py-2 rounded-xl hover:bg-red-700 transition-colors">
                        🗺️ Track
                      </button>
                    )}
                    {req.status==="Completed" && !req.rating && (
                      <button onClick={()=>setRateModal(req)} className="text-xs font-bold bg-amber-500 text-white px-3 py-2 rounded-xl hover:bg-amber-600 transition-colors">
                        ⭐ Rate
                      </button>
                    )}
                    {req.status==="Completed" && (
                      <button className="text-xs font-bold bg-blue-50 text-blue-600 px-3 py-2 rounded-xl hover:bg-blue-100 transition-colors">
                        🔁 Rebook
                      </button>
                    )}
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Track Modal */}
      <Modal open={!!trackModal} onClose={()=>setTrackModal(null)} title="Live Tracking" size="md">
        {trackModal && <LiveTracker request={trackModal} onClose={()=>setTrackModal(null)} />}
      </Modal>

      {/* Rate Modal */}
      <Modal open={!!rateModal} onClose={()=>setRateModal(null)} title="Rate Your Experience" size="sm">
        {rateModal && (
          <div className="space-y-4">
            <div className="text-center py-2">
              <div className="text-4xl mb-3">{SERVICES.find(s=>s.label===rateModal.service)?.icon||"🔧"}</div>
              <div className="font-bold text-gray-800">{rateModal.service}</div>
              <div className="text-sm text-gray-400">{rateModal.provider}</div>
              <div className="flex justify-center mt-4">
                <StarRating rating={myRating} interactive onRate={setMyRating} />
              </div>
              <div className="text-sm text-gray-400 mt-2">{myRating>0?["","Poor","Fair","Good","Very Good","Excellent!"][myRating]:"Tap to rate"}</div>
            </div>
            <textarea className="w-full bg-gray-50 border-2 border-gray-200 rounded-xl px-4 py-3 text-sm focus:border-red-400 outline-none resize-none" rows={3}
              placeholder="Write a review (optional)..." value={myReview} onChange={e=>setMyReview(e.target.value)} />
            <button onClick={()=>myRating>0&&handleRate(rateModal.id)} disabled={myRating===0}
              className={`w-full py-3 rounded-2xl font-black text-white transition-all ${myRating>0?"bg-red-600 hover:bg-red-700 shadow-lg":"bg-gray-200 text-gray-400 cursor-not-allowed"}`}>
              Submit Rating
            </button>
          </div>
        )}
      </Modal>
    </div>
  );
}

// ABOUT PAGE
function AboutPage({ onNavigate }) {
  const [openFaq, setOpenFaq] = useState(null);
  return (
    <div className="min-h-screen bg-gray-50 pt-20 sm:pt-24 pb-20">
      <div className="max-w-5xl mx-auto px-4 sm:px-6">
        {/* Hero */}
        <div className="py-10 text-center">
          <span className="text-red-600 font-bold text-xs uppercase tracking-widest">Our Mission</span>
          <h1 className="font-display text-3xl sm:text-4xl md:text-5xl font-black mt-2 mb-4">About RoadRescue</h1>
          <p className="text-gray-500 max-w-2xl mx-auto text-sm sm:text-base leading-relaxed">
            We believe no driver should ever feel helpless on the road. RoadRescue connects vehicle owners with trained assistance providers to deliver fast, reliable emergency support anytime, anywhere.
          </p>
        </div>

        {/* Mission cards */}
        <div className="grid md:grid-cols-2 gap-6 sm:gap-8 mb-14">
          <div className="bg-white rounded-3xl p-6 sm:p-8 shadow-sm border border-gray-100">
            <div className="text-4xl mb-4">🎯</div>
            <h2 className="font-display text-2xl font-bold mb-3">Our Mission</h2>
            <p className="text-gray-500 text-sm leading-relaxed">To reduce helplessness and danger for drivers across India by connecting them with fast, reliable, and professional roadside assistance providers through smart technology.</p>
          </div>
          <div className="bg-red-600 rounded-3xl p-6 sm:p-8 text-white shadow-xl">
            <div className="text-4xl mb-4">🛡️</div>
            <h2 className="font-display text-2xl font-bold mb-3">Our Promise</h2>
            <p className="text-red-100 text-sm leading-relaxed">Every provider on our platform is background-verified, licensed, and insured. We guarantee service quality and cover any damage caused during assistance.</p>
          </div>
        </div>

        {/* Values */}
        <div className="mb-14">
          <h2 className="font-display text-2xl sm:text-3xl font-black text-center mb-8">Why RoadRescue?</h2>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            {[
              { ic:"⚡", t:"Fast Response", d:"Average 18 min nationwide" },
              { ic:"🔒", t:"Verified Providers", d:"Background checked & insured" },
              { ic:"💳", t:"Pay After Service", d:"No upfront payment needed" },
              { ic:"📱", t:"Live Tracking", d:"Real-time GPS tracking" },
            ].map(v=>(
              <div key={v.t} className="bg-white rounded-2xl p-4 sm:p-5 text-center border border-gray-100 shadow-sm hover:-translate-y-1 hover:shadow-md transition-all">
                <div className="text-3xl sm:text-4xl mb-3">{v.ic}</div>
                <h3 className="font-bold text-sm mb-2">{v.t}</h3>
                <p className="text-gray-400 text-xs leading-relaxed">{v.d}</p>
              </div>
            ))}
          </div>
        </div>

        {/* FAQs */}
        <div className="mb-14">
          <h2 className="font-display text-2xl sm:text-3xl font-black text-center mb-8">Frequently Asked Questions</h2>
          <div className="space-y-3 max-w-3xl mx-auto">
            {FAQS.map((f,i)=>(
              <div key={i} className="bg-white rounded-2xl border border-gray-100 overflow-hidden shadow-sm">
                <button className="w-full px-5 sm:px-6 py-4 flex items-center justify-between text-left gap-3" onClick={()=>setOpenFaq(openFaq===i?null:i)}>
                  <span className="font-bold text-sm sm:text-base text-gray-800">{f.q}</span>
                  <span className="text-red-500 text-xl transition-transform flex-shrink-0" style={{ transform:openFaq===i?"rotate(180deg)":"rotate(0)" }}>⌄</span>
                </button>
                {openFaq===i && <div className="px-5 sm:px-6 pb-4 text-gray-500 text-sm leading-relaxed border-t border-gray-50 pt-3">{f.a}</div>}
              </div>
            ))}
          </div>
        </div>

        {/* CTA */}
        <div className="text-center rounded-3xl p-8 sm:p-12" style={{ background:"linear-gradient(135deg,#0a0a0a,#1a0505)" }}>
          <div className="text-5xl mb-4">🚛</div>
          <h2 className="font-display text-2xl sm:text-3xl font-black text-white mb-3">Stranded? We've Got You.</h2>
          <p className="text-gray-400 mb-6 text-sm sm:text-base">24/7 emergency support across Gujarat. Help is one tap away.</p>
          <button onClick={()=>onNavigate("home")} className="bg-red-600 hover:bg-red-700 text-white px-10 py-4 rounded-2xl font-black shadow-2xl transition-all hover:-translate-y-1 text-sm sm:text-base">
            🚨 Request Help Now
          </button>
        </div>
      </div>
    </div>
  );
}

/* ═══════════════════════════════════════════
   APP SHELL
═══════════════════════════════════════════ */
export default function App() {
  const [page, setPage] = useState("home");
  const [menuOpen, setMenuOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);
  const [requestModal, setRequestModal] = useState(false);
  const [preService, setPreService] = useState(null);
  const [successModal, setSuccessModal] = useState(null);
  const { toasts, add: addToast, remove: removeToast } = useToast();

  useEffect(() => {
    const h = () => setScrolled(window.scrollY > 50);
    window.addEventListener("scroll", h);
    return () => window.removeEventListener("scroll", h);
  }, []);

  useEffect(() => { window.scrollTo({ top:0, behavior:"smooth" }); setMenuOpen(false); }, [page]);

  const openRequest = (service=null) => { setPreService(service); setRequestModal(true); };

  const handleRequestSubmit = (data) => {
    const requestId = "RR-" + Date.now().toString().slice(-6);
    setRequestModal(false);
    setSuccessModal({ ...data, id: requestId });
    addToast(`🚨 Request ${requestId} sent! Provider dispatched in ${data.provider?.eta}.`, "success");
  };

  const NAV = [
    { id:"home",      label:"Home",       icon:"🏠" },
    { id:"services",  label:"Services",   icon:"🔧" },
    { id:"providers", label:"Providers",  icon:"🚛" },
    { id:"requests",  label:"My Requests",icon:"📋" },
    { id:"about",     label:"About",      icon:"ℹ️" },
  ];

  const darkNav = page==="home" && !scrolled;

  return (
    <div style={{ fontFamily:"'Barlow', sans-serif" }}>
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@700;900&family=Barlow:wght@400;500;600;700;800;900&display=swap');
        .font-display { font-family:'Playfair Display',Georgia,serif; }
        @keyframes floatY { 0%,100%{transform:translateY(0)} 50%{transform:translateY(-14px)} }
        @keyframes slideUp { from{transform:translateY(50px);opacity:0} to{transform:translateY(0);opacity:1} }
        @keyframes shrink { from{width:100%} to{width:0%} }
        @keyframes pulse-slow { 0%,100%{opacity:1} 50%{opacity:0.7} }
        .animate-pulse-slow { animation:pulse-slow 2s ease-in-out infinite; }
        .line-clamp-1{overflow:hidden;display:-webkit-box;-webkit-line-clamp:1;-webkit-box-orient:vertical;}
        .line-clamp-2{overflow:hidden;display:-webkit-box;-webkit-line-clamp:2;-webkit-box-orient:vertical;}
        *{box-sizing:border-box;}
        ::-webkit-scrollbar{width:6px} ::-webkit-scrollbar-track{background:#f1f1f1} ::-webkit-scrollbar-thumb{background:#dc2626;border-radius:3px}
      `}</style>

      <ToastContainer toasts={toasts} removeToast={removeToast} />

      {/* NAVBAR */}
      <nav className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${darkNav?"bg-transparent py-4":"bg-white/95 backdrop-blur-md shadow-md py-3"}`}>
        <div className="max-w-6xl mx-auto px-4 sm:px-6 flex items-center justify-between">
          <button onClick={()=>setPage("home")} className="flex items-center gap-2 group">
            <div className="w-9 h-9 bg-red-600 rounded-xl flex items-center justify-center text-lg font-black text-white shadow-lg group-hover:bg-red-700 transition-colors">🚛</div>
            <div>
              <span className={`font-display text-lg sm:text-xl font-black leading-none ${darkNav?"text-white":"text-gray-900"}`}>Road<span className="text-red-600">Rescue</span></span>
            </div>
          </button>

          {/* Desktop nav */}
          <div className="hidden md:flex items-center gap-1">
            {NAV.map(n=>(
              <button key={n.id} onClick={()=>setPage(n.id)}
                className={`px-3 lg:px-4 py-2 rounded-full text-sm font-semibold transition-all ${page===n.id?"bg-red-50 text-red-600":darkNav?"text-white/80 hover:text-white hover:bg-white/10":"text-gray-600 hover:text-red-600 hover:bg-red-50"}`}>
                {n.label}
              </button>
            ))}
          </div>

          <div className="flex items-center gap-2 sm:gap-3">
            <button onClick={()=>openRequest()} className="flex items-center gap-1.5 bg-red-600 hover:bg-red-700 text-white px-3 sm:px-5 py-2 sm:py-2.5 rounded-full font-bold text-xs sm:text-sm shadow-lg hover:shadow-red-200 transition-all animate-pulse-slow">
              🚨 <span className="hidden sm:inline">SOS Help</span><span className="sm:hidden">SOS</span>
            </button>
            <button onClick={()=>setMenuOpen(!menuOpen)} className={`md:hidden w-9 h-9 rounded-full flex items-center justify-center text-base transition-colors ${darkNav?"bg-white/20 text-white":"bg-gray-100 text-gray-700"}`}>
              {menuOpen?"✕":"☰"}
            </button>
          </div>
        </div>

        {/* Mobile menu */}
        {menuOpen && (
          <div className="md:hidden bg-white border-t border-gray-100 shadow-xl px-4 py-4" style={{ animation:"slideUp 0.25s ease" }}>
            <div className="grid grid-cols-2 gap-2 mb-3">
              {NAV.map(n=>(
                <button key={n.id} onClick={()=>setPage(n.id)}
                  className={`flex items-center gap-2 p-3 rounded-2xl text-sm font-bold transition-all text-left ${page===n.id?"bg-red-50 text-red-600":"bg-gray-50 text-gray-700 hover:bg-red-50 hover:text-red-600"}`}>
                  <span>{n.icon}</span><span>{n.label}</span>
                </button>
              ))}
            </div>
            <button onClick={()=>{ openRequest(); setMenuOpen(false); }} className="w-full bg-red-600 text-white py-3.5 rounded-2xl font-black text-sm flex items-center justify-center gap-2 animate-pulse-slow shadow-lg">
              🚨 Request Emergency Help
            </button>
            <a href="tel:+918000123456" className="w-full mt-2 bg-gray-900 text-white py-3 rounded-2xl font-bold text-sm flex items-center justify-center gap-2">
              📞 Call Helpline: 1800-123-456
            </a>
          </div>
        )}
      </nav>

      {/* PAGES */}
      <main>
        {page==="home"      && <HomePage       onNavigate={setPage} onEmergency={openRequest} />}
        {page==="services"  && <ServicesPage   onRequest={openRequest} />}
        {page==="providers" && <ProvidersPage  onBook={openRequest} />}
        {page==="requests"  && <RequestsPage   addToast={addToast} />}
        {page==="about"     && <AboutPage      onNavigate={setPage} />}
      </main>

      {/* REQUEST WIZARD MODAL */}
      <Modal open={requestModal} onClose={()=>setRequestModal(false)} title="🚨 Request Emergency Help" size="lg">
        <RequestWizard preService={preService} onClose={()=>setRequestModal(false)} onSubmit={handleRequestSubmit} />
      </Modal>

      {/* SUCCESS MODAL */}
      <Modal open={!!successModal} onClose={()=>setSuccessModal(null)} title="Request Confirmed!" size="md">
        {successModal && (
          <div className="text-center space-y-4">
            <div className="text-6xl sm:text-7xl mb-2" style={{ animation:"floatY 2s ease-in-out infinite" }}>🚛</div>
            <h3 className="font-display text-2xl font-black text-gray-900">Help is On the Way!</h3>
            <p className="text-gray-500 text-sm">Your emergency request has been confirmed and a provider has been dispatched.</p>
            <div className="bg-red-50 border border-red-200 rounded-2xl p-4 text-left space-y-2">
              <div className="flex justify-between text-sm"><span className="text-gray-400">Request ID</span><span className="font-black text-red-600">{successModal.id}</span></div>
              <div className="flex justify-between text-sm"><span className="text-gray-400">Service</span><span className="font-bold">{successModal.service?.label}</span></div>
              <div className="flex justify-between text-sm"><span className="text-gray-400">Provider</span><span className="font-bold">{successModal.provider?.name}</span></div>
              <div className="flex justify-between text-sm"><span className="text-gray-400">ETA</span><span className="font-bold text-green-600">🚀 {successModal.provider?.eta}</span></div>
              <div className="flex justify-between text-sm"><span className="text-gray-400">Amount</span><span className="font-black text-gray-900">₹{successModal.service?.price} (pay after)</span></div>
            </div>
            <div className="bg-blue-50 border border-blue-200 rounded-2xl p-3 text-left">
              <div className="text-blue-700 font-bold text-xs mb-1">📱 Tracking Link Sent</div>
              <div className="text-blue-600 text-xs">Real-time tracking SMS sent to {successModal.form?.phone}. Watch your provider arrive on the map!</div>
            </div>
            <div className="flex gap-3">
              <button onClick={()=>{ setSuccessModal(null); setPage("requests"); }} className="flex-1 py-3 rounded-2xl border-2 border-gray-200 font-bold text-gray-700 hover:bg-gray-50 text-sm">
                View in My Requests
              </button>
              <button onClick={()=>setSuccessModal(null)} className="flex-1 py-3 rounded-2xl bg-red-600 hover:bg-red-700 text-white font-black text-sm shadow-lg">
                Close
              </button>
            </div>
          </div>
        )}
      </Modal>

      {/* FOOTER */}
      <footer style={{ background:"#0a0a0a" }} className="text-white pt-12 sm:pt-16 pb-6">
        <div className="max-w-6xl mx-auto px-4 sm:px-6">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8 mb-10">
            <div className="col-span-2 md:col-span-1">
              <div className="flex items-center gap-2 mb-4">
                <div className="w-9 h-9 bg-red-600 rounded-xl flex items-center justify-center text-lg">🚛</div>
                <span className="font-display text-xl font-black">Road<span className="text-red-500">Rescue</span></span>
              </div>
              <p className="text-gray-400 text-xs sm:text-sm leading-relaxed mb-4">24/7 emergency vehicle support. Because no driver should ever feel stranded and helpless.</p>
              <a href="tel:18001234567" className="inline-flex items-center gap-2 bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-xl font-bold text-sm transition-colors">
                📞 1800-123-456
              </a>
            </div>
            {[
              { t:"Services",  links:SERVICES.map(s=>`${s.icon} ${s.label}`) },
              { t:"Company",   links:["About Us","Our Providers","Careers","Press","Blog"] },
              { t:"Support",   links:["Help Center","Track Request","Refund Policy","Privacy Policy","Terms of Use"] },
            ].map(col=>(
              <div key={col.t}>
                <h4 className="font-bold text-xs uppercase tracking-widest text-gray-400 mb-3">{col.t}</h4>
                <ul className="space-y-2">
                  {col.links.map(l=><li key={l}><button className="text-gray-500 hover:text-red-400 text-xs sm:text-sm transition-colors text-left">{l}</button></li>)}
                </ul>
              </div>
            ))}
          </div>
          <div className="border-t border-gray-800 pt-5 flex flex-col sm:flex-row items-center justify-between gap-3">
            <p className="text-gray-600 text-xs">© 2024 RoadRescue. All rights reserved. Registered & insured platform.</p>
            <div className="flex items-center gap-2">
              <div className="w-2 h-2 rounded-full bg-green-400 animate-pulse" />
              <span className="text-gray-500 text-xs">{PROVIDERS.filter(p=>p.active).length} providers online now</span>
            </div>
          </div>
        </div>
      </footer>
    </div>
  );
}