import { useState, useEffect, useRef } from "react";

// ═══════════════════════════════════════════════════════════════
// WANDERLUST — Complete Travel Agency Website
// Pages: Home, Destinations, Tours, About, Contact
// Style: Warm & Earthy — cream, terracotta, forest green, sand
// ═══════════════════════════════════════════════════════════════

const STYLES = `
  @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:ital,wght@0,400;0,600;0,700;0,900;1,400;1,600&family=Jost:wght@300;400;500;600&display=swap');

  *, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }

  :root {
    --cream:     #F5F0E8;
    --cream-dark:#EDE6D6;
    --sand:      #D4C5A9;
    --terra:     #C1754A;
    --terra-dark:#A85F35;
    --brown:     #6B4226;
    --green:     #3D6B4F;
    --green-light:#5A8A6A;
    --green-pale: #EAF2EC;
    --charcoal:  #2C2416;
    --warm-gray: #8B7D6B;
  }

  html { scroll-behavior: smooth; }

  body, #root {
    background: var(--cream);
    font-family: 'Jost', sans-serif;
    color: var(--charcoal);
    overflow-x: hidden;
  }

  .playfair { font-family: 'Playfair Display', serif; }
  .jost     { font-family: 'Jost', sans-serif; }

  /* Animations */
  @keyframes fadeUp   { from{opacity:0;transform:translateY(32px)} to{opacity:1;transform:translateY(0)} }
  @keyframes fadeIn   { from{opacity:0} to{opacity:1} }
  @keyframes slideRight { from{opacity:0;transform:translateX(-40px)} to{opacity:1;transform:translateX(0)} }
  @keyframes slideLeft  { from{opacity:0;transform:translateX(40px)}  to{opacity:1;transform:translateX(0)} }
  @keyframes scaleIn  { from{opacity:0;transform:scale(0.92)} to{opacity:1;transform:scale(1)} }
  @keyframes float    { 0%,100%{transform:translateY(0)} 50%{transform:translateY(-12px)} }
  @keyframes drift    { 0%,100%{transform:translateX(0) rotate(0deg)} 50%{transform:translateX(8px) rotate(2deg)} }
  @keyframes heroZoom { from{transform:scale(1.08)} to{transform:scale(1)} }
  @keyframes marquee  { from{transform:translateX(0)} to{transform:translateX(-50%)} }
  @keyframes shimmer  { 0%{background-position:-200% 0} 100%{background-position:200% 0} }
  @keyframes navSlide { from{opacity:0;transform:translateY(-20px)} to{opacity:1;transform:translateY(0)} }
  @keyframes pulse    { 0%,100%{opacity:1;transform:scale(1)} 50%{opacity:.7;transform:scale(.96)} }

  .fade-up     { animation: fadeUp .7s ease both; }
  .fade-in     { animation: fadeIn .5s ease both; }
  .slide-right { animation: slideRight .7s ease both; }
  .slide-left  { animation: slideLeft .7s ease both; }
  .scale-in    { animation: scaleIn .6s cubic-bezier(.34,1.2,.64,1) both; }
  .float-anim  { animation: float 5s ease-in-out infinite; }
  .drift-anim  { animation: drift 7s ease-in-out infinite; }
  .nav-slide   { animation: navSlide .4s ease both; }
  .hero-zoom   { animation: heroZoom 8s ease forwards; }
  .pulse-anim  { animation: pulse 2s ease-in-out infinite; }

  /* Stagger delays */
  .d1{animation-delay:.1s} .d2{animation-delay:.2s} .d3{animation-delay:.3s}
  .d4{animation-delay:.4s} .d5{animation-delay:.5s} .d6{animation-delay:.6s}

  /* Navbar */
  .navbar {
    position: fixed; top: 0; left: 0; right: 0; z-index: 100;
    transition: all .4s ease;
    padding: 0 clamp(16px,5vw,64px);
  }
  .navbar.scrolled {
    background: rgba(245,240,232,.96);
    backdrop-filter: blur(16px);
    border-bottom: 1px solid var(--sand);
    box-shadow: 0 4px 24px rgba(107,66,38,.08);
  }
  .navbar.top { background: transparent; }

  .nav-link {
    font-family: 'Jost', sans-serif;
    font-weight: 500;
    font-size: 14px;
    letter-spacing: .08em;
    text-transform: uppercase;
    color: inherit;
    text-decoration: none;
    position: relative;
    cursor: pointer;
    transition: color .2s;
  }
  .nav-link::after {
    content: '';
    position: absolute;
    bottom: -3px; left: 0; right: 100%;
    height: 1.5px;
    background: var(--terra);
    transition: right .3s ease;
  }
  .nav-link:hover::after, .nav-link.active::after { right: 0; }
  .nav-link.active { color: var(--terra); }

  /* Buttons */
  .btn-primary {
    display: inline-flex; align-items: center; gap: 8px;
    background: var(--terra);
    color: #fff;
    font-family: 'Jost', sans-serif;
    font-weight: 600;
    font-size: 13px;
    letter-spacing: .12em;
    text-transform: uppercase;
    padding: 14px 32px;
    border: none;
    cursor: pointer;
    transition: all .25s;
    text-decoration: none;
  }
  .btn-primary:hover { background: var(--terra-dark); transform: translateY(-2px); box-shadow: 0 8px 24px rgba(193,117,74,.35); }

  .btn-outline {
    display: inline-flex; align-items: center; gap: 8px;
    background: transparent;
    color: var(--terra);
    font-family: 'Jost', sans-serif;
    font-weight: 600;
    font-size: 13px;
    letter-spacing: .12em;
    text-transform: uppercase;
    padding: 13px 32px;
    border: 1.5px solid var(--terra);
    cursor: pointer;
    transition: all .25s;
  }
  .btn-outline:hover { background: var(--terra); color: #fff; transform: translateY(-2px); }

  .btn-white {
    display: inline-flex; align-items: center; gap: 8px;
    background: #fff;
    color: var(--terra);
    font-family: 'Jost', sans-serif;
    font-weight: 600;
    font-size: 13px;
    letter-spacing: .12em;
    text-transform: uppercase;
    padding: 14px 32px;
    border: none;
    cursor: pointer;
    transition: all .25s;
  }
  .btn-white:hover { background: var(--cream); transform: translateY(-2px); box-shadow: 0 8px 24px rgba(0,0,0,.12); }

  /* Cards */
  .dest-card {
    position: relative; overflow: hidden;
    border-radius: 0;
    cursor: pointer;
    transition: transform .4s ease;
  }
  .dest-card:hover { transform: translateY(-6px); }
  .dest-card:hover .card-img { transform: scale(1.06); }
  .dest-card:hover .card-overlay { opacity: 1; }

  .card-img {
    width: 100%; height: 100%;
    object-fit: cover;
    transition: transform .6s ease;
    display: block;
  }

  .card-overlay {
    position: absolute; inset: 0;
    background: linear-gradient(to top, rgba(44,36,22,.85) 0%, rgba(44,36,22,.2) 60%, transparent 100%);
    opacity: .7;
    transition: opacity .4s;
  }

  /* Tour card */
  .tour-card {
    background: #fff;
    border: 1px solid var(--cream-dark);
    transition: all .3s ease;
    overflow: hidden;
  }
  .tour-card:hover { box-shadow: 0 16px 48px rgba(107,66,38,.12); transform: translateY(-4px); }
  .tour-card:hover .tour-img { transform: scale(1.05); }
  .tour-img { transition: transform .5s ease; }

  /* Testimonial */
  .testimonial-card {
    background: #fff;
    border-left: 3px solid var(--terra);
    padding: 28px;
    transition: all .3s;
  }
  .testimonial-card:hover { box-shadow: 0 8px 32px rgba(107,66,38,.1); }

  /* Form */
  .form-input {
    width: 100%;
    background: #fff;
    border: 1.5px solid var(--cream-dark);
    padding: 14px 18px;
    font-family: 'Jost', sans-serif;
    font-size: 15px;
    color: var(--charcoal);
    outline: none;
    transition: border-color .2s;
  }
  .form-input:focus { border-color: var(--terra); }
  .form-input::placeholder { color: var(--warm-gray); }

  textarea.form-input { resize: vertical; min-height: 130px; }

  /* Section label */
  .section-label {
    font-family: 'Jost', sans-serif;
    font-weight: 600;
    font-size: 11px;
    letter-spacing: .3em;
    text-transform: uppercase;
    color: var(--terra);
    display: flex;
    align-items: center;
    gap: 10px;
  }
  .section-label::before {
    content: '';
    display: block;
    width: 28px; height: 1.5px;
    background: var(--terra);
  }

  /* Stats */
  .stat-item { text-align: center; }
  .stat-num {
    font-family: 'Playfair Display', serif;
    font-size: clamp(36px,6vw,60px);
    font-weight: 700;
    color: var(--terra);
    line-height: 1;
  }

  /* Marquee */
  .marquee-track { display: flex; animation: marquee 22s linear infinite; width: max-content; }
  .marquee-track:hover { animation-play-state: paused; }

  /* Mobile nav */
  .mobile-menu {
    position: fixed; inset: 0; z-index: 200;
    background: var(--cream);
    display: flex; flex-direction: column;
    align-items: center; justify-content: center;
    gap: 36px;
    transform: translateX(100%);
    transition: transform .4s cubic-bezier(.77,0,.175,1);
  }
  .mobile-menu.open { transform: translateX(0); }
  .mobile-nav-link {
    font-family: 'Playfair Display', serif;
    font-size: clamp(28px,8vw,48px);
    color: var(--charcoal);
    cursor: pointer;
    transition: color .2s;
  }
  .mobile-nav-link:hover { color: var(--terra); }
  .mobile-nav-link.active { color: var(--terra); }

  /* Divider */
  .divider { width: 60px; height: 2px; background: var(--terra); margin: 16px 0; }
  .divider.center { margin: 16px auto; }

  /* Badge */
  .badge {
    display: inline-block;
    background: var(--green-pale);
    color: var(--green);
    font-size: 11px;
    font-weight: 600;
    letter-spacing: .1em;
    text-transform: uppercase;
    padding: 4px 12px;
  }
  .badge.terra { background: rgba(193,117,74,.12); color: var(--terra); }

  /* Price tag */
  .price-tag {
    background: var(--terra);
    color: #fff;
    font-family: 'Playfair Display', serif;
    font-size: 22px;
    font-weight: 700;
    padding: 8px 20px;
    display: inline-block;
  }

  /* Scroll reveal (simple) */
  .reveal { opacity: 0; transform: translateY(28px); transition: opacity .7s ease, transform .7s ease; }
  .reveal.visible { opacity: 1; transform: translateY(0); }

  /* Rating stars */
  .stars { color: #D4A843; letter-spacing: 2px; }

  /* Footer */
  .footer-link {
    color: rgba(245,240,232,.5);
    text-decoration: none;
    font-size: 14px;
    transition: color .2s;
    cursor: pointer;
  }
  .footer-link:hover { color: var(--terra); }

  /* Responsive helpers */
  @media (max-width: 768px) {
    .hide-mobile { display: none !important; }
  }
  @media (min-width: 769px) {
    .hide-desktop { display: none !important; }
  }

  /* Custom scrollbar */
  ::-webkit-scrollbar { width: 6px; }
  ::-webkit-scrollbar-track { background: var(--cream); }
  ::-webkit-scrollbar-thumb { background: var(--sand); border-radius: 3px; }
  ::-webkit-scrollbar-thumb:hover { background: var(--terra); }

  /* Image placeholders using CSS gradients */
  .img-hero {
    background: linear-gradient(135deg,
      #8B6F47 0%, #C1754A 25%, #3D6B4F 50%, #D4C5A9 75%, #8B6F47 100%);
    background-size: 400% 400%;
    animation: shimmer 8s ease infinite;
  }
  .img-dest1 { background: linear-gradient(160deg,#2C5F2E,#6B8F71,#A8C5A0); }
  .img-dest2 { background: linear-gradient(160deg,#C19A6B,#8B6914,#D4A843); }
  .img-dest3 { background: linear-gradient(160deg,#1B4F72,#2E86AB,#A8D8EA); }
  .img-dest4 { background: linear-gradient(160deg,#922B21,#C0392B,#E8A0A0); }
  .img-dest5 { background: linear-gradient(160deg,#6B4226,#A0522D,#DEB887); }
  .img-dest6 { background: linear-gradient(160deg,#145A32,#27AE60,#A9DFBF); }
  .img-tour1 { background: linear-gradient(135deg,#2C5F2E 0%,#8FBC8F 100%); }
  .img-tour2 { background: linear-gradient(135deg,#8B4513 0%,#D2691E 100%); }
  .img-tour3 { background: linear-gradient(135deg,#191970 0%,#4169E1 50%,#87CEEB 100%); }
  .img-tour4 { background: linear-gradient(135deg,#556B2F 0%,#8FBC8F 100%); }
  .img-about { background: linear-gradient(135deg,#C1754A,#8B4513,#3D6B4F); }
  .img-team1 { background: linear-gradient(135deg,#D4A843,#8B6914); }
  .img-team2 { background: linear-gradient(135deg,#3D6B4F,#145A32); }
  .img-team3 { background: linear-gradient(135deg,#C1754A,#922B21); }
`;

// ═══════════════════════════════════════════════════════════════
// DATA
// ═══════════════════════════════════════════════════════════════

const DESTINATIONS = [
  {
    id: 1,
    name: "Santorini",
    country: "Greece",
    imgClass: "img-dest1",
    tag: "Island",
    price: "$1,299",
    rating: 4.9,
    reviews: 248,
  },
  {
    id: 2,
    name: "Marrakech",
    country: "Morocco",
    imgClass: "img-dest2",
    tag: "Desert",
    price: "$899",
    rating: 4.8,
    reviews: 187,
  },
  {
    id: 3,
    name: "Maldives",
    country: "S. Asia",
    imgClass: "img-dest3",
    tag: "Beach",
    price: "$2,199",
    rating: 5.0,
    reviews: 312,
  },
  {
    id: 4,
    name: "Kyoto",
    country: "Japan",
    imgClass: "img-dest4",
    tag: "Culture",
    price: "$1,499",
    rating: 4.9,
    reviews: 276,
  },
  {
    id: 5,
    name: "Patagonia",
    country: "Argentina",
    imgClass: "img-dest5",
    tag: "Adventure",
    price: "$1,799",
    rating: 4.7,
    reviews: 143,
  },
  {
    id: 6,
    name: "Bali",
    country: "Indonesia",
    imgClass: "img-dest6",
    tag: "Tropical",
    price: "$799",
    rating: 4.8,
    reviews: 398,
  },
];

const TOURS = [
  {
    id: 1,
    name: "Sacred Valley Trek",
    region: "Peru",
    days: 8,
    group: 12,
    price: 1299,
    imgClass: "img-tour1",
    tag: "Adventure",
    desc: "Hike ancient Inca trails through cloud forests and mountain passes to Machu Picchu.",
  },
  {
    id: 2,
    name: "Desert Caravan",
    region: "Morocco",
    days: 6,
    group: 8,
    price: 899,
    imgClass: "img-tour2",
    tag: "Cultural",
    desc: "Ride camels through the Sahara, sleep under stars, explore medinas and souks.",
  },
  {
    id: 3,
    name: "Greek Island Hopping",
    region: "Greece",
    days: 10,
    group: 16,
    price: 1899,
    imgClass: "img-tour3",
    tag: "Leisure",
    desc: "Sail the Aegean, visit Mykonos, Santorini & Rhodes with expert local guides.",
  },
  {
    id: 4,
    name: "Amazon Rainforest",
    region: "Brazil",
    days: 7,
    group: 10,
    price: 1599,
    imgClass: "img-tour4",
    tag: "Wildlife",
    desc: "Navigate the Amazon river, spot exotic wildlife and experience indigenous cultures.",
  },
];

const TESTIMONIALS = [
  {
    name: "Sarah Mitchell",
    location: "New York, USA",
    rating: 5,
    text: "Wanderlust planned every detail perfectly. Our Santorini trip was a dream — nothing was left to chance. Already planning our next adventure with them!",
    avatar: "SM",
  },
  {
    name: "James & Priya",
    location: "London, UK",
    rating: 5,
    text: "The Desert Caravan tour exceeded every expectation. Sleeping under Saharan stars was life-changing. The team's passion for travel shows in everything.",
    avatar: "JP",
  },
  {
    name: "Chen Wei",
    location: "Singapore",
    rating: 5,
    text: "Professional, knowledgeable, and genuinely caring. They found us hidden gems in Kyoto we never would have discovered on our own. Truly exceptional service.",
    avatar: "CW",
  },
];

const TEAM = [
  {
    name: "Elena Vasquez",
    role: "Founder & CEO",
    imgClass: "img-team1",
    bio: "20 years crafting transformative journeys across 80+ countries.",
  },
  {
    name: "Marcus Okafor",
    role: "Head of Experiences",
    imgClass: "img-team2",
    bio: "Former anthropologist turned travel curator with a passion for culture.",
  },
  {
    name: "Aiko Tanaka",
    role: "Destination Specialist",
    imgClass: "img-team3",
    bio: "Expert in Asia-Pacific destinations and immersive cultural programs.",
  },
];

const NAV_ITEMS = ["Home", "Destinations", "Tours", "About", "Contact"];

// ═══════════════════════════════════════════════════════════════
// HOOKS
// ═══════════════════════════════════════════════════════════════

function useScrolled(threshold = 60) {
  const [scrolled, setScrolled] = useState(false);
  useEffect(() => {
    const fn = () => setScrolled(window.scrollY > threshold);
    window.addEventListener("scroll", fn);
    return () => window.removeEventListener("scroll", fn);
  }, [threshold]);
  return scrolled;
}

function useReveal() {
  useEffect(() => {
    const els = document.querySelectorAll(".reveal");
    const obs = new IntersectionObserver(
      (entries) => {
        entries.forEach((e) => {
          if (e.isIntersecting) {
            e.target.classList.add("visible");
          }
        });
      },
      { threshold: 0.12 },
    );
    els.forEach((el) => obs.observe(el));
    return () => obs.disconnect();
  });
}

// ═══════════════════════════════════════════════════════════════
// NAVBAR
// ═══════════════════════════════════════════════════════════════

function Navbar({ page, setPage }) {
  const scrolled = useScrolled();
  const [menuOpen, setMenuOpen] = useState(false);
  const isHome = page === "Home";

  return (
    <>
      <nav
        className={`navbar ${scrolled ? "scrolled" : "top"}`}
        style={{
          height: 72,
          display: "flex",
          alignItems: "center",
          justifyContent: "space-between",
        }}
      >
        {/* Logo */}
        <div
          onClick={() => setPage("Home")}
          style={{
            cursor: "pointer",
            display: "flex",
            alignItems: "center",
            gap: 10,
          }}
        >
          <div
            style={{
              width: 36,
              height: 36,
              background: "var(--terra)",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
            }}
          >
            <span
              style={{
                color: "#fff",
                fontSize: 16,
                fontFamily: "'Playfair Display',serif",
                fontWeight: 700,
              }}
            >
              W
            </span>
          </div>
          <div>
            <div
              className="playfair"
              style={{
                fontSize: 18,
                fontWeight: 700,
                color: scrolled || !isHome ? "var(--charcoal)" : "#fff",
                lineHeight: 1,
              }}
            >
              Wanderlust
            </div>
            <div
              style={{
                fontSize: 9,
                letterSpacing: ".25em",
                textTransform: "uppercase",
                color:
                  scrolled || !isHome
                    ? "var(--warm-gray)"
                    : "rgba(255,255,255,.7)",
                fontFamily: "'Jost',sans-serif",
              }}
            >
              Travel & Journeys
            </div>
          </div>
        </div>

        {/* Desktop nav */}
        <div
          className="hide-mobile"
          style={{ display: "flex", alignItems: "center", gap: 36 }}
        >
          {NAV_ITEMS.map((item) => (
            <span
              key={item}
              onClick={() => setPage(item)}
              className={`nav-link ${page === item ? "active" : ""}`}
              style={{
                color:
                  scrolled || !isHome
                    ? "var(--charcoal)"
                    : "rgba(255,255,255,.9)",
              }}
            >
              {item}
            </span>
          ))}
          <button
            className="btn-primary"
            onClick={() => setPage("Contact")}
            style={{ padding: "10px 24px", fontSize: 12 }}
          >
            Book Now
          </button>
        </div>

        {/* Mobile hamburger */}
        <button
          className="hide-desktop"
          onClick={() => setMenuOpen(true)}
          style={{
            background: "none",
            border: "none",
            cursor: "pointer",
            padding: 8,
          }}
        >
          <div style={{ display: "flex", flexDirection: "column", gap: 5 }}>
            {[0, 1, 2].map((i) => (
              <div
                key={i}
                style={{
                  width: 26,
                  height: 2,
                  background: scrolled || !isHome ? "var(--charcoal)" : "#fff",
                }}
              />
            ))}
          </div>
        </button>
      </nav>

      {/* Mobile menu */}
      <div className={`mobile-menu ${menuOpen ? "open" : ""}`}>
        <button
          onClick={() => setMenuOpen(false)}
          style={{
            position: "absolute",
            top: 24,
            right: 24,
            background: "none",
            border: "none",
            cursor: "pointer",
            fontSize: 28,
            color: "var(--charcoal)",
          }}
        >
          ×
        </button>
        <div
          className="playfair"
          style={{
            position: "absolute",
            top: 24,
            left: 24,
            fontSize: 20,
            fontWeight: 700,
            color: "var(--terra)",
          }}
        >
          Wanderlust
        </div>
        {NAV_ITEMS.map((item, i) => (
          <span
            key={item}
            onClick={() => {
              setPage(item);
              setMenuOpen(false);
            }}
            className={`mobile-nav-link ${page === item ? "active" : ""}`}
            style={{ animationDelay: `${i * 0.07}s` }}
          >
            {item}
          </span>
        ))}
        <button
          className="btn-primary"
          onClick={() => {
            setPage("Contact");
            setMenuOpen(false);
          }}
        >
          Book Your Journey
        </button>
      </div>
    </>
  );
}

// ═══════════════════════════════════════════════════════════════
// FOOTER
// ═══════════════════════════════════════════════════════════════

function Footer({ setPage }) {
  return (
    <footer
      style={{
        background: "var(--charcoal)",
        color: "rgba(245,240,232,.85)",
        padding: "clamp(40px,8vw,80px) clamp(16px,5vw,64px) 32px",
      }}
    >
      <div style={{ maxWidth: 1200, margin: "0 auto" }}>
        <div
          style={{
            display: "grid",
            gridTemplateColumns: "repeat(auto-fit,minmax(200px,1fr))",
            gap: 40,
            marginBottom: 56,
          }}
        >
          {/* Brand */}
          <div>
            <div
              className="playfair"
              style={{
                fontSize: 26,
                fontWeight: 700,
                color: "#fff",
                marginBottom: 12,
              }}
            >
              Wanderlust
            </div>
            <p
              style={{
                fontSize: 14,
                lineHeight: 1.8,
                color: "rgba(245,240,232,.5)",
                maxWidth: 240,
              }}
            >
              Crafting extraordinary journeys for curious souls since 2008.
              Every trip is a story waiting to be told.
            </p>
            <div style={{ display: "flex", gap: 12, marginTop: 20 }}>
              {["f", "in", "tw", "yt"].map((s) => (
                <div
                  key={s}
                  style={{
                    width: 36,
                    height: 36,
                    border: "1px solid rgba(255,255,255,.15)",
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "center",
                    cursor: "pointer",
                    transition: "all .2s",
                    fontSize: 12,
                    color: "rgba(255,255,255,.6)",
                  }}
                  onMouseEnter={(e) => {
                    e.currentTarget.style.borderColor = "var(--terra)";
                    e.currentTarget.style.color = "var(--terra)";
                  }}
                  onMouseLeave={(e) => {
                    e.currentTarget.style.borderColor = "rgba(255,255,255,.15)";
                    e.currentTarget.style.color = "rgba(255,255,255,.6)";
                  }}
                >
                  {s}
                </div>
              ))}
            </div>
          </div>

          {/* Quick links */}
          <div>
            <div
              className="playfair"
              style={{
                fontSize: 16,
                fontWeight: 600,
                color: "#fff",
                marginBottom: 20,
              }}
            >
              Quick Links
            </div>
            <div style={{ display: "flex", flexDirection: "column", gap: 10 }}>
              {NAV_ITEMS.map((item) => (
                <span
                  key={item}
                  className="footer-link"
                  onClick={() => setPage(item)}
                >
                  {item}
                </span>
              ))}
            </div>
          </div>

          {/* Destinations */}
          <div>
            <div
              className="playfair"
              style={{
                fontSize: 16,
                fontWeight: 600,
                color: "#fff",
                marginBottom: 20,
              }}
            >
              Top Destinations
            </div>
            <div style={{ display: "flex", flexDirection: "column", gap: 10 }}>
              {[
                "Santorini, Greece",
                "Bali, Indonesia",
                "Marrakech, Morocco",
                "Kyoto, Japan",
                "Maldives",
              ].map((d) => (
                <span key={d} className="footer-link">
                  {d}
                </span>
              ))}
            </div>
          </div>

          {/* Contact */}
          <div>
            <div
              className="playfair"
              style={{
                fontSize: 16,
                fontWeight: 600,
                color: "#fff",
                marginBottom: 20,
              }}
            >
              Get In Touch
            </div>
            <div
              style={{
                display: "flex",
                flexDirection: "column",
                gap: 14,
                fontSize: 14,
                color: "rgba(245,240,232,.5)",
              }}
            >
              <div>📍 42 Explorer Lane, San Francisco, CA</div>
              <div>📞 +1 (800) 555-WANDER</div>
              <div>✉️ hello@wanderlust.travel</div>
              <div>🕐 Mon–Fri: 9am – 6pm PST</div>
            </div>
          </div>
        </div>

        <div
          style={{
            borderTop: "1px solid rgba(255,255,255,.08)",
            paddingTop: 24,
            display: "flex",
            flexWrap: "wrap",
            gap: 12,
            justifyContent: "space-between",
            alignItems: "center",
          }}
        >
          <p style={{ fontSize: 13, color: "rgba(245,240,232,.3)" }}>
            © 2024 Wanderlust Travel & Journeys. All rights reserved.
          </p>
          <div style={{ display: "flex", gap: 24 }}>
            {["Privacy Policy", "Terms of Service", "Cookie Policy"].map(
              (l) => (
                <span key={l} className="footer-link" style={{ fontSize: 12 }}>
                  {l}
                </span>
              ),
            )}
          </div>
        </div>
      </div>
    </footer>
  );
}

// ═══════════════════════════════════════════════════════════════
// PAGE: HOME
// ═══════════════════════════════════════════════════════════════

function HomePage({ setPage }) {
  useReveal();
  const [searchFocus, setSearchFocus] = useState(false);

  return (
    <div>
      {/* ── HERO ── */}
      <section
        style={{
          position: "relative",
          height: "100svh",
          minHeight: 600,
          overflow: "hidden",
          display: "flex",
          alignItems: "center",
        }}
      >
        {/* Background */}
        <div
          className="img-hero hero-zoom"
          style={{ position: "absolute", inset: 0 }}
        />
        <div
          style={{
            position: "absolute",
            inset: 0,
            background:
              "linear-gradient(to bottom, rgba(44,36,22,.3) 0%, rgba(44,36,22,.55) 60%, rgba(44,36,22,.75) 100%)",
          }}
        />

        {/* Decorative circles */}
        <div
          className="float-anim"
          style={{
            position: "absolute",
            top: "20%",
            right: "8%",
            width: 300,
            height: 300,
            border: "1px solid rgba(255,255,255,.08)",
            borderRadius: "50%",
            pointerEvents: "none",
          }}
        />
        <div
          className="drift-anim"
          style={{
            position: "absolute",
            bottom: "25%",
            left: "5%",
            width: 180,
            height: 180,
            border: "1px solid rgba(193,117,74,.2)",
            borderRadius: "50%",
            pointerEvents: "none",
          }}
        />

        {/* Hero content */}
        <div
          style={{
            position: "relative",
            zIndex: 2,
            maxWidth: 1200,
            margin: "0 auto",
            padding: "0 clamp(16px,5vw,64px)",
            width: "100%",
          }}
        >
          <div
            className="badge terra fade-in"
            style={{ marginBottom: 20, fontSize: 10, letterSpacing: ".2em" }}
          >
            ✦ Discover The World
          </div>
          <h1
            className="playfair fade-up d1"
            style={{
              fontSize: "clamp(40px,7vw,88px)",
              fontWeight: 900,
              color: "#fff",
              lineHeight: 1.05,
              marginBottom: 20,
              maxWidth: 720,
              textShadow: "0 2px 40px rgba(0,0,0,.3)",
            }}
          >
            Life is Short,
            <br />
            <em style={{ color: "#D4A843", fontStyle: "italic" }}>
              Travel Far
            </em>
          </h1>
          <p
            className="jost fade-up d2"
            style={{
              fontSize: "clamp(15px,2vw,19px)",
              color: "rgba(255,255,255,.8)",
              maxWidth: 520,
              lineHeight: 1.7,
              marginBottom: 40,
              fontWeight: 300,
            }}
          >
            We design bespoke journeys that awaken your sense of wonder. From
            hidden villages to iconic wonders — your story begins here.
          </p>
          <div
            className="fade-up d3"
            style={{ display: "flex", gap: 12, flexWrap: "wrap" }}
          >
            <button
              className="btn-primary"
              style={{ fontSize: 13 }}
              onClick={() => setPage("Tours")}
            >
              Explore Tours ↗
            </button>
            <button
              className="btn-white"
              style={{ fontSize: 13 }}
              onClick={() => setPage("Destinations")}
            >
              View Destinations
            </button>
          </div>
        </div>

        {/* Search bar */}
        <div
          style={{
            position: "absolute",
            bottom: 0,
            left: 0,
            right: 0,
            zIndex: 3,
            padding: "0 clamp(16px,5vw,64px)",
            transform: "translateY(50%)",
          }}
        >
          <div
            style={{
              maxWidth: 900,
              margin: "0 auto",
              background: "#fff",
              boxShadow: "0 20px 60px rgba(44,36,22,.2)",
              padding: "clamp(16px,3vw,28px)",
              display: "grid",
              gridTemplateColumns: "1fr 1fr 1fr auto",
              gap: 16,
            }}
            className="fade-up d4"
          >
            {[
              ["🌍 Destination", "Where do you want to go?"],
              ["📅 Check-in", "Select date"],
              ["👥 Travellers", "1 Adult"],
            ].map(([label, ph], i) => (
              <div key={i}>
                <div
                  style={{
                    fontSize: 10,
                    fontWeight: 600,
                    letterSpacing: ".15em",
                    textTransform: "uppercase",
                    color: "var(--warm-gray)",
                    marginBottom: 6,
                    fontFamily: "'Jost',sans-serif",
                  }}
                >
                  {label}
                </div>
                <input
                  placeholder={ph}
                  className="form-input"
                  style={{
                    padding: "8px 0",
                    border: "none",
                    borderBottom: "1.5px solid var(--cream-dark)",
                    borderRadius: 0,
                    fontSize: 14,
                  }}
                  readOnly
                />
              </div>
            ))}
            <button
              className="btn-primary"
              style={{
                alignSelf: "flex-end",
                whiteSpace: "nowrap",
                padding: "13px 28px",
              }}
            >
              Search
            </button>
          </div>
        </div>

        {/* Scroll hint */}
        <div
          className="pulse-anim hide-mobile"
          style={{
            position: "absolute",
            bottom: 36,
            right: "clamp(24px,5vw,64px)",
            zIndex: 2,
            display: "flex",
            flexDirection: "column",
            alignItems: "center",
            gap: 6,
          }}
        >
          <div
            style={{
              width: 1,
              height: 50,
              background:
                "linear-gradient(to bottom,rgba(255,255,255,.5),transparent)",
            }}
          />
          <div
            style={{
              fontSize: 10,
              letterSpacing: ".2em",
              textTransform: "uppercase",
              color: "rgba(255,255,255,.5)",
              fontFamily: "'Jost',sans-serif",
              writingMode: "vertical-rl",
            }}
          >
            Scroll
          </div>
        </div>
      </section>

      {/* ── MARQUEE ── */}
      <div
        style={{
          background: "var(--terra)",
          padding: "14px 0",
          overflow: "hidden",
          marginTop: "clamp(48px,8vw,90px)",
        }}
      >
        <div className="marquee-track">
          {[...Array(2)].map((_, r) =>
            [
              "✦ Santorini",
              "✦ Bali",
              "✦ Marrakech",
              "✦ Kyoto",
              "✦ Maldives",
              "✦ Patagonia",
              "✦ Tuscany",
              "✦ Tokyo",
              "✦ Safari",
            ].map((d, i) => (
              <span
                key={`${r}-${i}`}
                style={{
                  padding: "0 28px",
                  fontSize: 12,
                  fontFamily: "'Jost',sans-serif",
                  fontWeight: 600,
                  letterSpacing: ".25em",
                  textTransform: "uppercase",
                  color: "rgba(255,255,255,.85)",
                  whiteSpace: "nowrap",
                }}
              >
                {d}
              </span>
            )),
          )}
        </div>
      </div>

      {/* ── FEATURED DESTINATIONS ── */}
      <section
        style={{ padding: "clamp(60px,10vw,120px) clamp(16px,5vw,64px)" }}
      >
        <div style={{ maxWidth: 1200, margin: "0 auto" }}>
          <div
            style={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "flex-end",
              marginBottom: 48,
              flexWrap: "wrap",
              gap: 20,
            }}
          >
            <div>
              <div className="section-label reveal">Featured Destinations</div>
              <h2
                className="playfair reveal"
                style={{
                  fontSize: "clamp(28px,4vw,48px)",
                  fontWeight: 700,
                  color: "var(--charcoal)",
                  marginTop: 12,
                  lineHeight: 1.15,
                }}
              >
                Where Will You
                <br />
                <em style={{ color: "var(--terra)" }}>Go Next?</em>
              </h2>
            </div>
            <button
              className="btn-outline reveal"
              onClick={() => setPage("Destinations")}
            >
              See All Destinations
            </button>
          </div>

          {/* Grid */}
          <div
            style={{
              display: "grid",
              gridTemplateColumns:
                "repeat(auto-fit,minmax(clamp(260px,30vw,340px),1fr))",
              gap: 20,
            }}
          >
            {DESTINATIONS.slice(0, 3).map((d, i) => (
              <div
                key={d.id}
                className="dest-card reveal"
                style={{ height: 420 }}
                onClick={() => setPage("Destinations")}
              >
                <div
                  className={`card-img ${d.imgClass}`}
                  style={{ position: "absolute", inset: 0 }}
                />
                <div className="card-overlay" />
                <div style={{ position: "absolute", top: 20, left: 20 }}>
                  <span
                    className="badge"
                    style={{ fontSize: 10, letterSpacing: ".15em" }}
                  >
                    {d.tag}
                  </span>
                </div>
                <div
                  style={{
                    position: "absolute",
                    bottom: 24,
                    left: 24,
                    right: 24,
                  }}
                >
                  <div
                    style={{
                      color: "rgba(255,255,255,.7)",
                      fontSize: 12,
                      letterSpacing: ".2em",
                      textTransform: "uppercase",
                      fontFamily: "'Jost',sans-serif",
                      marginBottom: 4,
                    }}
                  >
                    {d.country}
                  </div>
                  <div
                    className="playfair"
                    style={{
                      fontSize: 26,
                      fontWeight: 700,
                      color: "#fff",
                      marginBottom: 8,
                    }}
                  >
                    {d.name}
                  </div>
                  <div
                    style={{
                      display: "flex",
                      justifyContent: "space-between",
                      alignItems: "center",
                    }}
                  >
                    <div
                      style={{
                        color: "rgba(255,255,255,.75)",
                        fontSize: 13,
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      <span className="stars">★</span> {d.rating} ({d.reviews})
                    </div>
                    <span
                      style={{
                        color: "#D4A843",
                        fontFamily: "'Playfair Display',serif",
                        fontWeight: 700,
                        fontSize: 18,
                      }}
                    >
                      From {d.price}
                    </span>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ── STATS ── */}
      <section
        style={{
          background: "var(--green)",
          padding: "clamp(48px,8vw,80px) clamp(16px,5vw,64px)",
        }}
      >
        <div
          style={{
            maxWidth: 1200,
            margin: "0 auto",
            display: "grid",
            gridTemplateColumns: "repeat(auto-fit,minmax(160px,1fr))",
            gap: 40,
          }}
        >
          {[
            ["12K+", "Happy Travellers"],
            ["80+", "Destinations"],
            ["15+", "Years Experience"],
            ["98%", "Satisfaction Rate"],
          ].map(([num, label], i) => (
            <div key={i} className="stat-item reveal">
              <div
                style={{
                  fontFamily: "'Playfair Display',serif",
                  fontSize: "clamp(36px,5vw,56px)",
                  fontWeight: 900,
                  color: "#D4A843",
                  lineHeight: 1,
                }}
              >
                {num}
              </div>
              <div
                style={{
                  width: 32,
                  height: 2,
                  background: "rgba(255,255,255,.3)",
                  margin: "10px auto",
                }}
              />
              <div
                style={{
                  fontFamily: "'Jost',sans-serif",
                  fontSize: 14,
                  color: "rgba(255,255,255,.75)",
                  letterSpacing: ".08em",
                }}
              >
                {label}
              </div>
            </div>
          ))}
        </div>
      </section>

      {/* ── WHY US ── */}
      <section
        style={{
          padding: "clamp(60px,10vw,120px) clamp(16px,5vw,64px)",
          background: "var(--cream-dark)",
        }}
      >
        <div
          style={{
            maxWidth: 1200,
            margin: "0 auto",
            display: "grid",
            gridTemplateColumns: "1fr 1fr",
            gap: "clamp(32px,6vw,80px)",
            alignItems: "center",
          }}
        >
          <div>
            <div className="section-label reveal">Why Wanderlust</div>
            <h2
              className="playfair reveal"
              style={{
                fontSize: "clamp(28px,3.5vw,44px)",
                fontWeight: 700,
                color: "var(--charcoal)",
                marginTop: 12,
                lineHeight: 1.2,
                marginBottom: 24,
              }}
            >
              Travel Differently.
              <br />
              <em style={{ color: "var(--terra)" }}>Travel Deeply.</em>
            </h2>
            <p
              style={{
                fontSize: 16,
                color: "var(--warm-gray)",
                lineHeight: 1.8,
                marginBottom: 32,
                fontWeight: 300,
              }}
            >
              We don't just book trips — we craft experiences that change how
              you see the world. Every itinerary is hand-built around your
              curiosity.
            </p>
            <div style={{ display: "flex", flexDirection: "column", gap: 20 }}>
              {[
                [
                  "🌿",
                  "Handpicked Experiences",
                  "Every destination vetted by our expert team of cultural explorers.",
                ],
                [
                  "🤝",
                  "Personal Travel Curator",
                  "A dedicated expert guides you from planning to touchdown.",
                ],
                [
                  "🛡️",
                  "Worry-Free Guarantee",
                  "24/7 support and full flexibility when plans change.",
                ],
              ].map(([icon, title, desc], i) => (
                <div
                  key={i}
                  className="reveal"
                  style={{ display: "flex", gap: 16, alignItems: "flex-start" }}
                >
                  <div
                    style={{
                      width: 44,
                      height: 44,
                      background: "rgba(193,117,74,.12)",
                      display: "flex",
                      alignItems: "center",
                      justifyContent: "center",
                      fontSize: 20,
                      flexShrink: 0,
                    }}
                  >
                    {icon}
                  </div>
                  <div>
                    <div
                      className="playfair"
                      style={{
                        fontSize: 16,
                        fontWeight: 600,
                        color: "var(--charcoal)",
                        marginBottom: 4,
                      }}
                    >
                      {title}
                    </div>
                    <div
                      style={{
                        fontSize: 14,
                        color: "var(--warm-gray)",
                        lineHeight: 1.7,
                      }}
                    >
                      {desc}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
          <div className="reveal slide-left" style={{ position: "relative" }}>
            <div
              className="img-about"
              style={{ height: "clamp(300px,50vw,520px)", width: "100%" }}
            />
            <div
              style={{
                position: "absolute",
                bottom: -20,
                left: -20,
                background: "#fff",
                padding: 24,
                boxShadow: "0 8px 32px rgba(44,36,22,.12)",
              }}
            >
              <div
                className="playfair"
                style={{ fontSize: 36, fontWeight: 900, color: "var(--terra)" }}
              >
                15+
              </div>
              <div
                style={{
                  fontSize: 12,
                  fontFamily: "'Jost',sans-serif",
                  letterSpacing: ".15em",
                  textTransform: "uppercase",
                  color: "var(--warm-gray)",
                }}
              >
                Years of Excellence
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* ── FEATURED TOURS ── */}
      <section
        style={{ padding: "clamp(60px,10vw,120px) clamp(16px,5vw,64px)" }}
      >
        <div style={{ maxWidth: 1200, margin: "0 auto" }}>
          <div style={{ textAlign: "center", marginBottom: 56 }}>
            <div
              className="section-label reveal"
              style={{ justifyContent: "center" }}
            >
              Curated Journeys
            </div>
            <h2
              className="playfair reveal"
              style={{
                fontSize: "clamp(28px,4vw,48px)",
                fontWeight: 700,
                marginTop: 12,
                color: "var(--charcoal)",
              }}
            >
              Our Signature <em style={{ color: "var(--terra)" }}>Tours</em>
            </h2>
          </div>
          <div
            style={{
              display: "grid",
              gridTemplateColumns: "repeat(auto-fit,minmax(260px,1fr))",
              gap: 24,
            }}
          >
            {TOURS.slice(0, 4).map((t, i) => (
              <div
                key={t.id}
                className="tour-card reveal"
                onClick={() => setPage("Tours")}
                style={{ cursor: "pointer" }}
              >
                <div
                  style={{
                    height: 220,
                    overflow: "hidden",
                    position: "relative",
                  }}
                >
                  <div
                    className={`tour-img ${t.imgClass}`}
                    style={{
                      width: "100%",
                      height: "100%",
                      position: "absolute",
                      inset: 0,
                    }}
                  />
                  <div style={{ position: "absolute", top: 16, left: 16 }}>
                    <span className="badge terra" style={{ fontSize: 10 }}>
                      {t.tag}
                    </span>
                  </div>
                  <div style={{ position: "absolute", bottom: 16, right: 16 }}>
                    <div className="price-tag">${t.price.toLocaleString()}</div>
                  </div>
                </div>
                <div style={{ padding: 24 }}>
                  <div
                    style={{
                      color: "var(--warm-gray)",
                      fontSize: 12,
                      letterSpacing: ".15em",
                      textTransform: "uppercase",
                      marginBottom: 6,
                      fontFamily: "'Jost',sans-serif",
                    }}
                  >
                    {t.region}
                  </div>
                  <div
                    className="playfair"
                    style={{
                      fontSize: 20,
                      fontWeight: 700,
                      color: "var(--charcoal)",
                      marginBottom: 10,
                    }}
                  >
                    {t.name}
                  </div>
                  <p
                    style={{
                      fontSize: 14,
                      color: "var(--warm-gray)",
                      lineHeight: 1.7,
                      marginBottom: 16,
                    }}
                  >
                    {t.desc}
                  </p>
                  <div
                    style={{
                      display: "flex",
                      justifyContent: "space-between",
                      alignItems: "center",
                      paddingTop: 16,
                      borderTop: "1px solid var(--cream-dark)",
                    }}
                  >
                    <div
                      style={{
                        display: "flex",
                        gap: 16,
                        fontSize: 13,
                        color: "var(--warm-gray)",
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      <span>🕐 {t.days} days</span>
                      <span>👥 Max {t.group}</span>
                    </div>
                    <span
                      style={{
                        color: "var(--terra)",
                        fontWeight: 600,
                        fontSize: 13,
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      Details →
                    </span>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ── TESTIMONIALS ── */}
      <section
        style={{
          background: "var(--green-pale)",
          padding: "clamp(60px,10vw,100px) clamp(16px,5vw,64px)",
        }}
      >
        <div style={{ maxWidth: 1200, margin: "0 auto" }}>
          <div style={{ textAlign: "center", marginBottom: 48 }}>
            <div
              className="section-label reveal"
              style={{ justifyContent: "center" }}
            >
              Traveller Stories
            </div>
            <h2
              className="playfair reveal"
              style={{
                fontSize: "clamp(26px,3.5vw,42px)",
                fontWeight: 700,
                marginTop: 12,
                color: "var(--charcoal)",
              }}
            >
              Voices from the <em style={{ color: "var(--terra)" }}>Road</em>
            </h2>
          </div>
          <div
            style={{
              display: "grid",
              gridTemplateColumns: "repeat(auto-fit,minmax(280px,1fr))",
              gap: 24,
            }}
          >
            {TESTIMONIALS.map((t, i) => (
              <div key={i} className="testimonial-card reveal">
                <div
                  className="stars"
                  style={{ fontSize: 18, marginBottom: 14 }}
                >
                  {"★".repeat(t.rating)}
                </div>
                <p
                  style={{
                    fontSize: 15,
                    color: "var(--charcoal)",
                    lineHeight: 1.8,
                    fontStyle: "italic",
                    marginBottom: 20,
                  }}
                >
                  "{t.text}"
                </p>
                <div style={{ display: "flex", alignItems: "center", gap: 12 }}>
                  <div
                    style={{
                      width: 44,
                      height: 44,
                      background: "var(--terra)",
                      borderRadius: "50%",
                      display: "flex",
                      alignItems: "center",
                      justifyContent: "center",
                      color: "#fff",
                      fontWeight: 700,
                      fontSize: 14,
                      fontFamily: "'Jost',sans-serif",
                    }}
                  >
                    {t.avatar}
                  </div>
                  <div>
                    <div
                      className="playfair"
                      style={{
                        fontSize: 15,
                        fontWeight: 600,
                        color: "var(--charcoal)",
                      }}
                    >
                      {t.name}
                    </div>
                    <div
                      style={{
                        fontSize: 12,
                        color: "var(--warm-gray)",
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      {t.location}
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ── CTA BANNER ── */}
      <section
        style={{
          background: "var(--brown)",
          padding: "clamp(48px,8vw,80px) clamp(16px,5vw,64px)",
          textAlign: "center",
        }}
      >
        <div style={{ maxWidth: 700, margin: "0 auto" }}>
          <div
            className="section-label reveal"
            style={{ justifyContent: "center", color: "rgba(212,168,67,.8)" }}
          >
            Start Your Journey
          </div>
          <h2
            className="playfair reveal"
            style={{
              fontSize: "clamp(28px,4vw,52px)",
              fontWeight: 900,
              color: "#fff",
              marginTop: 12,
              marginBottom: 16,
              lineHeight: 1.15,
            }}
          >
            Your Dream Trip
            <br />
            is One Click Away
          </h2>
          <p
            className="reveal"
            style={{
              fontSize: 17,
              color: "rgba(255,255,255,.65)",
              marginBottom: 36,
              lineHeight: 1.7,
              fontWeight: 300,
            }}
          >
            Let our travel experts design the perfect itinerary for you. No
            cookie-cutter tours — just pure, personalised adventure.
          </p>
          <div
            className="reveal"
            style={{
              display: "flex",
              gap: 12,
              justifyContent: "center",
              flexWrap: "wrap",
            }}
          >
            <button
              className="btn-primary"
              onClick={() => setPage("Contact")}
              style={{ background: "#D4A843", color: "var(--charcoal)" }}
            >
              Plan My Trip ✦
            </button>
            <button className="btn-white" onClick={() => setPage("Tours")}>
              Browse Tours
            </button>
          </div>
        </div>
      </section>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════════
// PAGE: DESTINATIONS
// ═══════════════════════════════════════════════════════════════

function DestinationsPage({ setPage }) {
  useReveal();
  const [filter, setFilter] = useState("All");
  const tags = ["All", ...[...new Set(DESTINATIONS.map((d) => d.tag))]];
  const filtered =
    filter === "All"
      ? DESTINATIONS
      : DESTINATIONS.filter((d) => d.tag === filter);

  return (
    <div>
      {/* Hero */}
      <div
        style={{
          paddingTop: 72,
          background: "var(--green)",
          padding: "120px clamp(16px,5vw,64px) 60px",
          textAlign: "center",
        }}
      >
        <div
          className="section-label fade-in"
          style={{ justifyContent: "center", color: "rgba(212,168,67,.8)" }}
        >
          Explore
        </div>
        <h1
          className="playfair fade-up"
          style={{
            fontSize: "clamp(36px,6vw,72px)",
            fontWeight: 900,
            color: "#fff",
            marginTop: 12,
          }}
        >
          Our Destinations
        </h1>
        <p
          className="fade-up d2"
          style={{
            fontSize: 17,
            color: "rgba(255,255,255,.65)",
            maxWidth: 540,
            margin: "16px auto 0",
            lineHeight: 1.7,
            fontWeight: 300,
          }}
        >
          From ancient ruins to tropical shores — discover the world's most
          extraordinary places.
        </p>
      </div>

      {/* Filters */}
      <div
        style={{
          background: "#fff",
          borderBottom: "1px solid var(--cream-dark)",
          padding: "20px clamp(16px,5vw,64px)",
          display: "flex",
          gap: 10,
          justifyContent: "center",
          flexWrap: "wrap",
        }}
      >
        {tags.map((t) => (
          <button
            key={t}
            onClick={() => setFilter(t)}
            style={{
              padding: "8px 22px",
              border: `1.5px solid ${filter === t ? "var(--terra)" : "var(--cream-dark)"}`,
              background: filter === t ? "var(--terra)" : "transparent",
              color: filter === t ? "#fff" : "var(--warm-gray)",
              fontFamily: "'Jost',sans-serif",
              fontSize: 13,
              fontWeight: 500,
              letterSpacing: ".06em",
              cursor: "pointer",
              transition: "all .2s",
            }}
          >
            {t}
          </button>
        ))}
      </div>

      {/* Grid */}
      <section style={{ padding: "clamp(40px,6vw,80px) clamp(16px,5vw,64px)" }}>
        <div
          style={{
            maxWidth: 1200,
            margin: "0 auto",
            display: "grid",
            gridTemplateColumns:
              "repeat(auto-fill,minmax(clamp(240px,28vw,360px),1fr))",
            gap: 24,
          }}
        >
          {filtered.map((d, i) => (
            <div
              key={d.id}
              className="dest-card scale-in"
              style={{ height: 400, animationDelay: `${i * 0.1}s` }}
            >
              <div
                className={`card-img ${d.imgClass}`}
                style={{ position: "absolute", inset: 0 }}
              />
              <div className="card-overlay" />
              <div
                style={{
                  position: "absolute",
                  top: 20,
                  left: 20,
                  display: "flex",
                  gap: 8,
                }}
              >
                <span className="badge" style={{ fontSize: 10 }}>
                  {d.tag}
                </span>
              </div>
              <div
                style={{
                  position: "absolute",
                  bottom: 24,
                  left: 24,
                  right: 24,
                }}
              >
                <div
                  style={{
                    color: "rgba(255,255,255,.65)",
                    fontSize: 11,
                    letterSpacing: ".2em",
                    textTransform: "uppercase",
                    fontFamily: "'Jost',sans-serif",
                    marginBottom: 4,
                  }}
                >
                  {d.country}
                </div>
                <div
                  className="playfair"
                  style={{
                    fontSize: 24,
                    fontWeight: 700,
                    color: "#fff",
                    marginBottom: 10,
                  }}
                >
                  {d.name}
                </div>
                <div
                  style={{
                    display: "flex",
                    justifyContent: "space-between",
                    alignItems: "center",
                  }}
                >
                  <div
                    style={{
                      fontSize: 13,
                      color: "rgba(255,255,255,.75)",
                      fontFamily: "'Jost',sans-serif",
                    }}
                  >
                    <span className="stars">★</span> {d.rating} · {d.reviews}{" "}
                    reviews
                  </div>
                  <span
                    style={{
                      color: "#D4A843",
                      fontFamily: "'Playfair Display',serif",
                      fontWeight: 700,
                      fontSize: 17,
                    }}
                  >
                    From {d.price}
                  </span>
                </div>
                <button
                  className="btn-white"
                  onClick={() => setPage("Contact")}
                  style={{
                    width: "100%",
                    marginTop: 14,
                    padding: "10px",
                    fontSize: 12,
                    justifyContent: "center",
                  }}
                >
                  Enquire Now
                </button>
              </div>
            </div>
          ))}
        </div>
      </section>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════════
// PAGE: TOURS
// ═══════════════════════════════════════════════════════════════

function ToursPage({ setPage }) {
  useReveal();
  return (
    <div>
      <div
        style={{
          paddingTop: 72,
          background: "var(--terra)",
          padding: "120px clamp(16px,5vw,64px) 60px",
          textAlign: "center",
        }}
      >
        <div
          className="section-label fade-in"
          style={{ justifyContent: "center", color: "rgba(255,255,255,.6)" }}
        >
          Curated Journeys
        </div>
        <h1
          className="playfair fade-up"
          style={{
            fontSize: "clamp(36px,6vw,72px)",
            fontWeight: 900,
            color: "#fff",
            marginTop: 12,
          }}
        >
          Our Tours
        </h1>
        <p
          className="fade-up d2"
          style={{
            fontSize: 17,
            color: "rgba(255,255,255,.7)",
            maxWidth: 540,
            margin: "16px auto 0",
            lineHeight: 1.7,
            fontWeight: 300,
          }}
        >
          Signature experiences crafted by our expert team. Small groups, big
          adventures.
        </p>
      </div>

      <section style={{ padding: "clamp(40px,6vw,80px) clamp(16px,5vw,64px)" }}>
        <div
          style={{
            maxWidth: 1200,
            margin: "0 auto",
            display: "flex",
            flexDirection: "column",
            gap: 32,
          }}
        >
          {TOURS.map((t, i) => (
            <div
              key={t.id}
              className="tour-card reveal"
              style={{
                display: "grid",
                gridTemplateColumns: "clamp(200px,35%,380px) 1fr",
                overflow: "hidden",
              }}
            >
              <div
                style={{
                  height: "clamp(220px,30vw,320px)",
                  overflow: "hidden",
                  position: "relative",
                }}
              >
                <div
                  className={`tour-img ${t.imgClass}`}
                  style={{
                    position: "absolute",
                    inset: 0,
                    width: "100%",
                    height: "100%",
                  }}
                />
                <div style={{ position: "absolute", top: 16, left: 16 }}>
                  <span className="badge terra" style={{ fontSize: 10 }}>
                    {t.tag}
                  </span>
                </div>
              </div>
              <div style={{ padding: "clamp(20px,3vw,40px)" }}>
                <div
                  style={{
                    color: "var(--warm-gray)",
                    fontSize: 12,
                    letterSpacing: ".2em",
                    textTransform: "uppercase",
                    marginBottom: 8,
                    fontFamily: "'Jost',sans-serif",
                  }}
                >
                  {t.region}
                </div>
                <div
                  className="playfair"
                  style={{
                    fontSize: "clamp(20px,2.5vw,30px)",
                    fontWeight: 700,
                    color: "var(--charcoal)",
                    marginBottom: 14,
                  }}
                >
                  {t.name}
                </div>
                <p
                  style={{
                    fontSize: 15,
                    color: "var(--warm-gray)",
                    lineHeight: 1.8,
                    marginBottom: 20,
                    fontWeight: 300,
                  }}
                >
                  {t.desc}
                </p>

                <div
                  style={{
                    display: "flex",
                    flexWrap: "wrap",
                    gap: 16,
                    marginBottom: 24,
                  }}
                >
                  {[
                    ["🗓", "Duration", `${t.days} Days`],
                    ["👥", "Group Size", `Max ${t.group}`],
                    ["🌟", "Difficulty", "Moderate"],
                    ["🏨", "Accommodation", "4★ Hotels"],
                  ].map(([icon, label, val]) => (
                    <div
                      key={label}
                      style={{
                        background: "var(--cream)",
                        padding: "10px 16px",
                        minWidth: 100,
                      }}
                    >
                      <div
                        style={{
                          fontSize: 11,
                          color: "var(--warm-gray)",
                          letterSpacing: ".12em",
                          textTransform: "uppercase",
                          fontFamily: "'Jost',sans-serif",
                          marginBottom: 2,
                        }}
                      >
                        {label}
                      </div>
                      <div
                        style={{
                          fontFamily: "'Jost',sans-serif",
                          fontWeight: 600,
                          color: "var(--charcoal)",
                          fontSize: 14,
                        }}
                      >
                        {icon} {val}
                      </div>
                    </div>
                  ))}
                </div>

                <div
                  style={{
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "space-between",
                    flexWrap: "wrap",
                    gap: 16,
                  }}
                >
                  <div>
                    <div
                      style={{
                        fontSize: 11,
                        color: "var(--warm-gray)",
                        letterSpacing: ".15em",
                        textTransform: "uppercase",
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      Price per person
                    </div>
                    <div
                      className="playfair"
                      style={{
                        fontSize: "clamp(24px,3vw,36px)",
                        fontWeight: 900,
                        color: "var(--terra)",
                      }}
                    >
                      ${t.price.toLocaleString()}
                    </div>
                  </div>
                  <button
                    className="btn-primary"
                    onClick={() => setPage("Contact")}
                  >
                    Book This Tour
                  </button>
                </div>
              </div>
            </div>
          ))}
        </div>
      </section>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════════
// PAGE: ABOUT
// ═══════════════════════════════════════════════════════════════

function AboutPage({ setPage }) {
  useReveal();
  return (
    <div>
      <div
        style={{
          paddingTop: 72,
          background: "var(--brown)",
          padding: "120px clamp(16px,5vw,64px) 60px",
          textAlign: "center",
        }}
      >
        <div
          className="section-label fade-in"
          style={{ justifyContent: "center", color: "rgba(212,168,67,.7)" }}
        >
          Our Story
        </div>
        <h1
          className="playfair fade-up"
          style={{
            fontSize: "clamp(36px,6vw,72px)",
            fontWeight: 900,
            color: "#fff",
            marginTop: 12,
          }}
        >
          About Wanderlust
        </h1>
        <p
          className="fade-up d2"
          style={{
            fontSize: 17,
            color: "rgba(255,255,255,.65)",
            maxWidth: 540,
            margin: "16px auto 0",
            lineHeight: 1.7,
            fontWeight: 300,
          }}
        >
          Born from a love of discovery. Built on a promise of authenticity.
        </p>
      </div>

      {/* Mission */}
      <section
        style={{ padding: "clamp(60px,10vw,100px) clamp(16px,5vw,64px)" }}
      >
        <div
          style={{
            maxWidth: 1200,
            margin: "0 auto",
            display: "grid",
            gridTemplateColumns: "1fr 1fr",
            gap: "clamp(32px,6vw,80px)",
            alignItems: "center",
          }}
        >
          <div>
            <div className="section-label reveal">Our Mission</div>
            <h2
              className="playfair reveal"
              style={{
                fontSize: "clamp(26px,3.5vw,42px)",
                fontWeight: 700,
                color: "var(--charcoal)",
                marginTop: 12,
                lineHeight: 1.2,
                marginBottom: 20,
              }}
            >
              More Than Travel.
              <br />
              <em style={{ color: "var(--terra)" }}>Transformation.</em>
            </h2>
            <p
              style={{
                fontSize: 16,
                color: "var(--warm-gray)",
                lineHeight: 1.8,
                marginBottom: 16,
                fontWeight: 300,
              }}
            >
              Wanderlust was founded in 2008 by Elena Vasquez after a solo
              journey across Southeast Asia changed her life. She returned home
              determined to share that gift with others.
            </p>
            <p
              style={{
                fontSize: 16,
                color: "var(--warm-gray)",
                lineHeight: 1.8,
                marginBottom: 28,
                fontWeight: 300,
              }}
            >
              Today we're a team of 40+ passionate explorers, cultural scholars,
              and local experts — united by one belief: travel done right
              changes everything.
            </p>
            <div
              style={{
                display: "grid",
                gridTemplateColumns: "1fr 1fr",
                gap: 20,
              }}
            >
              {[
                ["🌍", "B Corp Certified"],
                ["🌱", "Carbon Neutral by 2025"],
                ["🤝", "180+ Local Partners"],
                ["🏆", "Award-Winning Since 2012"],
              ].map(([icon, text]) => (
                <div
                  key={text}
                  style={{
                    display: "flex",
                    gap: 10,
                    alignItems: "center",
                    fontSize: 14,
                    color: "var(--charcoal)",
                    fontFamily: "'Jost',sans-serif",
                  }}
                >
                  <span style={{ fontSize: 20 }}>{icon}</span> {text}
                </div>
              ))}
            </div>
          </div>
          <div className="reveal" style={{ position: "relative" }}>
            <div
              className="img-about"
              style={{ width: "100%", height: "clamp(280px,45vw,480px)" }}
            />
            <div
              style={{
                position: "absolute",
                top: -16,
                right: -16,
                background: "var(--terra)",
                padding: "20px 24px",
                textAlign: "center",
              }}
            >
              <div
                className="playfair"
                style={{ fontSize: 32, fontWeight: 900, color: "#fff" }}
              >
                2008
              </div>
              <div
                style={{
                  fontSize: 11,
                  color: "rgba(255,255,255,.75)",
                  letterSpacing: ".15em",
                  textTransform: "uppercase",
                  fontFamily: "'Jost',sans-serif",
                }}
              >
                Est.
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Values */}
      <section
        style={{
          background: "var(--cream-dark)",
          padding: "clamp(60px,8vw,100px) clamp(16px,5vw,64px)",
        }}
      >
        <div style={{ maxWidth: 1200, margin: "0 auto", textAlign: "center" }}>
          <div
            className="section-label reveal"
            style={{ justifyContent: "center" }}
          >
            What We Stand For
          </div>
          <h2
            className="playfair reveal"
            style={{
              fontSize: "clamp(26px,3.5vw,42px)",
              fontWeight: 700,
              marginTop: 12,
              marginBottom: 48,
              color: "var(--charcoal)",
            }}
          >
            Our <em style={{ color: "var(--terra)" }}>Values</em>
          </h2>
          <div
            style={{
              display: "grid",
              gridTemplateColumns: "repeat(auto-fit,minmax(220px,1fr))",
              gap: 24,
            }}
          >
            {[
              [
                "🌿",
                "Sustainability",
                "We offset 150% of our carbon footprint and partner only with eco-conscious operators.",
              ],
              [
                "🫂",
                "Community",
                "Every tour directly benefits local communities through fair wages and cultural programs.",
              ],
              [
                "✦",
                "Authenticity",
                "No tourist traps. Only genuine experiences that connect you to real places and people.",
              ],
              [
                "❤️",
                "Passion",
                "We love travel — and it shows in every detail of every journey we craft.",
              ],
            ].map(([icon, title, desc]) => (
              <div
                key={title}
                className="reveal"
                style={{ background: "#fff", padding: 32, textAlign: "center" }}
              >
                <div style={{ fontSize: 36, marginBottom: 16 }}>{icon}</div>
                <div
                  className="playfair"
                  style={{
                    fontSize: 18,
                    fontWeight: 700,
                    color: "var(--charcoal)",
                    marginBottom: 10,
                  }}
                >
                  {title}
                </div>
                <p
                  style={{
                    fontSize: 14,
                    color: "var(--warm-gray)",
                    lineHeight: 1.8,
                  }}
                >
                  {desc}
                </p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Team */}
      <section
        style={{ padding: "clamp(60px,8vw,100px) clamp(16px,5vw,64px)" }}
      >
        <div style={{ maxWidth: 1200, margin: "0 auto", textAlign: "center" }}>
          <div
            className="section-label reveal"
            style={{ justifyContent: "center" }}
          >
            The People Behind the Magic
          </div>
          <h2
            className="playfair reveal"
            style={{
              fontSize: "clamp(26px,3.5vw,42px)",
              fontWeight: 700,
              marginTop: 12,
              marginBottom: 48,
              color: "var(--charcoal)",
            }}
          >
            Meet the <em style={{ color: "var(--terra)" }}>Team</em>
          </h2>
          <div
            style={{
              display: "grid",
              gridTemplateColumns: "repeat(auto-fit,minmax(240px,1fr))",
              gap: 28,
            }}
          >
            {TEAM.map((m, i) => (
              <div key={i} className="reveal" style={{ textAlign: "center" }}>
                <div
                  className={`${m.imgClass}`}
                  style={{ width: "100%", height: 280, marginBottom: 20 }}
                />
                <div
                  className="playfair"
                  style={{
                    fontSize: 20,
                    fontWeight: 700,
                    color: "var(--charcoal)",
                    marginBottom: 4,
                  }}
                >
                  {m.name}
                </div>
                <div
                  style={{
                    fontSize: 12,
                    letterSpacing: ".2em",
                    textTransform: "uppercase",
                    color: "var(--terra)",
                    fontFamily: "'Jost',sans-serif",
                    marginBottom: 10,
                  }}
                >
                  {m.role}
                </div>
                <p
                  style={{
                    fontSize: 14,
                    color: "var(--warm-gray)",
                    lineHeight: 1.7,
                  }}
                >
                  {m.bio}
                </p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CTA */}
      <section
        style={{
          background: "var(--green)",
          padding: "clamp(48px,7vw,80px) clamp(16px,5vw,64px)",
          textAlign: "center",
        }}
      >
        <h2
          className="playfair reveal"
          style={{
            fontSize: "clamp(26px,3.5vw,42px)",
            fontWeight: 700,
            color: "#fff",
            marginBottom: 16,
          }}
        >
          Ready to Travel <em style={{ color: "#D4A843" }}>With Us?</em>
        </h2>
        <p
          className="reveal"
          style={{
            fontSize: 16,
            color: "rgba(255,255,255,.65)",
            marginBottom: 28,
          }}
        >
          Join over 12,000 travellers who've trusted us with their most precious
          moments.
        </p>
        <div
          className="reveal"
          style={{
            display: "flex",
            gap: 12,
            justifyContent: "center",
            flexWrap: "wrap",
          }}
        >
          <button className="btn-white" onClick={() => setPage("Contact")}>
            Get In Touch
          </button>
          <button
            onClick={() => setPage("Tours")}
            style={{
              color: "rgba(255,255,255,.8)",
              background: "transparent",
              border: "1.5px solid rgba(255,255,255,.3)",
              padding: "13px 32px",
              fontFamily: "'Jost',sans-serif",
              fontWeight: 600,
              fontSize: 13,
              letterSpacing: ".1em",
              textTransform: "uppercase",
              cursor: "pointer",
              transition: "all .2s",
            }}
          >
            Explore Tours
          </button>
        </div>
      </section>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════════
// PAGE: CONTACT
// ═══════════════════════════════════════════════════════════════

function ContactPage() {
  useReveal();
  const [form, setForm] = useState({
    name: "",
    email: "",
    phone: "",
    dest: "",
    dates: "",
    budget: "",
    msg: "",
  });
  const [sent, setSent] = useState(false);

  const handleSubmit = (e) => {
    e.preventDefault();
    setSent(true);
  };

  return (
    <div>
      <div
        style={{
          paddingTop: 72,
          background: "var(--terra)",
          padding: "120px clamp(16px,5vw,64px) 60px",
          textAlign: "center",
        }}
      >
        <div
          className="section-label fade-in"
          style={{ justifyContent: "center", color: "rgba(255,255,255,.65)" }}
        >
          Let's Talk
        </div>
        <h1
          className="playfair fade-up"
          style={{
            fontSize: "clamp(36px,6vw,72px)",
            fontWeight: 900,
            color: "#fff",
            marginTop: 12,
          }}
        >
          Plan Your Journey
        </h1>
        <p
          className="fade-up d2"
          style={{
            fontSize: 17,
            color: "rgba(255,255,255,.7)",
            maxWidth: 500,
            margin: "16px auto 0",
            lineHeight: 1.7,
            fontWeight: 300,
          }}
        >
          Tell us your dream. We'll make it real.
        </p>
      </div>

      <section
        style={{ padding: "clamp(48px,8vw,100px) clamp(16px,5vw,64px)" }}
      >
        <div
          style={{
            maxWidth: 1100,
            margin: "0 auto",
            display: "grid",
            gridTemplateColumns: "1fr 1.5fr",
            gap: "clamp(32px,6vw,80px)",
          }}
        >
          {/* Info */}
          <div>
            <div className="section-label reveal">Contact Info</div>
            <h2
              className="playfair reveal"
              style={{
                fontSize: "clamp(22px,3vw,34px)",
                fontWeight: 700,
                color: "var(--charcoal)",
                marginTop: 12,
                marginBottom: 28,
              }}
            >
              We'd Love to
              <br />
              <em style={{ color: "var(--terra)" }}>Hear From You</em>
            </h2>
            <div
              style={{
                display: "flex",
                flexDirection: "column",
                gap: 20,
                marginBottom: 36,
              }}
            >
              {[
                ["📍", "Visit Us", "42 Explorer Lane, San Francisco, CA 94102"],
                ["📞", "Call Us", "+1 (800) 555-WANDER"],
                ["✉️", "Email Us", "hello@wanderlust.travel"],
                ["🕐", "Office Hours", "Mon–Fri: 9:00am – 6:00pm PST"],
              ].map(([icon, label, val]) => (
                <div
                  key={label}
                  className="reveal"
                  style={{ display: "flex", gap: 16, alignItems: "flex-start" }}
                >
                  <div
                    style={{
                      width: 44,
                      height: 44,
                      background: "rgba(193,117,74,.1)",
                      display: "flex",
                      alignItems: "center",
                      justifyContent: "center",
                      fontSize: 18,
                      flexShrink: 0,
                    }}
                  >
                    {icon}
                  </div>
                  <div>
                    <div
                      style={{
                        fontSize: 11,
                        color: "var(--warm-gray)",
                        letterSpacing: ".15em",
                        textTransform: "uppercase",
                        fontFamily: "'Jost',sans-serif",
                        marginBottom: 3,
                      }}
                    >
                      {label}
                    </div>
                    <div
                      style={{
                        fontSize: 15,
                        color: "var(--charcoal)",
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      {val}
                    </div>
                  </div>
                </div>
              ))}
            </div>

            {/* Map placeholder */}
            <div
              className="reveal"
              style={{
                background: "var(--green)",
                height: 220,
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                position: "relative",
                overflow: "hidden",
              }}
            >
              <div
                style={{
                  textAlign: "center",
                  color: "rgba(255,255,255,.7)",
                  fontFamily: "'Jost',sans-serif",
                }}
              >
                <div style={{ fontSize: 36, marginBottom: 8 }}>📍</div>
                <div style={{ fontSize: 14 }}>San Francisco, CA</div>
              </div>
              {/* Decorative grid */}
              <div style={{ position: "absolute", inset: 0, opacity: 0.15 }}>
                {[...Array(6)].map((_, i) => (
                  <div
                    key={i}
                    style={{
                      position: "absolute",
                      left: 0,
                      right: 0,
                      top: `${i * 20}%`,
                      height: 1,
                      background: "#fff",
                    }}
                  />
                ))}
                {[...Array(8)].map((_, i) => (
                  <div
                    key={i}
                    style={{
                      position: "absolute",
                      top: 0,
                      bottom: 0,
                      left: `${i * 14.2}%`,
                      width: 1,
                      background: "#fff",
                    }}
                  />
                ))}
              </div>
            </div>
          </div>

          {/* Form */}
          <div className="reveal slide-left">
            {sent ? (
              <div
                style={{
                  background: "var(--green-pale)",
                  border: "1px solid var(--green)",
                  padding: 48,
                  textAlign: "center",
                }}
              >
                <div style={{ fontSize: 56, marginBottom: 16 }}>✈️</div>
                <div
                  className="playfair"
                  style={{
                    fontSize: 28,
                    fontWeight: 700,
                    color: "var(--charcoal)",
                    marginBottom: 10,
                  }}
                >
                  We've Got Your Request!
                </div>
                <p
                  style={{
                    fontSize: 16,
                    color: "var(--warm-gray)",
                    lineHeight: 1.7,
                  }}
                >
                  Our travel team will reach out within 24 hours to start
                  crafting your perfect journey.
                </p>
                <button
                  className="btn-primary"
                  onClick={() => setSent(false)}
                  style={{ marginTop: 24 }}
                >
                  Send Another Enquiry
                </button>
              </div>
            ) : (
              <form
                onSubmit={handleSubmit}
                style={{ display: "flex", flexDirection: "column", gap: 16 }}
              >
                <div
                  style={{
                    display: "grid",
                    gridTemplateColumns: "1fr 1fr",
                    gap: 16,
                  }}
                >
                  <div>
                    <label
                      style={{
                        display: "block",
                        fontSize: 11,
                        fontWeight: 600,
                        letterSpacing: ".15em",
                        textTransform: "uppercase",
                        color: "var(--warm-gray)",
                        marginBottom: 6,
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      Full Name *
                    </label>
                    <input
                      className="form-input"
                      required
                      placeholder="Your name"
                      value={form.name}
                      onChange={(e) =>
                        setForm((f) => ({ ...f, name: e.target.value }))
                      }
                    />
                  </div>
                  <div>
                    <label
                      style={{
                        display: "block",
                        fontSize: 11,
                        fontWeight: 600,
                        letterSpacing: ".15em",
                        textTransform: "uppercase",
                        color: "var(--warm-gray)",
                        marginBottom: 6,
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      Email *
                    </label>
                    <input
                      className="form-input"
                      required
                      type="email"
                      placeholder="your@email.com"
                      value={form.email}
                      onChange={(e) =>
                        setForm((f) => ({ ...f, email: e.target.value }))
                      }
                    />
                  </div>
                </div>
                <div
                  style={{
                    display: "grid",
                    gridTemplateColumns: "1fr 1fr",
                    gap: 16,
                  }}
                >
                  <div>
                    <label
                      style={{
                        display: "block",
                        fontSize: 11,
                        fontWeight: 600,
                        letterSpacing: ".15em",
                        textTransform: "uppercase",
                        color: "var(--warm-gray)",
                        marginBottom: 6,
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      Phone
                    </label>
                    <input
                      className="form-input"
                      placeholder="+1 (555) 000-0000"
                      value={form.phone}
                      onChange={(e) =>
                        setForm((f) => ({ ...f, phone: e.target.value }))
                      }
                    />
                  </div>
                  <div>
                    <label
                      style={{
                        display: "block",
                        fontSize: 11,
                        fontWeight: 600,
                        letterSpacing: ".15em",
                        textTransform: "uppercase",
                        color: "var(--warm-gray)",
                        marginBottom: 6,
                        fontFamily: "'Jost',sans-serif",
                      }}
                    >
                      Budget (USD)
                    </label>
                    <select
                      className="form-input"
                      value={form.budget}
                      onChange={(e) =>
                        setForm((f) => ({ ...f, budget: e.target.value }))
                      }
                      style={{ cursor: "pointer" }}
                    >
                      <option value="">Select range</option>
                      {[
                        "Under $1,000",
                        "$1,000–$2,500",
                        "$2,500–$5,000",
                        "$5,000–$10,000",
                        "$10,000+",
                      ].map((b) => (
                        <option key={b}>{b}</option>
                      ))}
                    </select>
                  </div>
                </div>
                <div>
                  <label
                    style={{
                      display: "block",
                      fontSize: 11,
                      fontWeight: 600,
                      letterSpacing: ".15em",
                      textTransform: "uppercase",
                      color: "var(--warm-gray)",
                      marginBottom: 6,
                      fontFamily: "'Jost',sans-serif",
                    }}
                  >
                    Dream Destination
                  </label>
                  <input
                    className="form-input"
                    placeholder="e.g. Santorini, Bali, Japan..."
                    value={form.dest}
                    onChange={(e) =>
                      setForm((f) => ({ ...f, dest: e.target.value }))
                    }
                  />
                </div>
                <div>
                  <label
                    style={{
                      display: "block",
                      fontSize: 11,
                      fontWeight: 600,
                      letterSpacing: ".15em",
                      textTransform: "uppercase",
                      color: "var(--warm-gray)",
                      marginBottom: 6,
                      fontFamily: "'Jost',sans-serif",
                    }}
                  >
                    Travel Dates
                  </label>
                  <input
                    className="form-input"
                    placeholder="e.g. March 2025, flexible..."
                    value={form.dates}
                    onChange={(e) =>
                      setForm((f) => ({ ...f, dates: e.target.value }))
                    }
                  />
                </div>
                <div>
                  <label
                    style={{
                      display: "block",
                      fontSize: 11,
                      fontWeight: 600,
                      letterSpacing: ".15em",
                      textTransform: "uppercase",
                      color: "var(--warm-gray)",
                      marginBottom: 6,
                      fontFamily: "'Jost',sans-serif",
                    }}
                  >
                    Tell Us More *
                  </label>
                  <textarea
                    className="form-input"
                    required
                    placeholder="Share your dream trip — interests, group size, special requests..."
                    value={form.msg}
                    onChange={(e) =>
                      setForm((f) => ({ ...f, msg: e.target.value }))
                    }
                  />
                </div>
                <button
                  type="submit"
                  className="btn-primary"
                  style={{
                    justifyContent: "center",
                    fontSize: 13,
                    padding: "16px",
                  }}
                >
                  Send My Enquiry ✦
                </button>
                <p
                  style={{
                    fontSize: 12,
                    color: "var(--warm-gray)",
                    textAlign: "center",
                  }}
                >
                  We respond within 24 hours. No spam, ever.
                </p>
              </form>
            )}
          </div>
        </div>
      </section>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════════
// APP ROOT
// ═══════════════════════════════════════════════════════════════

export default function WanderlustWebsite() {
  const [page, setPage] = useState("Home");

  // Scroll to top on page change
  useEffect(() => {
    window.scrollTo({ top: 0, behavior: "smooth" });
  }, [page]);

  const pageMap = {
    Home: HomePage,
    Destinations: DestinationsPage,
    Tours: ToursPage,
    About: AboutPage,
    Contact: ContactPage,
  };
  const PageComponent = pageMap[page] || HomePage;

  return (
    <div className="echo-root" style={{ minHeight: "100vh" }}>
      <style>{STYLES}</style>
      <Navbar page={page} setPage={setPage} />
      <main>
        <PageComponent setPage={setPage} />
      </main>
      <Footer setPage={setPage} />
    </div>
  );
}
