import React, { useState, useEffect, useCallback, useMemo } from "react";
import {
  Users,
  Calendar,
  MapPin,
  Star,
  CheckCircle,
  Plus,
  Search,
  Edit,
  Trash2,
  X,
  Menu,
  ChevronDown,
  Award,
  Activity,
  Check,
  AlertCircle,
  Info,
  UserCheck,
  BarChart2,
  RefreshCw,
  Eye,
  ChevronRight,
  Zap,
  Clock,
  TrendingUp,
  Shield,
  Heart,
  Filter,
} from "lucide-react";

// ═══════════════════════════════════════════════════════════
// CONSTANTS
// ═══════════════════════════════════════════════════════════
const SKILLS = [
  "First Aid",
  "Event Planning",
  "Photography",
  "Teaching",
  "Fundraising",
  "Social Media",
  "Translation",
  "Graphic Design",
  "Web Development",
  "Mentoring",
  "Marketing",
  "Leadership",
  "Data Entry",
  "Cooking",
  "Driving",
];
const AVAIL = ["Weekdays", "Weekends", "Evenings", "Mornings", "Flexible"];
const CATS = [
  "Community Service",
  "Education",
  "Environment",
  "Healthcare",
  "Arts & Culture",
  "Animal Welfare",
  "Disaster Relief",
];
const STATUSES = ["active", "inactive", "pending"];
const S_COLOR = {
  active: "bg-emerald-100 text-emerald-700 border-emerald-200",
  inactive: "bg-gray-100 text-gray-500 border-gray-200",
  pending: "bg-amber-100 text-amber-700 border-amber-200",
};
const C_COLOR = {
  "Community Service": "from-blue-500 to-cyan-500",
  Education: "from-violet-500 to-purple-600",
  Environment: "from-emerald-500 to-green-600",
  Healthcare: "from-rose-500 to-pink-600",
  "Arts & Culture": "from-pink-500 to-fuchsia-600",
  "Animal Welfare": "from-amber-500 to-orange-500",
  "Disaster Relief": "from-red-500 to-rose-600",
};
const C_ICON = {
  "Community Service": "🤝",
  Education: "📚",
  Environment: "🌿",
  Healthcare: "🏥",
  "Arts & Culture": "🎨",
  "Animal Welfare": "🐾",
  "Disaster Relief": "🚨",
};

// ═══════════════════════════════════════════════════════════
// SEED DATA
// ═══════════════════════════════════════════════════════════
const SEED_VOLS = [
  {
    id: 1,
    name: "Sarah Chen",
    email: "sarah.chen@email.com",
    phone: "+1-234-567-8901",
    skills: ["First Aid", "Event Planning", "Photography"],
    location: "San Francisco, CA",
    availability: ["Weekends", "Evenings"],
    rating: 4.8,
    tasks: 12,
    status: "active",
    joined: "2024-01-15",
    notes: "Excellent team player, CPR certified",
  },
  {
    id: 2,
    name: "Marcus Johnson",
    email: "marcus.j@email.com",
    phone: "+1-234-567-8902",
    skills: ["Teaching", "Fundraising", "Social Media"],
    location: "Oakland, CA",
    availability: ["Weekdays"],
    rating: 4.9,
    tasks: 18,
    status: "active",
    joined: "2024-02-01",
    notes: "Strong leadership, organized fundraisers",
  },
  {
    id: 3,
    name: "Elena Rodriguez",
    email: "elena.r@email.com",
    phone: "+1-234-567-8903",
    skills: ["Translation", "Event Planning", "Graphic Design"],
    location: "Berkeley, CA",
    availability: ["Weekends"],
    rating: 4.7,
    tasks: 9,
    status: "active",
    joined: "2024-01-20",
    notes: "Bilingual English/Spanish",
  },
  {
    id: 4,
    name: "David Kim",
    email: "david.kim@email.com",
    phone: "+1-234-567-8904",
    skills: ["Web Development", "Teaching", "Mentoring"],
    location: "San Jose, CA",
    availability: ["Evenings", "Weekends"],
    rating: 5.0,
    tasks: 25,
    status: "active",
    joined: "2023-11-10",
    notes: "Top performer, 100% event attendance",
  },
  {
    id: 5,
    name: "Aisha Patel",
    email: "aisha.p@email.com",
    phone: "+1-234-567-8905",
    skills: ["First Aid", "Cooking", "Leadership"],
    location: "San Francisco, CA",
    availability: ["Flexible"],
    rating: 4.6,
    tasks: 7,
    status: "inactive",
    joined: "2024-03-01",
    notes: "Medical background, available on request",
  },
];
const SEED_EVTS = [
  {
    id: 1,
    title: "Community Food Drive",
    desc: "Help organize and distribute food to families in need.",
    date: "2026-03-15",
    time: "09:00",
    dur: "4 hrs",
    loc: "Community Center, SF",
    skills: ["Event Planning", "First Aid"],
    vols: [1, 3],
    max: 15,
    cat: "Community Service",
  },
  {
    id: 2,
    title: "Youth Coding Workshop",
    desc: "Teach basic programming to high school students aged 14–18.",
    date: "2026-03-20",
    time: "10:00",
    dur: "3 hrs",
    loc: "Tech Hub, Oakland",
    skills: ["Teaching", "Web Development"],
    vols: [2, 4],
    max: 8,
    cat: "Education",
  },
  {
    id: 3,
    title: "Beach Cleanup Drive",
    desc: "Environmental cleanup to restore our beautiful coastline.",
    date: "2026-03-25",
    time: "08:00",
    dur: "5 hrs",
    loc: "Ocean Beach, SF",
    skills: ["Event Planning"],
    vols: [],
    max: 30,
    cat: "Environment",
  },
  {
    id: 4,
    title: "Senior Tech Support Day",
    desc: "Help seniors learn smartphones, tablets, and digital literacy.",
    date: "2026-04-05",
    time: "11:00",
    dur: "3 hrs",
    loc: "Senior Center, Berkeley",
    skills: ["Teaching", "Mentoring"],
    vols: [4],
    max: 12,
    cat: "Community Service",
  },
];

// ═══════════════════════════════════════════════════════════
// GLOBAL STATE — single source of truth, no prop drilling issues
// ═══════════════════════════════════════════════════════════
const AppCtx = React.createContext(null);
const useApp = () => React.useContext(AppCtx);

// ═══════════════════════════════════════════════════════════
// TOAST — instant feedback on every action
// ═══════════════════════════════════════════════════════════
const ToastCtx = React.createContext(null);
const useToast = () => React.useContext(ToastCtx);

function ToastProvider({ children }) {
  const [toasts, setToasts] = useState([]);
  const push = useCallback((msg, type = "success") => {
    const id = Date.now() + Math.random();
    setToasts((prev) => [...prev, { id, msg, type }]);
    setTimeout(
      () => setToasts((prev) => prev.filter((t) => t.id !== id)),
      3800,
    );
  }, []);
  const dismiss = (id) => setToasts((prev) => prev.filter((t) => t.id !== id));
  const TMAP = {
    success: {
      bg: "bg-emerald-500",
      icon: <Check className="w-4 h-4 flex-shrink-0" />,
    },
    error: {
      bg: "bg-red-500",
      icon: <AlertCircle className="w-4 h-4 flex-shrink-0" />,
    },
    info: {
      bg: "bg-blue-500",
      icon: <Info className="w-4 h-4 flex-shrink-0" />,
    },
    warning: {
      bg: "bg-amber-500",
      icon: <AlertCircle className="w-4 h-4 flex-shrink-0" />,
    },
  };
  return (
    <ToastCtx.Provider value={push}>
      {children}
      <div
        className="fixed top-4 right-4 z-[999] flex flex-col gap-2 pointer-events-none"
        style={{ maxWidth: 340 }}
      >
        {toasts.map((t) => {
          const { bg, icon } = TMAP[t.type] || TMAP.info;
          return (
            <div
              key={t.id}
              className={`pointer-events-auto flex items-center gap-3 px-4 py-3 rounded-2xl shadow-2xl text-white text-sm font-semibold ${bg}`}
              style={{
                animation: "toastSlide .32s cubic-bezier(.22,1,.36,1) both",
              }}
            >
              {icon}
              <span className="flex-1 leading-snug">{t.msg}</span>
              <button
                onClick={() => dismiss(t.id)}
                className="opacity-60 hover:opacity-100 transition-opacity ml-1"
              >
                <X className="w-3.5 h-3.5" />
              </button>
            </div>
          );
        })}
      </div>
    </ToastCtx.Provider>
  );
}

// ═══════════════════════════════════════════════════════════
// TINY SHARED COMPONENTS
// ═══════════════════════════════════════════════════════════
const cx = (...c) => c.filter(Boolean).join(" ");
const INP =
  "w-full px-4 py-2.5 border border-gray-200 rounded-xl text-sm focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-transparent bg-gray-50 focus:bg-white transition-all placeholder-gray-400";

const Avi = ({ name, size = "md", color }) => {
  const sz =
    {
      sm: "w-8 h-8 text-xs",
      md: "w-10 h-10 text-sm",
      lg: "w-12 h-12 text-base",
    }[size] || "w-10 h-10 text-sm";
  const initials = name
    .trim()
    .split(/\s+/)
    .map((n) => n[0])
    .join("")
    .slice(0, 2)
    .toUpperCase();
  return (
    <div
      className={cx(
        sz,
        "rounded-xl flex items-center justify-center text-white font-bold flex-shrink-0 shadow-sm",
        color || "bg-gradient-to-br from-orange-400 to-rose-500",
      )}
    >
      {initials}
    </div>
  );
};

const Pill = ({ label, hot }) => (
  <span
    className={cx(
      "inline-block px-2.5 py-0.5 rounded-lg text-xs font-semibold",
      hot
        ? "bg-gradient-to-r from-orange-500 to-rose-500 text-white"
        : "bg-orange-50 text-orange-700 border border-orange-100",
    )}
  >
    {label}
  </span>
);

const Stars = ({ val }) => (
  <span className="inline-flex items-center gap-1 bg-amber-50 border border-amber-200 px-2 py-0.5 rounded-lg text-xs font-bold text-amber-700">
    <Star className="w-3 h-3 fill-amber-400 text-amber-400" />
    {val > 0 ? val.toFixed(1) : "New"}
  </span>
);

const Bar = ({ val, max, green }) => {
  const pct = max > 0 ? Math.min((val / max) * 100, 100) : 0;
  return (
    <div className="w-full h-2 bg-gray-100 rounded-full overflow-hidden">
      <div
        className={cx(
          "h-full rounded-full transition-all duration-700",
          green
            ? "bg-gradient-to-r from-emerald-400 to-teal-400"
            : "bg-gradient-to-r from-orange-400 to-rose-400",
        )}
        style={{ width: `${pct}%` }}
      />
    </div>
  );
};

const Lbl = ({ text, req, err, children }) => (
  <div>
    <label className="block text-sm font-semibold text-gray-700 mb-1.5">
      {text}
      {req && <span className="text-red-400 ml-0.5">*</span>}
    </label>
    {children}
    {err && (
      <p className="mt-1 text-xs text-red-500 font-medium flex items-center gap-1">
        <AlertCircle className="w-3 h-3" />
        {err}
      </p>
    )}
  </div>
);

const Chk = ({ label, on, toggle, col = "orange" }) => {
  const border = {
    orange: on
      ? "border-orange-400 bg-orange-50 text-orange-800"
      : "border-gray-200 hover:border-orange-300 hover:bg-orange-50/40",
    blue: on
      ? "border-blue-400 bg-blue-50 text-blue-800"
      : "border-gray-200 hover:border-blue-300",
    rose: on
      ? "border-rose-400 bg-rose-50 text-rose-800"
      : "border-gray-200 hover:border-rose-300",
  };
  const dot = {
    orange: "bg-orange-500",
    blue: "bg-blue-500",
    rose: "bg-rose-500",
  };
  return (
    <label
      className={cx(
        "flex items-center gap-2 p-2.5 border rounded-xl cursor-pointer transition-all text-sm select-none",
        border[col],
      )}
    >
      <div
        className={cx(
          "w-4 h-4 rounded flex items-center justify-center flex-shrink-0 transition-all",
          on ? dot[col] : "border border-gray-300 bg-white",
        )}
      >
        {on && <Check className="w-2.5 h-2.5 text-white" />}
      </div>
      <span className="font-medium leading-tight">{label}</span>
      <input
        type="checkbox"
        className="sr-only"
        checked={on}
        onChange={toggle}
      />
    </label>
  );
};

// ═══════════════════════════════════════════════════════════
// MODAL SHELL — no page refresh, pure overlay
// ═══════════════════════════════════════════════════════════
function ModalShell({ open, close, title, sub, children, wide = "max-w-2xl" }) {
  useEffect(() => {
    document.body.style.overflow = open ? "hidden" : "";
    return () => {
      document.body.style.overflow = "";
    };
  }, [open]);
  if (!open) return null;
  return (
    <div
      className="fixed inset-0 z-50 flex items-end sm:items-center justify-center sm:p-4"
      style={{ animation: "fadein .18s ease both" }}
    >
      <div
        className="absolute inset-0 bg-black/50 backdrop-blur-[3px]"
        onClick={close}
      />
      <div
        className={cx(
          "relative bg-white w-full rounded-t-3xl sm:rounded-3xl shadow-2xl max-h-[94vh] flex flex-col",
          wide,
        )}
        style={{ animation: "slideup .32s cubic-bezier(.22,1,.36,1) both" }}
      >
        <div className="flex items-start justify-between px-6 pt-6 pb-4 border-b border-gray-100 flex-shrink-0">
          <div>
            <h3 className="text-xl font-bold text-gray-900 leading-tight">
              {title}
            </h3>
            {sub && <p className="text-sm text-gray-500 mt-0.5">{sub}</p>}
          </div>
          <button
            onClick={close}
            className="p-2 hover:bg-gray-100 rounded-xl transition-colors -mt-1 flex-shrink-0"
          >
            <X className="w-5 h-5 text-gray-500" />
          </button>
        </div>
        <div className="overflow-y-auto flex-1 px-6 py-5 space-y-4">
          {children}
        </div>
      </div>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════
// VOLUNTEER FORM MODAL
// ═══════════════════════════════════════════════════════════
const BLANK_VOL = {
  name: "",
  email: "",
  phone: "",
  location: "",
  skills: [],
  availability: [],
  status: "active",
  notes: "",
};

function VolForm({ open, close, onSave, editing }) {
  const [f, setF] = useState(BLANK_VOL);
  const [err, setErr] = useState({});

  // Reset form whenever modal opens
  useEffect(() => {
    if (open) {
      setF(editing ? { ...editing } : { ...BLANK_VOL });
      setErr({});
    }
  }, [open, editing]);

  const set = (k, v) => {
    setF((p) => ({ ...p, [k]: v }));
    setErr((p) => ({ ...p, [k]: "" }));
  };
  const tog = (k, v) =>
    set(k, f[k].includes(v) ? f[k].filter((x) => x !== v) : [...f[k], v]);

  const validate = () => {
    const e = {};
    if (!f.name.trim()) e.name = "Full name is required";
    if (!f.email.trim() || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(f.email))
      e.email = "Valid email required";
    if (!f.phone.trim()) e.phone = "Phone number required";
    if (!f.location.trim()) e.location = "Location required";
    if (!f.skills.length) e.skills = "Select at least one skill";
    if (!f.availability.length)
      e.availability = "Select at least one time slot";
    return e;
  };

  const submit = () => {
    const e = validate();
    if (Object.keys(e).length) {
      setErr(e);
      return;
    }
    onSave(f);
  };

  return (
    <ModalShell
      open={open}
      close={close}
      title={editing ? "Edit Volunteer" : "Add New Volunteer"}
      sub={
        editing
          ? `Editing profile for ${editing.name}`
          : "Register a new volunteer to the system"
      }
    >
      {/* Name */}
      <Lbl text="Full Name" req err={err.name}>
        <input
          className={INP}
          value={f.name}
          onChange={(e) => set("name", e.target.value)}
          placeholder="e.g. Jane Smith"
          autoFocus
        />
      </Lbl>

      {/* Email + Phone */}
      <div className="grid sm:grid-cols-2 gap-4">
        <Lbl text="Email" req err={err.email}>
          <input
            className={INP}
            type="email"
            value={f.email}
            onChange={(e) => set("email", e.target.value)}
            placeholder="jane@email.com"
          />
        </Lbl>
        <Lbl text="Phone" req err={err.phone}>
          <input
            className={INP}
            value={f.phone}
            onChange={(e) => set("phone", e.target.value)}
            placeholder="+1 234-567-8900"
          />
        </Lbl>
      </div>

      {/* Location + Status */}
      <div className="grid sm:grid-cols-2 gap-4">
        <Lbl text="Location" req err={err.location}>
          <input
            className={INP}
            value={f.location}
            onChange={(e) => set("location", e.target.value)}
            placeholder="City, State"
          />
        </Lbl>
        <Lbl text="Status">
          <select
            className={INP}
            value={f.status}
            onChange={(e) => set("status", e.target.value)}
          >
            <option value="active">✅ Active</option>
            <option value="inactive">⏸ Inactive</option>
            <option value="pending">⏳ Pending</option>
          </select>
        </Lbl>
      </div>

      {/* Skills */}
      <Lbl text="Skills" req err={err.skills}>
        <div className="grid grid-cols-2 sm:grid-cols-3 gap-2 mt-1">
          {SKILLS.map((s) => (
            <Chk
              key={s}
              label={s}
              on={f.skills.includes(s)}
              toggle={() => tog("skills", s)}
              col="orange"
            />
          ))}
        </div>
      </Lbl>

      {/* Availability */}
      <Lbl text="Availability" req err={err.availability}>
        <div className="grid grid-cols-2 sm:grid-cols-3 gap-2 mt-1">
          {AVAIL.map((a) => (
            <Chk
              key={a}
              label={a}
              on={f.availability.includes(a)}
              toggle={() => tog("availability", a)}
              col="blue"
            />
          ))}
        </div>
      </Lbl>

      {/* Notes */}
      <Lbl text="Internal Notes (optional)">
        <textarea
          className={cx(INP, "resize-none")}
          rows={2}
          value={f.notes}
          onChange={(e) => set("notes", e.target.value)}
          placeholder="Any relevant notes about this volunteer..."
        />
      </Lbl>

      {/* Actions */}
      <div className="flex gap-3 pt-2 border-t border-gray-100">
        <button
          onClick={close}
          className="flex-1 py-2.5 border border-gray-200 rounded-xl text-sm font-semibold text-gray-600 hover:bg-gray-50 transition-colors"
        >
          Cancel
        </button>
        <button
          onClick={submit}
          className="flex-1 py-2.5 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl text-sm font-bold shadow-md hover:from-orange-600 hover:to-rose-600 transition-all active:scale-[.98]"
        >
          {editing ? "Save Changes" : "+ Add Volunteer"}
        </button>
      </div>
    </ModalShell>
  );
}

// ═══════════════════════════════════════════════════════════
// EVENT FORM MODAL
// ═══════════════════════════════════════════════════════════
const BLANK_EVT = {
  title: "",
  desc: "",
  date: "",
  time: "09:00",
  dur: "3 hours",
  loc: "",
  skills: [],
  max: 10,
  cat: "Community Service",
};

function EvtForm({ open, close, onSave, editing }) {
  const [f, setF] = useState(BLANK_EVT);
  const [err, setErr] = useState({});

  useEffect(() => {
    if (open) {
      setF(editing ? { ...editing } : { ...BLANK_EVT });
      setErr({});
    }
  }, [open, editing]);

  const set = (k, v) => {
    setF((p) => ({ ...p, [k]: v }));
    setErr((p) => ({ ...p, [k]: "" }));
  };
  const togSkill = (s) =>
    set(
      "skills",
      f.skills.includes(s) ? f.skills.filter((x) => x !== s) : [...f.skills, s],
    );

  const validate = () => {
    const e = {};
    if (!f.title.trim()) e.title = "Title required";
    if (!f.desc.trim()) e.desc = "Description required";
    if (!f.date) e.date = "Date required";
    if (!f.loc.trim()) e.loc = "Location required";
    if (!f.skills.length) e.skills = "Select at least one skill";
    if (!f.max || f.max < 1) e.max = "Must be at least 1";
    return e;
  };

  const submit = () => {
    const e = validate();
    if (Object.keys(e).length) {
      setErr(e);
      return;
    }
    onSave(f);
  };

  return (
    <ModalShell
      open={open}
      close={close}
      title={editing ? "Edit Event" : "Create New Event"}
      sub={
        editing
          ? `Editing: ${editing.title}`
          : "Set up a new volunteer opportunity"
      }
    >
      <Lbl text="Event Title" req err={err.title}>
        <input
          className={INP}
          value={f.title}
          onChange={(e) => set("title", e.target.value)}
          placeholder="Community Food Drive"
          autoFocus
        />
      </Lbl>

      <Lbl text="Description" req err={err.desc}>
        <textarea
          className={cx(INP, "resize-none")}
          rows={2}
          value={f.desc}
          onChange={(e) => set("desc", e.target.value)}
          placeholder="What will volunteers do at this event?"
        />
      </Lbl>

      <div className="grid sm:grid-cols-3 gap-4">
        <Lbl text="Date" req err={err.date}>
          <input
            className={INP}
            type="date"
            value={f.date}
            onChange={(e) => set("date", e.target.value)}
          />
        </Lbl>
        <Lbl text="Time">
          <input
            className={INP}
            type="time"
            value={f.time}
            onChange={(e) => set("time", e.target.value)}
          />
        </Lbl>
        <Lbl text="Duration">
          <input
            className={INP}
            value={f.dur}
            onChange={(e) => set("dur", e.target.value)}
            placeholder="3 hours"
          />
        </Lbl>
      </div>

      <div className="grid sm:grid-cols-2 gap-4">
        <Lbl text="Location" req err={err.loc}>
          <input
            className={INP}
            value={f.loc}
            onChange={(e) => set("loc", e.target.value)}
            placeholder="Venue, City"
          />
        </Lbl>
        <Lbl text="Max Volunteers" req err={err.max}>
          <input
            className={INP}
            type="number"
            min="1"
            value={f.max}
            onChange={(e) => set("max", parseInt(e.target.value) || 1)}
          />
        </Lbl>
      </div>

      <Lbl text="Category">
        <select
          className={INP}
          value={f.cat}
          onChange={(e) => set("cat", e.target.value)}
        >
          {CATS.map((c) => (
            <option key={c} value={c}>
              {C_ICON[c]} {c}
            </option>
          ))}
        </select>
      </Lbl>

      <Lbl text="Required Skills" req err={err.skills}>
        <div className="grid grid-cols-2 sm:grid-cols-3 gap-2 mt-1">
          {SKILLS.map((s) => (
            <Chk
              key={s}
              label={s}
              on={f.skills.includes(s)}
              toggle={() => togSkill(s)}
              col="rose"
            />
          ))}
        </div>
      </Lbl>

      <div className="flex gap-3 pt-2 border-t border-gray-100">
        <button
          onClick={close}
          className="flex-1 py-2.5 border border-gray-200 rounded-xl text-sm font-semibold text-gray-600 hover:bg-gray-50 transition-colors"
        >
          Cancel
        </button>
        <button
          onClick={submit}
          className="flex-1 py-2.5 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl text-sm font-bold shadow-md hover:from-orange-600 hover:to-rose-600 transition-all active:scale-[.98]"
        >
          {editing ? "Save Changes" : "+ Create Event"}
        </button>
      </div>
    </ModalShell>
  );
}

// ═══════════════════════════════════════════════════════════
// SMART ASSIGN MODAL
// ═══════════════════════════════════════════════════════════
function AssignModal({ open, close, eventId, evts, vols, onAssign }) {
  const [q, setQ] = useState("");
  useEffect(() => {
    if (open) setQ("");
  }, [open]);

  // Derive event fresh from evts — always before any return
  const event = evts.find((e) => e.id === eventId) || null;

  // ALL hooks must run unconditionally before any early return
  const ranked = useMemo(() => {
    if (!event) return [];
    return vols
      .filter((v) => !event.vols.includes(v.id) && v.status === "active")
      .map((v) => ({
        ...v,
        score: event.skills.filter((s) => v.skills.includes(s)).length,
      }))
      .sort((a, b) => b.score - a.score || b.rating - a.rating);
  }, [vols, event]); // event reference changes when evts changes — correct

  // Early return AFTER all hooks
  if (!open || !event) return null;

  const full = event.vols.length >= event.max;
  const shown = ranked.filter(
    (v) =>
      !q ||
      v.name.toLowerCase().includes(q.toLowerCase()) ||
      v.email.toLowerCase().includes(q.toLowerCase()) ||
      v.skills.some((s) => s.toLowerCase().includes(q.toLowerCase())),
  );

  return (
    <ModalShell
      open={open}
      close={close}
      title="Smart Assign Volunteers"
      sub={`${event.title} · ${event.vols.length}/${event.max} slots filled`}
      wide="max-w-3xl"
    >
      {/* Event summary */}
      <div className="p-4 bg-orange-50 border border-orange-100 rounded-2xl space-y-2">
        <div className="flex flex-wrap gap-1.5">
          {event.skills.map((s) => (
            <Pill key={s} label={s} hot />
          ))}
        </div>
        <Bar val={event.vols.length} max={event.max} />
        <p className="text-xs text-gray-500">
          {full
            ? "✅ All slots filled — event fully staffed"
            : `${event.max - event.vols.length} volunteer slot(s) still available`}
        </p>
      </div>

      {!full && (
        <>
          {/* Search */}
          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400" />
            <input
              className={cx(INP, "pl-9")}
              value={q}
              onChange={(e) => setQ(e.target.value)}
              placeholder="Search by name, email, or skill…"
              autoFocus
            />
            {q && (
              <button
                onClick={() => setQ("")}
                className="absolute right-3 top-1/2 -translate-y-1/2 p-0.5 hover:bg-gray-100 rounded-full"
              >
                <X className="w-3.5 h-3.5 text-gray-400" />
              </button>
            )}
          </div>

          {/* Count */}
          <p className="text-xs text-gray-500 font-medium -mt-2">
            {shown.length} volunteer{shown.length !== 1 ? "s" : ""} available
            {shown.some((v) => v.score > 0) && (
              <span className="ml-1 text-orange-600">
                · ranked by skill match
              </span>
            )}
          </p>

          {/* List */}
          {shown.length > 0 ? (
            <div className="space-y-2 max-h-72 overflow-y-auto pr-0.5">
              {shown.map((v) => (
                <div
                  key={v.id}
                  className="flex flex-col sm:flex-row sm:items-center gap-3 p-3.5 bg-white border border-gray-100 rounded-xl hover:border-orange-200 hover:shadow-sm transition-all"
                >
                  <div className="flex items-center gap-3 flex-1 min-w-0">
                    <Avi name={v.name} size="sm" />
                    <div className="min-w-0 flex-1">
                      <div className="flex flex-wrap items-center gap-2 mb-1">
                        <span className="font-bold text-sm text-gray-800">
                          {v.name}
                        </span>
                        <Stars val={v.rating} />
                        {v.score > 0 && (
                          <span className="px-2 py-0.5 bg-emerald-100 text-emerald-700 rounded-full text-xs font-bold">
                            {v.score * 2}pt match
                          </span>
                        )}
                      </div>
                      <div className="flex flex-wrap gap-1">
                        {v.skills.map((s) => (
                          <Pill
                            key={s}
                            label={s}
                            hot={event.skills.includes(s)}
                          />
                        ))}
                      </div>
                    </div>
                  </div>
                  <button
                    onClick={() => onAssign(event.id, v.id)}
                    className="flex-shrink-0 px-5 py-2 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl text-xs font-bold hover:from-orange-600 hover:to-rose-600 shadow-sm transition-all active:scale-95 whitespace-nowrap"
                  >
                    + Assign
                  </button>
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-10 text-gray-400">
              <Users className="w-10 h-10 mx-auto mb-2 opacity-30" />
              <p className="text-sm font-medium">
                {q
                  ? "No volunteers match your search"
                  : "No eligible active volunteers to assign"}
              </p>
            </div>
          )}
        </>
      )}

      {full && (
        <div className="text-center py-10">
          <UserCheck className="w-12 h-12 mx-auto mb-3 text-emerald-400" />
          <p className="font-bold text-gray-800 text-lg">
            Event Fully Staffed!
          </p>
          <p className="text-sm text-gray-500 mt-1">
            All {event.max} volunteer slots are filled.
          </p>
        </div>
      )}
    </ModalShell>
  );
}

// ═══════════════════════════════════════════════════════════
// PROFILE VIEW MODAL
// ═══════════════════════════════════════════════════════════
function ProfileModal({ open, close, vol, evts }) {
  if (!vol) return null;
  const myEvts = evts.filter((e) => e.vols.includes(vol.id));
  return (
    <ModalShell
      open={open}
      close={close}
      title={vol.name}
      sub={vol.email}
      wide="max-w-lg"
    >
      {/* Header card */}
      <div className="flex items-center gap-4 p-4 bg-gradient-to-r from-orange-50 to-rose-50 border border-orange-100 rounded-2xl">
        <Avi name={vol.name} size="lg" />
        <div className="min-w-0 flex-1">
          <div className="flex flex-wrap items-center gap-2 mb-1.5">
            <Stars val={vol.rating} />
            <span
              className={cx(
                "px-2.5 py-0.5 rounded-full border text-xs font-bold",
                S_COLOR[vol.status],
              )}
            >
              {vol.status}
            </span>
          </div>
          <p className="text-sm text-gray-600 flex items-center gap-1.5">
            <MapPin className="w-3.5 h-3.5 text-orange-400 flex-shrink-0" />
            {vol.location}
          </p>
          <p className="text-xs text-gray-400 mt-0.5">
            Joined{" "}
            {new Date(vol.joined).toLocaleDateString("en-US", {
              year: "numeric",
              month: "long",
              day: "numeric",
            })}
          </p>
        </div>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-3 gap-3">
        {[
          {
            label: "Tasks Done",
            val: vol.tasks,
            color: "text-orange-600",
            bg: "bg-orange-50",
          },
          {
            label: "Events",
            val: myEvts.length,
            color: "text-blue-600",
            bg: "bg-blue-50",
          },
          {
            label: "Rating",
            val: vol.rating > 0 ? vol.rating.toFixed(1) : "—",
            color: "text-amber-600",
            bg: "bg-amber-50",
          },
        ].map((s) => (
          <div key={s.label} className={cx(s.bg, "rounded-xl p-3 text-center")}>
            <p className={cx("text-2xl font-black", s.color)}>{s.val}</p>
            <p className="text-xs font-semibold text-gray-500 mt-0.5">
              {s.label}
            </p>
          </div>
        ))}
      </div>

      {/* Contact */}
      <div>
        <p className="text-xs font-bold text-gray-400 uppercase tracking-widest mb-2">
          Contact
        </p>
        <div className="bg-gray-50 rounded-xl p-3 space-y-1.5 text-sm">
          <p>
            <span className="text-gray-500 w-16 inline-block">Phone:</span>
            <span className="font-semibold">{vol.phone}</span>
          </p>
          <p>
            <span className="text-gray-500 w-16 inline-block">Email:</span>
            <span className="font-semibold">{vol.email}</span>
          </p>
        </div>
      </div>

      {/* Skills */}
      <div>
        <p className="text-xs font-bold text-gray-400 uppercase tracking-widest mb-2">
          Skills ({vol.skills.length})
        </p>
        <div className="flex flex-wrap gap-1.5">
          {vol.skills.map((s) => (
            <Pill key={s} label={s} />
          ))}
        </div>
      </div>

      {/* Availability */}
      <div>
        <p className="text-xs font-bold text-gray-400 uppercase tracking-widest mb-2">
          Availability
        </p>
        <div className="flex flex-wrap gap-1.5">
          {vol.availability.map((a) => (
            <span
              key={a}
              className="px-2.5 py-1 bg-blue-50 text-blue-700 border border-blue-100 rounded-lg text-xs font-semibold"
            >
              {a}
            </span>
          ))}
        </div>
      </div>

      {/* Assigned events */}
      {myEvts.length > 0 && (
        <div>
          <p className="text-xs font-bold text-gray-400 uppercase tracking-widest mb-2">
            Assigned Events ({myEvts.length})
          </p>
          <div className="space-y-2">
            {myEvts.map((e) => (
              <div
                key={e.id}
                className="flex items-center gap-3 p-3 bg-gray-50 rounded-xl"
              >
                <span className="text-lg flex-shrink-0">
                  {C_ICON[e.cat] || "📋"}
                </span>
                <div>
                  <p className="text-sm font-semibold text-gray-800">
                    {e.title}
                  </p>
                  <p className="text-xs text-gray-500">
                    {new Date(e.date).toLocaleDateString()} ·{" "}
                    {e.loc.split(",")[0]}
                  </p>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Notes */}
      {vol.notes && (
        <div>
          <p className="text-xs font-bold text-gray-400 uppercase tracking-widest mb-2">
            Notes
          </p>
          <p className="text-sm text-gray-700 bg-amber-50 border border-amber-100 rounded-xl p-3 leading-relaxed">
            {vol.notes}
          </p>
        </div>
      )}
    </ModalShell>
  );
}

// ═══════════════════════════════════════════════════════════
// CONFIRM DELETE MODAL
// ═══════════════════════════════════════════════════════════
function DeleteModal({ open, close, onConfirm, type, name }) {
  return (
    <ModalShell
      open={open}
      close={close}
      title="Confirm Delete"
      wide="max-w-sm"
    >
      <div className="text-center py-2">
        <div className="w-14 h-14 bg-red-100 rounded-2xl flex items-center justify-center mx-auto mb-4">
          <Trash2 className="w-7 h-7 text-red-500" />
        </div>
        <p className="text-gray-700 text-sm leading-relaxed">
          Delete <strong>{name}</strong>?{" "}
          {type === "vol"
            ? "This volunteer will be removed from all assigned events."
            : "All assignment data for this event will be permanently deleted."}{" "}
          This cannot be undone.
        </p>
      </div>
      <div className="flex gap-3">
        <button
          onClick={close}
          className="flex-1 py-2.5 border border-gray-200 rounded-xl text-sm font-semibold text-gray-600 hover:bg-gray-50 transition-colors"
        >
          Cancel
        </button>
        <button
          onClick={onConfirm}
          className="flex-1 py-2.5 bg-red-500 text-white rounded-xl text-sm font-bold hover:bg-red-600 transition-colors active:scale-[.98]"
        >
          Delete
        </button>
      </div>
    </ModalShell>
  );
}

// ═══════════════════════════════════════════════════════════
// VOLUNTEER CARD
// ═══════════════════════════════════════════════════════════
function VolCard({ vol, evtCount, onView, onEdit, onDelete }) {
  return (
    <div className="bg-white rounded-2xl border border-gray-100 shadow-sm hover:shadow-lg hover:-translate-y-1 transition-all duration-300 overflow-hidden group">
      <div className="h-1 bg-gradient-to-r from-orange-400 to-rose-400" />
      <div className="p-5">
        {/* Top */}
        <div className="flex items-start justify-between mb-3">
          <div className="flex items-center gap-3 flex-1 min-w-0">
            <Avi name={vol.name} />
            <div className="min-w-0">
              <h3 className="font-bold text-gray-900 text-sm sm:text-base truncate">
                {vol.name}
              </h3>
              <p className="text-xs text-gray-500 truncate">{vol.email}</p>
            </div>
          </div>
          <Stars val={vol.rating} />
        </div>

        {/* Location */}
        <p className="flex items-center gap-1.5 text-xs text-gray-500 mb-3">
          <MapPin className="w-3.5 h-3.5 text-orange-400 flex-shrink-0" />
          <span className="truncate">{vol.location}</span>
        </p>

        {/* Skills */}
        <div className="flex flex-wrap gap-1.5 mb-3">
          {vol.skills.slice(0, 4).map((s) => (
            <Pill key={s} label={s} />
          ))}
          {vol.skills.length > 4 && (
            <span className="text-xs text-gray-400 self-center">
              +{vol.skills.length - 4} more
            </span>
          )}
        </div>

        {/* Availability */}
        <div className="flex flex-wrap gap-1.5 mb-4">
          {vol.availability.map((a) => (
            <span
              key={a}
              className="px-2 py-0.5 bg-blue-50 text-blue-700 rounded-lg text-xs font-semibold"
            >
              {a}
            </span>
          ))}
        </div>

        {/* Footer */}
        <div className="flex items-center justify-between pt-3 border-t border-gray-50">
          <div className="flex items-center gap-2 flex-wrap">
            <span
              className={cx(
                "px-2 py-0.5 rounded-full border text-xs font-bold",
                S_COLOR[vol.status],
              )}
            >
              {vol.status}
            </span>
            <span className="text-xs text-gray-400 flex items-center gap-1">
              <CheckCircle className="w-3 h-3 text-emerald-400" />
              {vol.tasks} tasks
            </span>
            {evtCount > 0 && (
              <span className="text-xs text-gray-400 flex items-center gap-1">
                <Calendar className="w-3 h-3 text-blue-400" />
                {evtCount} event{evtCount > 1 ? "s" : ""}
              </span>
            )}
          </div>
          <div className="flex gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
            <button
              onClick={() => onView(vol)}
              className="p-1.5 hover:bg-blue-50 rounded-lg transition-colors"
              title="View Profile"
            >
              <Eye className="w-3.5 h-3.5 text-blue-500" />
            </button>
            <button
              onClick={() => onEdit(vol)}
              className="p-1.5 hover:bg-orange-50 rounded-lg transition-colors"
              title="Edit"
            >
              <Edit className="w-3.5 h-3.5 text-orange-500" />
            </button>
            <button
              onClick={() => onDelete(vol)}
              className="p-1.5 hover:bg-red-50 rounded-lg transition-colors"
              title="Delete"
            >
              <Trash2 className="w-3.5 h-3.5 text-red-400" />
            </button>
          </div>
        </div>
      </div>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════
// EVENT CARD
// ═══════════════════════════════════════════════════════════
function EvtCard({ evt, vols, onEdit, onDelete, onAssign, onUnassign }) {
  const assigned = evt.vols
    .map((id) => vols.find((v) => v.id === id))
    .filter(Boolean);
  const full = evt.vols.length >= evt.max;
  const grad = C_COLOR[evt.cat] || "from-gray-400 to-gray-500";
  return (
    <div className="bg-white rounded-2xl border border-gray-100 shadow-sm hover:shadow-lg hover:-translate-y-0.5 transition-all duration-300 overflow-hidden">
      <div className={cx("h-1.5 bg-gradient-to-r", grad)} />
      <div className="p-5">
        {/* Title + category */}
        <div className="flex items-start gap-3 mb-2">
          <span className="text-xl flex-shrink-0 mt-0.5">
            {C_ICON[evt.cat] || "📋"}
          </span>
          <div className="flex-1 min-w-0">
            <h3 className="font-bold text-gray-900 text-sm sm:text-base leading-snug">
              {evt.title}
            </h3>
            <span
              className={cx(
                "text-xs font-semibold px-2 py-0.5 rounded-lg mt-1 inline-block",
                full
                  ? "bg-emerald-100 text-emerald-700"
                  : "bg-orange-100 text-orange-700",
              )}
            >
              {evt.cat}
            </span>
          </div>
        </div>

        <p className="text-xs text-gray-500 mb-3 line-clamp-2">{evt.desc}</p>

        {/* Meta */}
        <div className="grid grid-cols-2 gap-x-4 gap-y-1 text-xs text-gray-600 mb-3">
          <span className="flex items-center gap-1.5">
            <Calendar className="w-3.5 h-3.5 text-orange-400 flex-shrink-0" />
            {new Date(evt.date).toLocaleDateString("en-US", {
              month: "short",
              day: "numeric",
              year: "numeric",
            })}
          </span>
          <span className="flex items-center gap-1.5">
            <Clock className="w-3.5 h-3.5 text-orange-400 flex-shrink-0" />
            {evt.time} · {evt.dur}
          </span>
          <span className="flex items-center gap-1.5 col-span-2">
            <MapPin className="w-3.5 h-3.5 text-orange-400 flex-shrink-0" />
            <span className="truncate">{evt.loc}</span>
          </span>
        </div>

        {/* Required skills */}
        <div className="flex flex-wrap gap-1.5 mb-3">
          {evt.skills.map((s) => (
            <Pill key={s} label={s} hot />
          ))}
        </div>

        {/* Capacity */}
        <div className="mb-3">
          <div className="flex justify-between text-xs mb-1.5">
            <span className="text-gray-500 font-medium flex items-center gap-1">
              <Users className="w-3 h-3 text-orange-400" />
              {evt.vols.length}/{evt.max} volunteers
            </span>
            <span
              className={cx(
                "font-bold",
                full ? "text-emerald-600" : "text-orange-600",
              )}
            >
              {full
                ? "✅ Full"
                : `${evt.max - evt.vols.length} spot${evt.max - evt.vols.length > 1 ? "s" : ""} left`}
            </span>
          </div>
          <Bar val={evt.vols.length} max={evt.max} green={full} />
        </div>

        {/* Assigned volunteers chips */}
        {assigned.length > 0 && (
          <div className="flex flex-wrap gap-1.5 mb-3">
            {assigned.map((v) => (
              <span
                key={v.id}
                className="flex items-center gap-1 pl-2 pr-1 py-0.5 bg-orange-50 border border-orange-100 rounded-full text-xs font-bold text-orange-800"
              >
                {v.name.split(" ")[0]}
                <button
                  onClick={() => onUnassign(evt.id, v.id)}
                  className="hover:bg-orange-200 rounded-full p-0.5 transition-colors"
                  title={`Remove ${v.name}`}
                >
                  <X className="w-2.5 h-2.5" />
                </button>
              </span>
            ))}
          </div>
        )}

        {/* Actions */}
        <div className="flex gap-2 pt-3 border-t border-gray-50">
          {!full && (
            <button
              onClick={() => onAssign(evt.id)}
              className="flex-1 flex items-center justify-center gap-1.5 py-2 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl text-xs font-bold hover:from-orange-600 hover:to-rose-600 shadow-sm transition-all active:scale-95"
            >
              <Zap className="w-3.5 h-3.5" /> Smart Assign
            </button>
          )}
          <button
            onClick={() => onEdit(evt)}
            className="p-2 border border-gray-200 hover:bg-orange-50 rounded-xl transition-colors"
          >
            <Edit className="w-4 h-4 text-orange-500" />
          </button>
          <button
            onClick={() => onDelete(evt)}
            className="p-2 border border-gray-200 hover:bg-red-50 rounded-xl transition-colors"
          >
            <Trash2 className="w-4 h-4 text-red-400" />
          </button>
        </div>
      </div>
    </div>
  );
}

// ═══════════════════════════════════════════════════════════
// STAT CARD
// ═══════════════════════════════════════════════════════════
function StatCard({ label, value, sub, Icon, bg, trend }) {
  return (
    <div className="bg-white rounded-2xl p-5 border border-gray-100 shadow-sm hover:shadow-md hover:-translate-y-0.5 transition-all group cursor-default">
      <div className="flex items-start justify-between mb-3">
        <div
          className={cx(
            "w-10 h-10 rounded-xl flex items-center justify-center shadow-sm group-hover:scale-110 transition-transform",
            bg,
          )}
        >
          <Icon className="w-5 h-5 text-white" />
        </div>
        {trend != null && (
          <span
            className={cx(
              "text-[11px] font-bold px-2 py-1 rounded-lg",
              trend >= 0
                ? "bg-emerald-50 text-emerald-600"
                : "bg-red-50 text-red-600",
            )}
          >
            {trend >= 0 ? "+" : ""}
            {trend}%
          </span>
        )}
      </div>
      <p className="text-[11px] font-bold text-gray-400 uppercase tracking-wider mb-0.5">
        {label}
      </p>
      <p className="text-3xl font-black text-gray-900 leading-none mb-1">
        {value}
      </p>
      {sub && <p className="text-xs text-gray-400">{sub}</p>}
    </div>
  );
}

// ═══════════════════════════════════════════════════════════
// MAIN APPLICATION
// ═══════════════════════════════════════════════════════════
function App() {
  const toast = useToast();

  // ── Core state — single source of truth ──
  const [vols, setVols] = useState(SEED_VOLS);
  const [evts, setEvts] = useState(SEED_EVTS);
  const [tab, setTab] = useState("dashboard");
  const [mNav, setMNav] = useState(false);

  // ── Modal state ──
  const [volForm, setVolForm] = useState({ open: false, editing: null });
  const [evtForm, setEvtForm] = useState({ open: false, editing: null });
  const [assign, setAssign] = useState({ open: false, eventId: null }); // store ID not object
  const [profile, setProfile] = useState({ open: false, vol: null });
  const [confirm, setConfirm] = useState({
    open: false,
    type: null,
    item: null,
  });

  // ── Filter state — LIVE derived, no stale refs ──
  const [vQ, setVQ] = useState(""); // volunteer search
  const [vSk, setVSk] = useState(""); // skill filter
  const [vSt, setVSt] = useState(""); // status filter
  const [eQ, setEQ] = useState(""); // event search
  const [eCt, setECt] = useState(""); // category filter

  // ══ Derived filtered lists — recomputed every render from live state ══
  // This is the KEY fix: no separate state, always in sync with vols/evts
  const filteredVols = useMemo(
    () =>
      vols.filter((v) => {
        const q = vQ.toLowerCase();
        return (
          (!q ||
            v.name.toLowerCase().includes(q) ||
            v.email.toLowerCase().includes(q) ||
            v.location.toLowerCase().includes(q) ||
            v.phone.includes(q) ||
            v.skills.some((s) => s.toLowerCase().includes(q))) &&
          (!vSk || v.skills.includes(vSk)) &&
          (!vSt || v.status === vSt)
        );
      }),
    [vols, vQ, vSk, vSt],
  ); // re-runs whenever vols state changes

  const filteredEvts = useMemo(
    () =>
      evts.filter((e) => {
        const q = eQ.toLowerCase();
        return (
          (!q ||
            e.title.toLowerCase().includes(q) ||
            e.desc.toLowerCase().includes(q) ||
            e.loc.toLowerCase().includes(q)) &&
          (!eCt || e.cat === eCt)
        );
      }),
    [evts, eQ, eCt],
  );

  // ══ Statistics — always derived from latest state ══
  const stats = useMemo(
    () => ({
      totalVols: vols.length,
      activeVols: vols.filter((v) => v.status === "active").length,
      totalEvts: evts.length,
      upcoming: evts.filter((e) => e.status !== "completed").length,
      assignments: evts.reduce((s, e) => s + e.vols.length, 0),
      tasks: vols.reduce((s, v) => s + v.tasks, 0),
      avgRating: vols.length
        ? (vols.reduce((s, v) => s + v.rating, 0) / vols.length).toFixed(1)
        : "—",
      fullyStaffed: evts.filter((e) => e.vols.length >= e.max).length,
    }),
    [vols, evts],
  );

  // ══════════════════════════════════════════════
  // VOLUNTEER CRUD — all update `vols` state
  // filteredVols auto-recomputes → instant sync
  // ══════════════════════════════════════════════
  const saveVolunteer = (form) => {
    if (volForm.editing) {
      // EDIT: replace the item in-place
      setVols((prev) =>
        prev.map((v) => (v.id === volForm.editing.id ? { ...v, ...form } : v)),
      );
      toast(`✅ ${form.name} updated successfully`, "success");
    } else {
      // ADD: prepend so it's immediately visible at the top
      const newVol = {
        id: Date.now(),
        rating: 0,
        tasks: 0,
        joined: new Date().toISOString().split("T")[0],
        ...form,
      };
      setVols((prev) => [newVol, ...prev]);
      // ↑ This single line updates vols → filteredVols auto-recomputes →
      //   search results, dashboard stats, assignment tab ALL update instantly.
      //   Zero page refresh, zero manual sync needed.
      toast(`🎉 ${form.name} added! Showing in search & all tabs.`, "success");
    }
    setVolForm({ open: false, editing: null });
  };

  const deleteVolunteer = (vol) => {
    // Remove from vols AND from all event assignment lists
    setVols((prev) => prev.filter((v) => v.id !== vol.id));
    setEvts((prev) =>
      prev.map((e) => ({ ...e, vols: e.vols.filter((id) => id !== vol.id) })),
    );
    toast(`🗑️ ${vol.name} removed from system`, "info");
    setConfirm({ open: false });
  };

  // ══════════════════════════════════════════════
  // EVENT CRUD
  // ══════════════════════════════════════════════
  const saveEvent = (form) => {
    if (evtForm.editing) {
      setEvts((prev) =>
        prev.map((e) => (e.id === evtForm.editing.id ? { ...e, ...form } : e)),
      );
      toast(`✅ "${form.title}" updated`, "success");
    } else {
      const newEvt = { id: Date.now(), vols: [], status: "upcoming", ...form };
      setEvts((prev) => [newEvt, ...prev]);
      toast(`📅 "${form.title}" created!`, "success");
    }
    setEvtForm({ open: false, editing: null });
  };

  const deleteEvent = (evt) => {
    setEvts((prev) => prev.filter((e) => e.id !== evt.id));
    toast(`🗑️ "${evt.title}" deleted`, "info");
    setConfirm({ open: false });
  };

  // ══════════════════════════════════════════════
  // ASSIGN / UNASSIGN
  // ══════════════════════════════════════════════
  const assignVol = (evtId, volId) => {
    const evt = evts.find((e) => e.id === evtId);
    const vol = vols.find((v) => v.id === volId);
    if (!evt || !vol) {
      toast("❌ Volunteer or event not found", "error");
      return;
    }
    if (evt.vols.length >= evt.max) {
      toast("❌ Event is fully staffed!", "error");
      return;
    }
    if (evt.vols.includes(volId)) {
      toast("ℹ️ Already assigned to this event", "warning");
      return;
    }
    // Update evts state — AssignModal auto-derives fresh event from evts, no stale ref
    setEvts((prev) =>
      prev.map((e) =>
        e.id === evtId ? { ...e, vols: [...e.vols, volId] } : e,
      ),
    );
    toast(`✅ ${vol.name} assigned to "${evt.title}"`, "success");
  };

  const unassignVol = (evtId, volId) => {
    const evt = evts.find((e) => e.id === evtId);
    const vol = vols.find((v) => v.id === volId);
    setEvts((prev) =>
      prev.map((e) =>
        e.id === evtId
          ? { ...e, vols: e.vols.filter((id) => id !== volId) }
          : e,
      ),
    );
    toast(`↩️ ${vol?.name} removed from "${evt?.title}"`, "info");
  };

  // ══ Confirm actions ══
  const handleConfirm = () => {
    if (confirm.type === "vol") deleteVolunteer(confirm.item);
    else deleteEvent(confirm.item);
  };

  // ══ Tabs ══
  const TABS = [
    { id: "dashboard", label: "Dashboard", Icon: BarChart2 },
    { id: "volunteers", label: "Volunteers", Icon: Users, count: vols.length },
    { id: "events", label: "Events", Icon: Calendar, count: evts.length },
    { id: "assignments", label: "Assignments", Icon: UserCheck },
  ];

  return (
    <div className="min-h-screen bg-slate-50">
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Sora:wght@400;600;700;800;900&family=DM+Sans:opsz,wght@9..40,400;9..40,500;9..40,600;9..40,700&display=swap');
        *, *::before, *::after { box-sizing: border-box; }
        * { font-family: 'DM Sans', sans-serif; }
        h1,h2,h3,h4,h5,h6 { font-family: 'Sora', sans-serif; }
        @keyframes fadein  { from { opacity:0 } to { opacity:1 } }
        @keyframes slideup { from { opacity:0; transform:translateY(26px) } to { opacity:1; transform:translateY(0) } }
        @keyframes toastSlide { from { opacity:0; transform:translateX(48px) } to { opacity:1; transform:translateX(0) } }
        .line-clamp-2 { display:-webkit-box;-webkit-line-clamp:2;-webkit-box-orient:vertical;overflow:hidden; }
        ::-webkit-scrollbar { width:4px; height:4px; }
        ::-webkit-scrollbar-track { background:transparent; }
        ::-webkit-scrollbar-thumb { background:#e2e8f0; border-radius:4px; }
        ::-webkit-scrollbar-thumb:hover { background:#cbd5e1; }
      `}</style>

      {/* ══ TOPNAV ══ */}
      <nav className="bg-white border-b border-gray-100 sticky top-0 z-40 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            {/* Logo */}
            <div className="flex items-center gap-3">
              <div className="w-9 h-9 bg-gradient-to-br from-orange-500 to-rose-500 rounded-xl flex items-center justify-center shadow-lg">
                <Heart className="w-5 h-5 text-white fill-white" />
              </div>
              <div>
                <span className="font-black text-xl text-gray-900 tracking-tight">
                  VolunteerHub
                </span>
                <p className="hidden sm:block text-[10px] text-gray-400 font-bold uppercase tracking-widest -mt-0.5">
                  Management System
                </p>
              </div>
            </div>

            {/* Desktop tabs */}
            <div className="hidden md:flex bg-gray-100/80 rounded-2xl p-1 gap-0.5">
              {TABS.map(({ id, label, Icon, count }) => (
                <button
                  key={id}
                  onClick={() => setTab(id)}
                  className={cx(
                    "flex items-center gap-2 px-4 py-2 rounded-xl text-sm font-semibold transition-all",
                    tab === id
                      ? "bg-white text-gray-900 shadow-sm"
                      : "text-gray-500 hover:text-gray-800 hover:bg-white/70",
                  )}
                >
                  <Icon className="w-4 h-4" />
                  {label}
                  {count != null && (
                    <span
                      className={cx(
                        "px-1.5 py-0.5 rounded-full text-[10px] font-black transition-colors",
                        tab === id
                          ? "bg-orange-100 text-orange-600"
                          : "bg-gray-200 text-gray-500",
                      )}
                    >
                      {count}
                    </span>
                  )}
                </button>
              ))}
            </div>

            {/* Right side */}
            <div className="flex items-center gap-2">
              <div className="hidden sm:flex items-center gap-1.5 px-3 py-1.5 bg-emerald-50 border border-emerald-100 rounded-xl">
                <span className="w-2 h-2 bg-emerald-400 rounded-full animate-pulse" />
                <span className="text-xs font-bold text-emerald-700">
                  {stats.activeVols} Active
                </span>
              </div>
              <button
                onClick={() => setMNav((p) => !p)}
                className="md:hidden p-2 hover:bg-gray-100 rounded-xl transition-colors"
              >
                <Menu className="w-5 h-5 text-gray-600" />
              </button>
            </div>
          </div>

          {/* Mobile nav */}
          {mNav && (
            <div className="md:hidden border-t border-gray-100 py-2 space-y-0.5">
              {TABS.map(({ id, label, Icon, count }) => (
                <button
                  key={id}
                  onClick={() => {
                    setTab(id);
                    setMNav(false);
                  }}
                  className={cx(
                    "w-full flex items-center gap-3 px-4 py-2.5 rounded-xl text-sm font-semibold transition-all",
                    tab === id
                      ? "bg-orange-50 text-orange-700"
                      : "text-gray-600 hover:bg-gray-50",
                  )}
                >
                  <Icon className="w-4 h-4" />
                  {label}
                  {count != null && (
                    <span className="ml-auto text-xs bg-gray-100 text-gray-500 px-2 py-0.5 rounded-full">
                      {count}
                    </span>
                  )}
                </button>
              ))}
            </div>
          )}
        </div>
      </nav>

      {/* ══ MAIN ══ */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6 sm:py-8">
        {/* ────────────── DASHBOARD ────────────── */}
        {tab === "dashboard" && (
          <div className="space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-end justify-between gap-4">
              <div>
                <h2 className="text-3xl sm:text-4xl font-black text-gray-900">
                  Dashboard
                </h2>
                <p className="text-gray-500 mt-1 text-sm">
                  Live stats — updates instantly when you add, edit, or delete
                  data
                </p>
              </div>
              <div className="flex gap-2">
                <button
                  onClick={() => setVolForm({ open: true, editing: null })}
                  className="px-4 py-2 bg-white border border-gray-200 rounded-xl text-sm font-semibold text-gray-700 hover:border-orange-300 hover:bg-orange-50 transition-all"
                >
                  + Volunteer
                </button>
                <button
                  onClick={() => setEvtForm({ open: true, editing: null })}
                  className="px-4 py-2 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl text-sm font-bold shadow-md hover:from-orange-600 hover:to-rose-600 transition-all"
                >
                  + New Event
                </button>
              </div>
            </div>

            {/* Stats grid */}
            <div className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-6 gap-3 sm:gap-4">
              <StatCard
                label="Volunteers"
                value={stats.totalVols}
                sub={`${stats.activeVols} active`}
                Icon={Users}
                bg="bg-gradient-to-br from-blue-400 to-blue-600"
                trend={8}
              />
              <StatCard
                label="Events"
                value={stats.totalEvts}
                sub={`${stats.upcoming} upcoming`}
                Icon={Calendar}
                bg="bg-gradient-to-br from-orange-400 to-orange-600"
                trend={12}
              />
              <StatCard
                label="Assignments"
                value={stats.assignments}
                sub="across all events"
                Icon={CheckCircle}
                bg="bg-gradient-to-br from-emerald-400 to-emerald-600"
                trend={5}
              />
              <StatCard
                label="Avg Rating"
                value={stats.avgRating}
                sub="out of 5.0"
                Icon={Star}
                bg="bg-gradient-to-br from-amber-400 to-amber-600"
              />
              <StatCard
                label="Tasks Done"
                value={stats.tasks}
                sub="all volunteers"
                Icon={Award}
                bg="bg-gradient-to-br from-purple-400 to-purple-600"
                trend={15}
              />
              <StatCard
                label="Fully Staffed"
                value={stats.fullyStaffed}
                sub={`of ${stats.totalEvts} events`}
                Icon={Shield}
                bg="bg-gradient-to-br from-rose-400 to-pink-600"
              />
            </div>

            {/* Two-col layout */}
            <div className="grid lg:grid-cols-2 gap-6">
              {/* Upcoming events */}
              <div className="bg-white rounded-2xl border border-gray-100 shadow-sm p-5">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Upcoming Events</h3>
                  <button
                    onClick={() => setTab("events")}
                    className="text-xs text-orange-600 font-bold hover:underline flex items-center gap-0.5"
                  >
                    View all <ChevronRight className="w-3 h-3" />
                  </button>
                </div>
                <div className="space-y-3">
                  {evts.slice(0, 4).map((e) => (
                    <div
                      key={e.id}
                      className="flex items-center gap-3 p-3 bg-gray-50 rounded-xl hover:bg-orange-50 transition-colors cursor-pointer group"
                      onClick={() => setTab("events")}
                    >
                      <span className="text-xl flex-shrink-0">
                        {C_ICON[e.cat] || "📋"}
                      </span>
                      <div className="flex-1 min-w-0">
                        <p className="font-bold text-gray-800 text-sm truncate">
                          {e.title}
                        </p>
                        <p className="text-xs text-gray-500">
                          {new Date(e.date).toLocaleDateString("en-US", {
                            month: "short",
                            day: "numeric",
                          })}{" "}
                          · {e.loc.split(",")[0]}
                        </p>
                        <Bar val={e.vols.length} max={e.max} />
                      </div>
                      <span className="text-xs font-bold text-gray-500 flex-shrink-0">
                        {e.vols.length}/{e.max}
                      </span>
                    </div>
                  ))}
                  {evts.length === 0 && (
                    <p className="text-sm text-gray-400 text-center py-6">
                      No events yet
                    </p>
                  )}
                </div>
              </div>

              {/* Top volunteers */}
              <div className="bg-white rounded-2xl border border-gray-100 shadow-sm p-5">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Top Volunteers</h3>
                  <button
                    onClick={() => setTab("volunteers")}
                    className="text-xs text-orange-600 font-bold hover:underline flex items-center gap-0.5"
                  >
                    View all <ChevronRight className="w-3 h-3" />
                  </button>
                </div>
                <div className="space-y-3">
                  {[...vols]
                    .sort((a, b) => b.rating - a.rating || b.tasks - a.tasks)
                    .slice(0, 5)
                    .map((v, i) => (
                      <div
                        key={v.id}
                        className="flex items-center gap-3 p-3 bg-gray-50 rounded-xl hover:bg-orange-50 transition-colors cursor-pointer"
                        onClick={() => setProfile({ open: true, vol: v })}
                      >
                        <span
                          className={cx(
                            "w-6 h-6 rounded-full flex items-center justify-center text-xs font-black flex-shrink-0",
                            i === 0
                              ? "bg-amber-400 text-white"
                              : i === 1
                                ? "bg-slate-300 text-slate-700"
                                : i === 2
                                  ? "bg-orange-300 text-white"
                                  : "bg-gray-100 text-gray-500",
                          )}
                        >
                          {i + 1}
                        </span>
                        <Avi name={v.name} size="sm" />
                        <div className="flex-1 min-w-0">
                          <p className="font-bold text-gray-800 text-sm truncate">
                            {v.name}
                          </p>
                          <p className="text-xs text-gray-500">
                            {v.tasks} tasks · {v.location.split(",")[0]}
                          </p>
                        </div>
                        <Stars val={v.rating} />
                      </div>
                    ))}
                  {vols.length === 0 && (
                    <p className="text-sm text-gray-400 text-center py-6">
                      No volunteers yet
                    </p>
                  )}
                </div>
              </div>
            </div>
          </div>
        )}

        {/* ────────────── VOLUNTEERS ────────────── */}
        {tab === "volunteers" && (
          <div className="space-y-5">
            {/* Header */}
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
              <div>
                <h2 className="text-3xl sm:text-4xl font-black text-gray-900">
                  Volunteers
                </h2>
                <p className="text-gray-500 mt-1 text-sm">
                  Showing{" "}
                  <span className="font-bold text-orange-600">
                    {filteredVols.length}
                  </span>{" "}
                  of {vols.length} volunteers
                  {(vQ || vSk || vSt) && (
                    <button
                      onClick={() => {
                        setVQ("");
                        setVSk("");
                        setVSt("");
                      }}
                      className="ml-2 text-orange-500 hover:underline text-xs font-bold"
                    >
                      <RefreshCw className="w-3 h-3 inline mr-0.5" />
                      Clear all filters
                    </button>
                  )}
                </p>
              </div>
              <button
                onClick={() => setVolForm({ open: true, editing: null })}
                className="w-full sm:w-auto flex items-center justify-center gap-2 px-5 py-2.5 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl font-bold text-sm shadow-md hover:from-orange-600 hover:to-rose-600 transition-all hover:-translate-y-0.5 active:translate-y-0"
              >
                <Plus className="w-4 h-4" /> Add Volunteer
              </button>
            </div>

            {/* Filters — tied to filteredVols which re-derives from vols state */}
            <div className="bg-white rounded-2xl border border-gray-100 p-4 shadow-sm">
              <div className="grid grid-cols-1 sm:grid-cols-3 gap-3">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400" />
                  <input
                    className={cx(INP, "pl-9 pr-8")}
                    value={vQ}
                    onChange={(e) => setVQ(e.target.value)}
                    placeholder="Name, email, skill, city…"
                  />
                  {vQ && (
                    <button
                      onClick={() => setVQ("")}
                      className="absolute right-3 top-1/2 -translate-y-1/2 p-0.5 hover:bg-gray-100 rounded-full"
                    >
                      <X className="w-3.5 h-3.5 text-gray-400" />
                    </button>
                  )}
                </div>
                <div className="relative">
                  <Filter className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400" />
                  <select
                    className={cx(INP, "pl-9")}
                    value={vSk}
                    onChange={(e) => setVSk(e.target.value)}
                  >
                    <option value="">All Skills</option>
                    {SKILLS.map((s) => (
                      <option key={s} value={s}>
                        {s}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400 pointer-events-none" />
                </div>
                <div className="relative">
                  <select
                    className={INP}
                    value={vSt}
                    onChange={(e) => setVSt(e.target.value)}
                  >
                    <option value="">All Statuses</option>
                    {STATUSES.map((s) => (
                      <option key={s} value={s}>
                        {s.charAt(0).toUpperCase() + s.slice(1)}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400 pointer-events-none" />
                </div>
              </div>
            </div>

            {/* Grid */}
            {filteredVols.length > 0 ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                {filteredVols.map((v) => (
                  <VolCard
                    key={v.id}
                    vol={v}
                    evtCount={evts.filter((e) => e.vols.includes(v.id)).length}
                    onView={(vol) => setProfile({ open: true, vol })}
                    onEdit={(vol) => setVolForm({ open: true, editing: vol })}
                    onDelete={(vol) =>
                      setConfirm({ open: true, type: "vol", item: vol })
                    }
                  />
                ))}
              </div>
            ) : (
              <div className="bg-white rounded-2xl border border-gray-100 shadow-sm flex flex-col items-center justify-center py-16 text-center px-4">
                <div className="w-16 h-16 bg-orange-50 rounded-2xl flex items-center justify-center mb-4">
                  <Users className="w-8 h-8 text-orange-300" />
                </div>
                <h3 className="text-lg font-bold text-gray-700 mb-1">
                  No volunteers found
                </h3>
                <p className="text-sm text-gray-500 mb-5">
                  {vQ || vSk || vSt
                    ? "Try different search terms or clear your filters"
                    : "Add your first volunteer to get started"}
                </p>
                {vQ || vSk || vSt ? (
                  <button
                    onClick={() => {
                      setVQ("");
                      setVSk("");
                      setVSt("");
                    }}
                    className="px-5 py-2 border border-orange-300 text-orange-600 rounded-xl text-sm font-bold hover:bg-orange-50 transition-colors"
                  >
                    Clear Filters
                  </button>
                ) : (
                  <button
                    onClick={() => setVolForm({ open: true, editing: null })}
                    className="px-5 py-2 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl text-sm font-bold shadow-sm"
                  >
                    + Add First Volunteer
                  </button>
                )}
              </div>
            )}
          </div>
        )}

        {/* ────────────── EVENTS ────────────── */}
        {tab === "events" && (
          <div className="space-y-5">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
              <div>
                <h2 className="text-3xl sm:text-4xl font-black text-gray-900">
                  Events
                </h2>
                <p className="text-gray-500 mt-1 text-sm">
                  Showing{" "}
                  <span className="font-bold text-orange-600">
                    {filteredEvts.length}
                  </span>{" "}
                  of {evts.length} events
                  {(eQ || eCt) && (
                    <button
                      onClick={() => {
                        setEQ("");
                        setECt("");
                      }}
                      className="ml-2 text-orange-500 hover:underline text-xs font-bold"
                    >
                      <RefreshCw className="w-3 h-3 inline mr-0.5" />
                      Clear filters
                    </button>
                  )}
                </p>
              </div>
              <button
                onClick={() => setEvtForm({ open: true, editing: null })}
                className="w-full sm:w-auto flex items-center justify-center gap-2 px-5 py-2.5 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl font-bold text-sm shadow-md hover:from-orange-600 hover:to-rose-600 transition-all hover:-translate-y-0.5"
              >
                <Plus className="w-4 h-4" /> Create Event
              </button>
            </div>

            <div className="bg-white rounded-2xl border border-gray-100 p-4 shadow-sm">
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400" />
                  <input
                    className={cx(INP, "pl-9 pr-8")}
                    value={eQ}
                    onChange={(e) => setEQ(e.target.value)}
                    placeholder="Search events…"
                  />
                  {eQ && (
                    <button
                      onClick={() => setEQ("")}
                      className="absolute right-3 top-1/2 -translate-y-1/2 p-0.5 hover:bg-gray-100 rounded-full"
                    >
                      <X className="w-3.5 h-3.5 text-gray-400" />
                    </button>
                  )}
                </div>
                <div className="relative">
                  <select
                    className={INP}
                    value={eCt}
                    onChange={(e) => setECt(e.target.value)}
                  >
                    <option value="">All Categories</option>
                    {CATS.map((c) => (
                      <option key={c} value={c}>
                        {C_ICON[c]} {c}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400 pointer-events-none" />
                </div>
              </div>
            </div>

            {filteredEvts.length > 0 ? (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {filteredEvts.map((e) => (
                  <EvtCard
                    key={e.id}
                    evt={e}
                    vols={vols}
                    onEdit={(evt) => setEvtForm({ open: true, editing: evt })}
                    onDelete={(evt) =>
                      setConfirm({ open: true, type: "evt", item: evt })
                    }
                    onAssign={(evtId) =>
                      setAssign({ open: true, eventId: evtId })
                    }
                    onUnassign={unassignVol}
                  />
                ))}
              </div>
            ) : (
              <div className="bg-white rounded-2xl border border-gray-100 shadow-sm flex flex-col items-center justify-center py-16 text-center px-4">
                <div className="w-16 h-16 bg-orange-50 rounded-2xl flex items-center justify-center mb-4">
                  <Calendar className="w-8 h-8 text-orange-300" />
                </div>
                <h3 className="text-lg font-bold text-gray-700 mb-1">
                  No events found
                </h3>
                <p className="text-sm text-gray-500 mb-5">
                  {eQ || eCt
                    ? "Try different filters"
                    : "Create your first event"}
                </p>
                <button
                  onClick={() => setEvtForm({ open: true, editing: null })}
                  className="px-5 py-2 bg-gradient-to-r from-orange-500 to-rose-500 text-white rounded-xl text-sm font-bold shadow-sm"
                >
                  + Create Event
                </button>
              </div>
            )}
          </div>
        )}

        {/* ────────────── ASSIGNMENTS ────────────── */}
        {tab === "assignments" && (
          <div className="space-y-5">
            <div>
              <h2 className="text-3xl sm:text-4xl font-black text-gray-900">
                Assignments
              </h2>
              <p className="text-gray-500 mt-1 text-sm">
                Live volunteer-to-event assignments — synced across all tabs
              </p>
            </div>

            {/* Summary */}
            <div className="grid grid-cols-2 sm:grid-cols-4 gap-3">
              {[
                {
                  label: "Total Assignments",
                  val: stats.assignments,
                  col: "text-orange-600",
                  bg: "bg-orange-50",
                },
                {
                  label: "Events Active",
                  val: evts.filter((e) => e.vols.length > 0).length,
                  col: "text-blue-600",
                  bg: "bg-blue-50",
                },
                {
                  label: "Fully Staffed",
                  val: stats.fullyStaffed,
                  col: "text-emerald-600",
                  bg: "bg-emerald-50",
                },
                {
                  label: "Needs Volunteers",
                  val: evts.filter((e) => e.vols.length < e.max).length,
                  col: "text-rose-600",
                  bg: "bg-rose-50",
                },
              ].map((s) => (
                <div
                  key={s.label}
                  className={cx(
                    s.bg,
                    "rounded-2xl border border-gray-100 p-4 text-center shadow-sm",
                  )}
                >
                  <p className={cx("text-2xl sm:text-3xl font-black", s.col)}>
                    {s.val}
                  </p>
                  <p className="text-xs font-semibold text-gray-500 mt-0.5 leading-tight">
                    {s.label}
                  </p>
                </div>
              ))}
            </div>

            {/* Event assignment cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {evts.map((e) => {
                const aVols = e.vols
                  .map((id) => vols.find((v) => v.id === id))
                  .filter(Boolean);
                const isFull = e.vols.length >= e.max;
                return (
                  <div
                    key={e.id}
                    className="bg-white rounded-2xl border border-gray-100 shadow-sm overflow-hidden hover:shadow-md transition-shadow"
                  >
                    <div
                      className={cx(
                        "h-1.5 bg-gradient-to-r",
                        C_COLOR[e.cat] || "from-gray-400 to-gray-500",
                      )}
                    />
                    <div className="p-5">
                      {/* Header */}
                      <div className="flex items-start justify-between gap-3 mb-3">
                        <div>
                          <h3 className="font-bold text-gray-900 flex items-center gap-2">
                            <span>{C_ICON[e.cat] || "📋"}</span>
                            <span>{e.title}</span>
                          </h3>
                          <p className="text-xs text-gray-500 mt-0.5">
                            {new Date(e.date).toLocaleDateString()} ·{" "}
                            {e.loc.split(",")[0]}
                          </p>
                        </div>
                        <span
                          className={cx(
                            "px-2.5 py-1 rounded-xl text-xs font-bold flex-shrink-0",
                            isFull
                              ? "bg-emerald-100 text-emerald-700"
                              : e.vols.length > 0
                                ? "bg-amber-100 text-amber-700"
                                : "bg-gray-100 text-gray-600",
                          )}
                        >
                          {isFull ? "✅ Full" : `${e.vols.length}/${e.max}`}
                        </span>
                      </div>

                      <Bar val={e.vols.length} max={e.max} green={isFull} />

                      {/* Assigned list */}
                      <div className="mt-4 space-y-2">
                        {aVols.length > 0 ? (
                          aVols.map((v) => (
                            <div
                              key={v.id}
                              className="flex items-center justify-between p-2.5 bg-gray-50 rounded-xl hover:bg-orange-50 transition-colors group"
                            >
                              <div className="flex items-center gap-2.5">
                                <Avi name={v.name} size="sm" />
                                <div>
                                  <p className="text-sm font-bold text-gray-800">
                                    {v.name}
                                  </p>
                                  <p className="text-xs text-gray-500">
                                    {v.skills
                                      .filter((s) => e.skills.includes(s))
                                      .join(", ") || "General volunteer"}
                                  </p>
                                </div>
                              </div>
                              <div className="flex items-center gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                <Stars val={v.rating} />
                                <button
                                  onClick={() => unassignVol(e.id, v.id)}
                                  className="p-1 hover:bg-red-100 rounded-lg transition-colors"
                                  title="Unassign"
                                >
                                  <X className="w-3.5 h-3.5 text-red-500" />
                                </button>
                              </div>
                            </div>
                          ))
                        ) : (
                          <p className="text-center text-sm text-gray-400 py-4">
                            No volunteers assigned yet
                          </p>
                        )}
                      </div>

                      {!isFull && (
                        <button
                          onClick={() =>
                            setAssign({ open: true, eventId: e.id })
                          }
                          className="mt-3 w-full py-2.5 border-2 border-dashed border-orange-200 text-orange-600 rounded-xl text-sm font-bold hover:border-orange-400 hover:bg-orange-50 transition-all flex items-center justify-center gap-1.5"
                        >
                          <Plus className="w-4 h-4" /> Assign Volunteer
                        </button>
                      )}
                    </div>
                  </div>
                );
              })}
              {evts.length === 0 && (
                <div className="col-span-2 bg-white rounded-2xl border border-gray-100 shadow-sm flex flex-col items-center justify-center py-16 text-center">
                  <Calendar className="w-10 h-10 text-gray-200 mb-3" />
                  <p className="text-gray-500 text-sm">
                    No events to manage.{" "}
                    <button
                      onClick={() => setTab("events")}
                      className="text-orange-600 font-bold hover:underline"
                    >
                      Create an event
                    </button>
                  </p>
                </div>
              )}
            </div>
          </div>
        )}
      </main>

      {/* ══ ALL MODALS ══ */}
      <VolForm
        open={volForm.open}
        close={() => setVolForm({ open: false, editing: null })}
        onSave={saveVolunteer}
        editing={volForm.editing}
      />

      <EvtForm
        open={evtForm.open}
        close={() => setEvtForm({ open: false, editing: null })}
        onSave={saveEvent}
        editing={evtForm.editing}
      />

      <AssignModal
        open={assign.open}
        close={() => setAssign({ open: false, eventId: null })}
        eventId={assign.eventId}
        evts={evts}
        vols={vols}
        onAssign={assignVol}
      />

      <ProfileModal
        open={profile.open}
        close={() => setProfile({ open: false, vol: null })}
        vol={profile.vol}
        evts={evts}
      />

      <DeleteModal
        open={confirm.open}
        close={() => setConfirm({ open: false })}
        onConfirm={handleConfirm}
        type={confirm.type}
        name={confirm.type === "vol" ? confirm.item?.name : confirm.item?.title}
      />
    </div>
  );
}

export default function VolunteerManagementSystem() {
  return (
    <ToastProvider>
      <App />
    </ToastProvider>
  );
}
