import React, { useState, useEffect } from "react";

export default function ToyShare() {
  const [currentView, setCurrentView] = useState("home"); // home, browse, toy-detail, cart, checkout, profile, my-bookings
  const [selectedToy, setSelectedToy] = useState(null);
  const [cart, setCart] = useState([]);
  const [showMobileMenu, setShowMobileMenu] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState("All");
  const [bookings, setBookings] = useState([]);
  const [user, setUser] = useState({
    name: "Raj Patel",
    email: "raj@example.com",
    wallet: 2500,
    points: 450,
  });

  // Toy inventory data
  const toys = [
    {
      id: 1,
      name: "LEGO Star Wars Millennium Falcon",
      category: "Building Sets",
      price: 150,
      dailyRate: 50,
      deposit: 500,
      rating: 4.9,
      reviews: 234,
      age: "9+",
      stock: 3,
      image: "🚀",
      description: "Build the iconic Millennium Falcon with 1351 pieces",
      features: [
        "1351 pieces",
        "7 minifigures",
        "Opening cockpit",
        "Rotating gun turrets",
      ],
      available: true,
    },
    {
      id: 2,
      name: "Barbie Dreamhouse",
      category: "Dolls",
      price: 120,
      dailyRate: 40,
      deposit: 400,
      rating: 4.7,
      reviews: 189,
      age: "3+",
      stock: 5,
      image: "🏠",
      description: "Three-story dreamhouse with pool and elevator",
      features: ["3 floors", "8 rooms", "Working elevator", "Pool slide"],
      available: true,
    },
    {
      id: 3,
      name: "Hot Wheels Ultimate Garage",
      category: "Vehicles",
      price: 80,
      dailyRate: 30,
      deposit: 300,
      rating: 4.8,
      reviews: 156,
      age: "5+",
      stock: 0,
      image: "🏎️",
      description: "Multi-level garage with car elevator and race tracks",
      features: [
        "Holds 140+ cars",
        "Motorized elevator",
        "2 car launchers",
        "T-Rex attack",
      ],
      available: false,
    },
    {
      id: 4,
      name: "Nerf Ultra One Blaster",
      category: "Action Toys",
      price: 60,
      dailyRate: 25,
      deposit: 200,
      rating: 4.6,
      reviews: 287,
      age: "8+",
      stock: 8,
      image: "🎯",
      description: "Motorized blaster with ultra dart technology",
      features: [
        "Fires 120 feet",
        "25-dart drum",
        "Motorized",
        "Includes 25 darts",
      ],
      available: true,
    },
    {
      id: 5,
      name: "Play-Doh Kitchen Creations",
      category: "Creative Play",
      price: 40,
      dailyRate: 15,
      deposit: 150,
      rating: 4.5,
      reviews: 342,
      age: "3+",
      stock: 12,
      image: "🎨",
      description: "Ultimate kitchen playset with 40+ accessories",
      features: [
        "40+ accessories",
        "5 Play-Doh colors",
        "Oven & mixer",
        "Food molds",
      ],
      available: true,
    },
    {
      id: 6,
      name: "Remote Control Excavator",
      category: "Vehicles",
      price: 90,
      dailyRate: 35,
      deposit: 350,
      rating: 4.7,
      reviews: 198,
      age: "6+",
      stock: 4,
      image: "🚜",
      description: "Full-function RC excavator with working lights",
      features: [
        "360° rotation",
        "Working lights",
        "2.4GHz remote",
        "Rechargeable",
      ],
      available: true,
    },
    {
      id: 7,
      name: "Baby Alive Potty Dance",
      category: "Dolls",
      price: 50,
      dailyRate: 20,
      deposit: 180,
      rating: 4.4,
      reviews: 267,
      age: "3+",
      stock: 6,
      image: "👶",
      description: "Interactive baby doll that drinks and uses potty",
      features: [
        "Drinks water",
        "Uses potty",
        "Sings & dances",
        "Includes accessories",
      ],
      available: true,
    },
    {
      id: 8,
      name: "LEGO Technic Bugatti",
      category: "Building Sets",
      price: 200,
      dailyRate: 70,
      deposit: 800,
      rating: 4.9,
      reviews: 145,
      age: "16+",
      stock: 2,
      image: "🏎️",
      description: "Ultimate LEGO Technic supercar with 3599 pieces",
      features: [
        "3599 pieces",
        "Working gearbox",
        "W16 engine",
        "Unique serial number",
      ],
      available: true,
    },
  ];

  const categories = [
    "All",
    "Building Sets",
    "Dolls",
    "Vehicles",
    "Action Toys",
    "Creative Play",
    "Educational",
    "Outdoor",
  ];

  // Filter toys
  const filteredToys = toys.filter((toy) => {
    const matchesCategory =
      selectedCategory === "All" || toy.category === selectedCategory;
    const matchesSearch =
      toy.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
      toy.description.toLowerCase().includes(searchQuery.toLowerCase());
    return matchesCategory && matchesSearch;
  });

  // Add to cart
  const addToCart = (toy, days = 3) => {
    const existing = cart.find((item) => item.id === toy.id);
    if (existing) {
      setCart(
        cart.map((item) =>
          item.id === toy.id ? { ...item, days: item.days + days } : item,
        ),
      );
    } else {
      setCart([...cart, { ...toy, days, bookingId: Date.now() }]);
    }
  };

  // Remove from cart
  const removeFromCart = (toyId) => {
    setCart(cart.filter((item) => item.id !== toyId));
  };

  // Calculate totals
  const cartTotal = cart.reduce(
    (sum, item) => sum + item.dailyRate * item.days,
    0,
  );
  const depositTotal = cart.reduce((sum, item) => sum + item.deposit, 0);
  const totalAmount = cartTotal + depositTotal;

  // Process checkout
  const processCheckout = () => {
    const newBookings = cart.map((item) => ({
      ...item,
      bookingId: Date.now() + Math.random(),
      startDate: new Date().toISOString().split("T")[0],
      endDate: new Date(Date.now() + item.days * 24 * 60 * 60 * 1000)
        .toISOString()
        .split("T")[0],
      status: "Active",
      paidAmount: item.dailyRate * item.days,
      depositPaid: item.deposit,
    }));

    setBookings([...bookings, ...newBookings]);
    setUser({
      ...user,
      wallet: user.wallet - totalAmount,
      points: user.points + Math.floor(totalAmount / 10),
    });
    setCart([]);
    setCurrentView("my-bookings");
  };

  return (
    <div
      style={{
        minHeight: "100vh",
        background: "#FFF5F7",
        fontFamily: '"DM Sans", system-ui, sans-serif',
        position: "relative",
      }}
    >
      {/* Navigation */}
      <nav
        style={{
          background: "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)",
          padding: "clamp(12px, 2vw, 20px) clamp(15px, 3vw, 40px)",
          position: "sticky",
          top: 0,
          zIndex: 100,
          boxShadow: "0 4px 30px rgba(255, 107, 157, 0.3)",
        }}
      >
        <div
          style={{
            width: "100%",
            margin: "0 auto",
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            gap: "20px",
            padding: "0 clamp(20px, 4vw, 60px)",
          }}
        >
          {/* Logo */}
          <div
            onClick={() => setCurrentView("home")}
            style={{
              display: "flex",
              alignItems: "center",
              gap: "12px",
              cursor: "pointer",
            }}
          >
            <div
              style={{
                fontSize: "clamp(32px, 5vw, 42px)",
                filter: "drop-shadow(2px 2px 4px rgba(0,0,0,0.2))",
              }}
            >
              🧸
            </div>
            <div>
              <h1
                style={{
                  margin: 0,
                  color: "white",
                  fontSize: "clamp(20px, 4vw, 28px)",
                  fontWeight: "800",
                  letterSpacing: "-0.5px",
                }}
              >
                ToyShare
              </h1>
              <p
                style={{
                  margin: 0,
                  color: "rgba(255,255,255,0.9)",
                  fontSize: "clamp(10px, 2vw, 12px)",
                  fontWeight: "500",
                }}
              >
                Rent. Play. Return.
              </p>
            </div>
          </div>

          {/* Desktop Menu */}
          <div
            className="desktop-menu"
            style={{
              display: "flex",
              gap: "clamp(15px, 3vw, 30px)",
              alignItems: "center",
            }}
          >
            <button
              onClick={() => setCurrentView("browse")}
              style={{
                background:
                  currentView === "browse"
                    ? "rgba(255,255,255,0.3)"
                    : "transparent",
                color: "white",
                border: "none",
                padding: "10px 20px",
                borderRadius: "12px",
                fontSize: "clamp(13px, 2vw, 15px)",
                fontWeight: "600",
                cursor: "pointer",
                transition: "all 0.3s",
              }}
            >
              Browse Toys
            </button>
            <button
              onClick={() => setCurrentView("my-bookings")}
              style={{
                background:
                  currentView === "my-bookings"
                    ? "rgba(255,255,255,0.3)"
                    : "transparent",
                color: "white",
                border: "none",
                padding: "10px 20px",
                borderRadius: "12px",
                fontSize: "clamp(13px, 2vw, 15px)",
                fontWeight: "600",
                cursor: "pointer",
                transition: "all 0.3s",
              }}
            >
              My Bookings
            </button>
            <button
              onClick={() => setCurrentView("cart")}
              style={{
                background: "white",
                color: "#FF6B9D",
                border: "none",
                padding: "10px 20px",
                borderRadius: "12px",
                fontSize: "clamp(13px, 2vw, 15px)",
                fontWeight: "700",
                cursor: "pointer",
                display: "flex",
                alignItems: "center",
                gap: "8px",
                boxShadow: "0 4px 15px rgba(0,0,0,0.1)",
                position: "relative",
              }}
            >
              🛒 Cart
              {cart.length > 0 && (
                <span
                  style={{
                    background: "#FF6B9D",
                    color: "white",
                    borderRadius: "50%",
                    width: "22px",
                    height: "22px",
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "center",
                    fontSize: "11px",
                    fontWeight: "700",
                  }}
                >
                  {cart.length}
                </span>
              )}
            </button>
            <div
              onClick={() => setCurrentView("profile")}
              style={{
                background: "rgba(255,255,255,0.2)",
                padding: "8px 15px",
                borderRadius: "12px",
                display: "flex",
                alignItems: "center",
                gap: "10px",
                cursor: "pointer",
                border: "2px solid rgba(255,255,255,0.3)",
              }}
            >
              <div
                style={{
                  background: "white",
                  borderRadius: "50%",
                  width: "35px",
                  height: "35px",
                  display: "flex",
                  alignItems: "center",
                  justifyContent: "center",
                  fontSize: "18px",
                }}
              >
                👤
              </div>
              <div className="desktop-only">
                <div
                  style={{ fontSize: "12px", color: "rgba(255,255,255,0.8)" }}
                >
                  Wallet
                </div>
                <div
                  style={{
                    fontSize: "14px",
                    color: "white",
                    fontWeight: "700",
                  }}
                >
                  ₹{user.wallet}
                </div>
              </div>
            </div>
          </div>

          {/* Mobile Menu Button */}
          <button
            className="mobile-menu-btn"
            onClick={() => setShowMobileMenu(!showMobileMenu)}
            style={{
              background: "rgba(255,255,255,0.2)",
              border: "2px solid rgba(255,255,255,0.3)",
              borderRadius: "10px",
              padding: "10px",
              fontSize: "24px",
              cursor: "pointer",
              display: "none",
            }}
          >
            ☰
          </button>
        </div>
      </nav>

      {/* Mobile Menu */}
      {showMobileMenu && (
        <div
          className="mobile-menu"
          style={{
            position: "fixed",
            top: "80px",
            left: 0,
            right: 0,
            background: "white",
            boxShadow: "0 10px 40px rgba(0,0,0,0.2)",
            padding: "20px",
            zIndex: 99,
            animation: "slideDown 0.3s ease-out",
          }}
        >
          <button
            onClick={() => {
              setCurrentView("browse");
              setShowMobileMenu(false);
            }}
            style={{
              width: "100%",
              background: "#FFF5F7",
              border: "none",
              padding: "15px",
              borderRadius: "12px",
              fontSize: "15px",
              fontWeight: "600",
              marginBottom: "10px",
              cursor: "pointer",
              textAlign: "left",
            }}
          >
            🎮 Browse Toys
          </button>
          <button
            onClick={() => {
              setCurrentView("my-bookings");
              setShowMobileMenu(false);
            }}
            style={{
              width: "100%",
              background: "#FFF5F7",
              border: "none",
              padding: "15px",
              borderRadius: "12px",
              fontSize: "15px",
              fontWeight: "600",
              marginBottom: "10px",
              cursor: "pointer",
              textAlign: "left",
            }}
          >
            📋 My Bookings
          </button>
          <button
            onClick={() => {
              setCurrentView("profile");
              setShowMobileMenu(false);
            }}
            style={{
              width: "100%",
              background: "#FFF5F7",
              border: "none",
              padding: "15px",
              borderRadius: "12px",
              fontSize: "15px",
              fontWeight: "600",
              cursor: "pointer",
              textAlign: "left",
            }}
          >
            👤 Profile & Wallet: ₹{user.wallet}
          </button>
        </div>
      )}

      {/* Main Content */}
      <div
        style={{
          width: "100%",
          margin: "0 auto",
          padding: "clamp(20px, 3vw, 60px)",
        }}
      >
        {/* HOME VIEW */}
        {currentView === "home" && (
          <div className="home-view">
            {/* Hero Section */}
            <div
              style={{
                background: "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                borderRadius: "clamp(20px, 3vw, 30px)",
                padding: "clamp(30px, 5vw, 60px)",
                marginBottom: "40px",
                position: "relative",
                overflow: "hidden",
                border: "3px solid #FFD6E0",
              }}
            >
              <div
                style={{
                  position: "absolute",
                  top: "-50px",
                  right: "-50px",
                  fontSize: "clamp(100px, 15vw, 200px)",
                  opacity: 0.1,
                  transform: "rotate(-15deg)",
                }}
              >
                🎈
              </div>

              <h2
                style={{
                  fontSize: "clamp(28px, 6vw, 48px)",
                  color: "#FF6B9D",
                  marginBottom: "15px",
                  fontWeight: "800",
                  lineHeight: "1.2",
                }}
              >
                Rent Premium Toys
                <br />
                <span style={{ color: "#C06C84" }}>Anytime, Anywhere! 🎉</span>
              </h2>

              <p
                style={{
                  fontSize: "clamp(14px, 2.5vw, 18px)",
                  color: "#666",
                  marginBottom: "30px",
                  maxWidth: "600px",
                  lineHeight: "1.6",
                }}
              >
                Save money, save space, and keep playtime exciting with our
                rotating toy library. Premium toys delivered to your doorstep!
                🚀
              </p>

              <button
                onClick={() => setCurrentView("browse")}
                style={{
                  background:
                    "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)",
                  color: "white",
                  border: "none",
                  padding: "clamp(12px, 2vw, 18px) clamp(25px, 4vw, 40px)",
                  borderRadius: "15px",
                  fontSize: "clamp(14px, 2.5vw, 18px)",
                  fontWeight: "700",
                  cursor: "pointer",
                  boxShadow: "0 8px 25px rgba(255, 107, 157, 0.4)",
                  transition: "transform 0.2s",
                }}
                onMouseEnter={(e) => (e.target.style.transform = "scale(1.05)")}
                onMouseLeave={(e) => (e.target.style.transform = "scale(1)")}
              >
                Start Browsing 🎮
              </button>
            </div>

            {/* Features */}
            <h3
              style={{
                fontSize: "clamp(20px, 4vw, 28px)",
                color: "#333",
                marginBottom: "25px",
                fontWeight: "700",
              }}
            >
              Why Choose ToyShare? ✨
            </h3>

            <div
              className="features-grid"
              style={{
                display: "grid",
                gridTemplateColumns: "repeat(auto-fit, minmax(220px, 1fr))",
                gap: "clamp(15px, 3vw, 25px)",
                marginBottom: "40px",
              }}
            >
              {[
                {
                  icon: "💰",
                  title: "Save Money",
                  desc: "Rent instead of buying expensive toys",
                },
                {
                  icon: "🏠",
                  title: "Save Space",
                  desc: "No clutter, just fun toys when needed",
                },
                {
                  icon: "🔄",
                  title: "Keep It Fresh",
                  desc: "Rotate toys to keep kids engaged",
                },
                {
                  icon: "🚚",
                  title: "Home Delivery",
                  desc: "Delivered & picked up from your door",
                },
              ].map((feature, idx) => (
                <div
                  key={idx}
                  style={{
                    background: "white",
                    padding: "clamp(20px, 3vw, 30px)",
                    borderRadius: "20px",
                    border: "3px solid #FFE5EC",
                    textAlign: "center",
                    boxShadow: "0 5px 20px rgba(255, 107, 157, 0.1)",
                  }}
                >
                  <div
                    style={{
                      fontSize: "clamp(35px, 6vw, 50px)",
                      marginBottom: "10px",
                    }}
                  >
                    {feature.icon}
                  </div>
                  <h4
                    style={{
                      fontSize: "clamp(16px, 3vw, 20px)",
                      color: "#FF6B9D",
                      marginBottom: "8px",
                      fontWeight: "700",
                    }}
                  >
                    {feature.title}
                  </h4>
                  <p
                    style={{
                      fontSize: "clamp(13px, 2vw, 15px)",
                      color: "#666",
                      margin: 0,
                    }}
                  >
                    {feature.desc}
                  </p>
                </div>
              ))}
            </div>

            {/* Popular Toys */}
            <h3
              style={{
                fontSize: "clamp(20px, 4vw, 28px)",
                color: "#333",
                marginBottom: "25px",
                fontWeight: "700",
              }}
            >
              Popular Toys 🔥
            </h3>

            <div
              className="popular-toys-grid"
              style={{
                display: "grid",
                gridTemplateColumns:
                  "repeat(auto-fill, minmax(min(100%, 260px), 1fr))",
                gap: "clamp(15px, 2vw, 25px)",
              }}
            >
              {toys.slice(0, 4).map((toy) => (
                <div
                  key={toy.id}
                  onClick={() => {
                    setSelectedToy(toy);
                    setCurrentView("toy-detail");
                  }}
                  style={{
                    background: "white",
                    borderRadius: "20px",
                    overflow: "hidden",
                    border: "3px solid #FFE5EC",
                    cursor: "pointer",
                    transition: "all 0.3s",
                    boxShadow: "0 5px 20px rgba(255, 107, 157, 0.1)",
                  }}
                  onMouseEnter={(e) => {
                    e.currentTarget.style.transform = "translateY(-8px)";
                    e.currentTarget.style.boxShadow =
                      "0 15px 40px rgba(255, 107, 157, 0.2)";
                  }}
                  onMouseLeave={(e) => {
                    e.currentTarget.style.transform = "translateY(0)";
                    e.currentTarget.style.boxShadow =
                      "0 5px 20px rgba(255, 107, 157, 0.1)";
                  }}
                >
                  <div
                    style={{
                      height: "clamp(140px, 20vw, 180px)",
                      background:
                        "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                      display: "flex",
                      alignItems: "center",
                      justifyContent: "center",
                      fontSize: "clamp(60px, 10vw, 80px)",
                      position: "relative",
                    }}
                  >
                    {toy.image}
                    {!toy.available && (
                      <div
                        style={{
                          position: "absolute",
                          top: "10px",
                          right: "10px",
                          background: "#FF6B9D",
                          color: "white",
                          padding: "5px 10px",
                          borderRadius: "8px",
                          fontSize: "11px",
                          fontWeight: "700",
                        }}
                      >
                        Out of Stock
                      </div>
                    )}
                  </div>
                  <div style={{ padding: "clamp(12px, 2vw, 15px)" }}>
                    <h4
                      style={{
                        fontSize: "clamp(14px, 2.5vw, 16px)",
                        color: "#333",
                        margin: "0 0 8px 0",
                        fontWeight: "700",
                      }}
                    >
                      {toy.name}
                    </h4>
                    <div
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        alignItems: "center",
                        marginBottom: "10px",
                      }}
                    >
                      <span
                        style={{
                          fontSize: "clamp(18px, 3vw, 22px)",
                          color: "#FF6B9D",
                          fontWeight: "800",
                        }}
                      >
                        ₹{toy.dailyRate}/day
                      </span>
                      <span
                        style={{
                          background: "#FFE5EC",
                          color: "#FF6B9D",
                          padding: "4px 10px",
                          borderRadius: "8px",
                          fontSize: "clamp(11px, 2vw, 13px)",
                          fontWeight: "600",
                        }}
                      >
                        ⭐ {toy.rating}
                      </span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* BROWSE VIEW */}
        {currentView === "browse" && (
          <div className="browse-view">
            <h2
              style={{
                fontSize: "clamp(24px, 5vw, 36px)",
                color: "#333",
                marginBottom: "20px",
                fontWeight: "800",
              }}
            >
              Browse All Toys 🎮
            </h2>

            {/* Search Bar */}
            <div
              style={{
                background: "white",
                padding: "clamp(15px, 2vw, 20px)",
                borderRadius: "20px",
                marginBottom: "20px",
                border: "3px solid #FFE5EC",
                boxShadow: "0 5px 20px rgba(255, 107, 157, 0.1)",
              }}
            >
              <input
                type="text"
                placeholder="🔍 Search toys..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                style={{
                  width: "100%",
                  border: "none",
                  outline: "none",
                  fontSize: "clamp(14px, 2.5vw, 16px)",
                  background: "transparent",
                }}
              />
            </div>

            {/* Categories */}
            <div
              style={{
                overflowX: "auto",
                marginBottom: "30px",
                WebkitOverflowScrolling: "touch",
              }}
            >
              <div
                style={{
                  display: "flex",
                  gap: "10px",
                  paddingBottom: "10px",
                }}
              >
                {categories.map((cat) => (
                  <button
                    key={cat}
                    onClick={() => setSelectedCategory(cat)}
                    style={{
                      background:
                        selectedCategory === cat
                          ? "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)"
                          : "white",
                      color: selectedCategory === cat ? "white" : "#666",
                      border:
                        selectedCategory === cat ? "none" : "2px solid #FFE5EC",
                      padding:
                        "clamp(8px, 1.5vw, 12px) clamp(15px, 2.5vw, 20px)",
                      borderRadius: "12px",
                      fontSize: "clamp(12px, 2vw, 14px)",
                      fontWeight: "600",
                      cursor: "pointer",
                      whiteSpace: "nowrap",
                      flexShrink: 0,
                      transition: "all 0.3s",
                    }}
                  >
                    {cat}
                  </button>
                ))}
              </div>
            </div>

            {/* Toys Grid */}
            <div
              className="toys-grid"
              style={{
                display: "grid",
                gridTemplateColumns:
                  "repeat(auto-fill, minmax(min(100%, 260px), 1fr))",
                gap: "clamp(15px, 2vw, 25px)",
              }}
            >
              {filteredToys.map((toy) => (
                <div
                  key={toy.id}
                  onClick={() => {
                    setSelectedToy(toy);
                    setCurrentView("toy-detail");
                  }}
                  style={{
                    background: "white",
                    borderRadius: "20px",
                    overflow: "hidden",
                    border: "3px solid #FFE5EC",
                    cursor: "pointer",
                    transition: "all 0.3s",
                    boxShadow: "0 5px 20px rgba(255, 107, 157, 0.1)",
                    opacity: toy.available ? 1 : 0.6,
                  }}
                  onMouseEnter={(e) => {
                    if (toy.available) {
                      e.currentTarget.style.transform = "translateY(-8px)";
                      e.currentTarget.style.boxShadow =
                        "0 15px 40px rgba(255, 107, 157, 0.2)";
                    }
                  }}
                  onMouseLeave={(e) => {
                    e.currentTarget.style.transform = "translateY(0)";
                    e.currentTarget.style.boxShadow =
                      "0 5px 20px rgba(255, 107, 157, 0.1)";
                  }}
                >
                  <div
                    style={{
                      height: "clamp(140px, 20vw, 180px)",
                      background:
                        "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                      display: "flex",
                      alignItems: "center",
                      justifyContent: "center",
                      fontSize: "clamp(60px, 10vw, 80px)",
                      position: "relative",
                    }}
                  >
                    {toy.image}
                    {!toy.available && (
                      <div
                        style={{
                          position: "absolute",
                          top: "10px",
                          right: "10px",
                          background: "#FF6B9D",
                          color: "white",
                          padding: "5px 10px",
                          borderRadius: "8px",
                          fontSize: "11px",
                          fontWeight: "700",
                        }}
                      >
                        Out of Stock
                      </div>
                    )}
                    {toy.stock <= 3 && toy.available && (
                      <div
                        style={{
                          position: "absolute",
                          top: "10px",
                          left: "10px",
                          background: "#FFA500",
                          color: "white",
                          padding: "5px 10px",
                          borderRadius: "8px",
                          fontSize: "11px",
                          fontWeight: "700",
                        }}
                      >
                        Only {toy.stock} left
                      </div>
                    )}
                  </div>
                  <div style={{ padding: "clamp(12px, 2vw, 15px)" }}>
                    <div
                      style={{
                        fontSize: "clamp(10px, 1.5vw, 11px)",
                        color: "#999",
                        marginBottom: "5px",
                        fontWeight: "600",
                      }}
                    >
                      {toy.category}
                    </div>
                    <h4
                      style={{
                        fontSize: "clamp(14px, 2.5vw, 16px)",
                        color: "#333",
                        margin: "0 0 8px 0",
                        fontWeight: "700",
                        minHeight: "40px",
                      }}
                    >
                      {toy.name}
                    </h4>
                    <div
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        alignItems: "center",
                        marginBottom: "10px",
                      }}
                    >
                      <div>
                        <div
                          style={{
                            fontSize: "clamp(18px, 3vw, 22px)",
                            color: "#FF6B9D",
                            fontWeight: "800",
                          }}
                        >
                          ₹{toy.dailyRate}/day
                        </div>
                        <div
                          style={{
                            fontSize: "clamp(10px, 1.5vw, 11px)",
                            color: "#999",
                          }}
                        >
                          Deposit: ₹{toy.deposit}
                        </div>
                      </div>
                      <div>
                        <div
                          style={{
                            background: "#FFE5EC",
                            color: "#FF6B9D",
                            padding: "4px 10px",
                            borderRadius: "8px",
                            fontSize: "clamp(11px, 2vw, 13px)",
                            fontWeight: "600",
                            marginBottom: "5px",
                          }}
                        >
                          ⭐ {toy.rating}
                        </div>
                        <div
                          style={{
                            fontSize: "clamp(10px, 1.5vw, 11px)",
                            color: "#999",
                            textAlign: "center",
                          }}
                        >
                          {toy.reviews} reviews
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>

            {filteredToys.length === 0 && (
              <div
                style={{
                  textAlign: "center",
                  padding: "60px 20px",
                  color: "#999",
                  background: "white",
                  borderRadius: "25px",
                  border: "3px solid #FFE5EC",
                  minHeight: "400px",
                  display: "flex",
                  flexDirection: "column",
                  justifyContent: "center",
                  alignItems: "center",
                }}
              >
                <div
                  style={{
                    fontSize: "clamp(60px, 10vw, 80px)",
                    marginBottom: "20px",
                  }}
                >
                  🔍
                </div>
                <h3
                  style={{
                    fontSize: "clamp(18px, 3vw, 24px)",
                    marginBottom: "10px",
                  }}
                >
                  No toys found
                </h3>
                <p style={{ fontSize: "clamp(14px, 2vw, 16px)" }}>
                  Try adjusting your search or filters
                </p>
              </div>
            )}
          </div>
        )}

        {/* TOY DETAIL VIEW */}
        {currentView === "toy-detail" && selectedToy && (
          <div className="toy-detail-view">
            <button
              onClick={() => setCurrentView("browse")}
              style={{
                background: "white",
                border: "2px solid #FFE5EC",
                borderRadius: "12px",
                padding: "10px 20px",
                fontSize: "14px",
                fontWeight: "600",
                color: "#666",
                cursor: "pointer",
                marginBottom: "20px",
              }}
            >
              ← Back to Browse
            </button>

            <div
              style={{
                background: "white",
                borderRadius: "25px",
                overflow: "hidden",
                border: "3px solid #FFE5EC",
                boxShadow: "0 10px 40px rgba(255, 107, 157, 0.15)",
              }}
            >
              <div
                className="toy-detail-grid"
                style={{
                  display: "grid",
                  gridTemplateColumns: "1fr",
                  gap: "30px",
                }}
              >
                {/* Image Section */}
                <div
                  style={{
                    background:
                      "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                    padding: "clamp(40px, 6vw, 80px)",
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "center",
                    position: "relative",
                    minHeight: "400px",
                  }}
                >
                  <div
                    style={{
                      fontSize: "clamp(120px, 20vw, 200px)",
                    }}
                  >
                    {selectedToy.image}
                  </div>
                  {!selectedToy.available && (
                    <div
                      style={{
                        position: "absolute",
                        top: "20px",
                        right: "20px",
                        background: "#FF6B9D",
                        color: "white",
                        padding: "10px 20px",
                        borderRadius: "12px",
                        fontSize: "14px",
                        fontWeight: "700",
                      }}
                    >
                      Out of Stock
                    </div>
                  )}
                </div>

                {/* Details Section */}
                <div
                  style={{
                    padding: "clamp(20px, 4vw, 40px)",
                  }}
                >
                  <div
                    style={{
                      fontSize: "clamp(12px, 2vw, 14px)",
                      color: "#FF6B9D",
                      fontWeight: "600",
                      marginBottom: "10px",
                    }}
                  >
                    {selectedToy.category}
                  </div>

                  <h2
                    style={{
                      fontSize: "clamp(24px, 4vw, 36px)",
                      color: "#333",
                      marginBottom: "15px",
                      fontWeight: "800",
                      lineHeight: "1.2",
                    }}
                  >
                    {selectedToy.name}
                  </h2>

                  <div
                    style={{
                      display: "flex",
                      gap: "15px",
                      marginBottom: "20px",
                      flexWrap: "wrap",
                    }}
                  >
                    <div
                      style={{
                        background: "#FFE5EC",
                        padding: "8px 15px",
                        borderRadius: "10px",
                        fontSize: "clamp(13px, 2vw, 15px)",
                        fontWeight: "600",
                        color: "#FF6B9D",
                      }}
                    >
                      ⭐ {selectedToy.rating} ({selectedToy.reviews} reviews)
                    </div>
                    <div
                      style={{
                        background: "#E8F5E9",
                        padding: "8px 15px",
                        borderRadius: "10px",
                        fontSize: "clamp(13px, 2vw, 15px)",
                        fontWeight: "600",
                        color: "#4CAF50",
                      }}
                    >
                      Age: {selectedToy.age}
                    </div>
                    {selectedToy.available && (
                      <div
                        style={{
                          background: "#E3F2FD",
                          padding: "8px 15px",
                          borderRadius: "10px",
                          fontSize: "clamp(13px, 2vw, 15px)",
                          fontWeight: "600",
                          color: "#2196F3",
                        }}
                      >
                        {selectedToy.stock} in stock
                      </div>
                    )}
                  </div>

                  <p
                    style={{
                      fontSize: "clamp(14px, 2vw, 16px)",
                      color: "#666",
                      lineHeight: "1.6",
                      marginBottom: "20px",
                    }}
                  >
                    {selectedToy.description}
                  </p>

                  <div
                    style={{
                      background: "#FFF9E6",
                      padding: "20px",
                      borderRadius: "15px",
                      marginBottom: "25px",
                      border: "2px solid #FFE5A0",
                    }}
                  >
                    <h4
                      style={{
                        fontSize: "clamp(14px, 2vw, 16px)",
                        color: "#333",
                        marginBottom: "12px",
                        fontWeight: "700",
                      }}
                    >
                      ✨ Key Features:
                    </h4>
                    <ul
                      style={{
                        margin: 0,
                        paddingLeft: "20px",
                        color: "#666",
                        fontSize: "clamp(13px, 2vw, 14px)",
                        lineHeight: "1.8",
                      }}
                    >
                      {selectedToy.features.map((feature, idx) => (
                        <li key={idx}>{feature}</li>
                      ))}
                    </ul>
                  </div>

                  <div
                    style={{
                      background:
                        "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                      padding: "25px",
                      borderRadius: "15px",
                      marginBottom: "25px",
                    }}
                  >
                    <div
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        alignItems: "flex-start",
                        flexWrap: "wrap",
                        gap: "15px",
                      }}
                    >
                      <div>
                        <div
                          style={{
                            fontSize: "clamp(12px, 2vw, 14px)",
                            color: "#666",
                            marginBottom: "5px",
                          }}
                        >
                          Daily Rental
                        </div>
                        <div
                          style={{
                            fontSize: "clamp(28px, 5vw, 40px)",
                            color: "#FF6B9D",
                            fontWeight: "800",
                          }}
                        >
                          ₹{selectedToy.dailyRate}
                        </div>
                        <div
                          style={{
                            fontSize: "clamp(11px, 2vw, 12px)",
                            color: "#999",
                          }}
                        >
                          per day
                        </div>
                      </div>
                      <div>
                        <div
                          style={{
                            fontSize: "clamp(12px, 2vw, 14px)",
                            color: "#666",
                            marginBottom: "5px",
                          }}
                        >
                          Refundable Deposit
                        </div>
                        <div
                          style={{
                            fontSize: "clamp(22px, 4vw, 28px)",
                            color: "#C06C84",
                            fontWeight: "700",
                          }}
                        >
                          ₹{selectedToy.deposit}
                        </div>
                        <div
                          style={{
                            fontSize: "clamp(11px, 2vw, 12px)",
                            color: "#999",
                          }}
                        >
                          returned after
                        </div>
                      </div>
                    </div>
                  </div>

                  {selectedToy.available ? (
                    <div
                      style={{
                        display: "flex",
                        gap: "15px",
                        flexWrap: "wrap",
                      }}
                    >
                      <button
                        onClick={() => {
                          addToCart(selectedToy, 3);
                          setCurrentView("cart");
                        }}
                        style={{
                          flex: 1,
                          minWidth: "200px",
                          background:
                            "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)",
                          color: "white",
                          border: "none",
                          padding: "18px 30px",
                          borderRadius: "15px",
                          fontSize: "clamp(15px, 2.5vw, 18px)",
                          fontWeight: "700",
                          cursor: "pointer",
                          boxShadow: "0 8px 25px rgba(255, 107, 157, 0.4)",
                          transition: "transform 0.2s",
                        }}
                        onMouseEnter={(e) =>
                          (e.target.style.transform = "scale(1.05)")
                        }
                        onMouseLeave={(e) =>
                          (e.target.style.transform = "scale(1)")
                        }
                      >
                        Add to Cart 🛒
                      </button>
                      <button
                        onClick={() => setCurrentView("browse")}
                        style={{
                          background: "white",
                          color: "#FF6B9D",
                          border: "3px solid #FF6B9D",
                          padding: "18px 30px",
                          borderRadius: "15px",
                          fontSize: "clamp(15px, 2.5vw, 18px)",
                          fontWeight: "700",
                          cursor: "pointer",
                          transition: "all 0.2s",
                        }}
                      >
                        Continue Browsing
                      </button>
                    </div>
                  ) : (
                    <div
                      style={{
                        background: "#FFE5EC",
                        padding: "20px",
                        borderRadius: "15px",
                        textAlign: "center",
                        color: "#FF6B9D",
                        fontSize: "clamp(14px, 2vw, 16px)",
                        fontWeight: "600",
                      }}
                    >
                      😔 Currently out of stock. Check back soon!
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        )}

        {/* CART VIEW */}
        {currentView === "cart" && (
          <div className="cart-view">
            <h2
              style={{
                fontSize: "clamp(24px, 5vw, 36px)",
                color: "#333",
                marginBottom: "25px",
                fontWeight: "800",
              }}
            >
              Your Cart 🛒
            </h2>

            {cart.length === 0 ? (
              <div
                style={{
                  background: "white",
                  borderRadius: "25px",
                  padding: "clamp(40px, 6vw, 80px)",
                  textAlign: "center",
                  border: "3px solid #FFE5EC",
                  minHeight: "500px",
                  display: "flex",
                  flexDirection: "column",
                  justifyContent: "center",
                  alignItems: "center",
                }}
              >
                <div
                  style={{
                    fontSize: "clamp(80px, 12vw, 120px)",
                    marginBottom: "20px",
                  }}
                >
                  🛒
                </div>
                <h3
                  style={{
                    fontSize: "clamp(20px, 4vw, 28px)",
                    color: "#333",
                    marginBottom: "15px",
                    fontWeight: "700",
                  }}
                >
                  Your cart is empty
                </h3>
                <p
                  style={{
                    fontSize: "clamp(14px, 2vw, 16px)",
                    color: "#666",
                    marginBottom: "25px",
                  }}
                >
                  Start adding some amazing toys!
                </p>
                <button
                  onClick={() => setCurrentView("browse")}
                  style={{
                    background:
                      "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)",
                    color: "white",
                    border: "none",
                    padding: "15px 35px",
                    borderRadius: "15px",
                    fontSize: "clamp(14px, 2.5vw, 16px)",
                    fontWeight: "700",
                    cursor: "pointer",
                    boxShadow: "0 8px 25px rgba(255, 107, 157, 0.4)",
                  }}
                >
                  Browse Toys 🎮
                </button>
              </div>
            ) : (
              <div
                className="cart-grid"
                style={{
                  display: "grid",
                  gridTemplateColumns: "1fr",
                  gap: "25px",
                }}
              >
                {/* Cart Items */}
                <div>
                  {cart.map((item) => (
                    <div
                      key={item.bookingId}
                      style={{
                        background: "white",
                        borderRadius: "20px",
                        padding: "clamp(15px, 3vw, 25px)",
                        marginBottom: "15px",
                        border: "3px solid #FFE5EC",
                        boxShadow: "0 5px 20px rgba(255, 107, 157, 0.1)",
                      }}
                    >
                      <div
                        style={{
                          display: "flex",
                          gap: "20px",
                          alignItems: "flex-start",
                        }}
                      >
                        <div
                          style={{
                            background:
                              "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                            borderRadius: "15px",
                            padding: "20px",
                            fontSize: "clamp(40px, 6vw, 60px)",
                            flexShrink: 0,
                          }}
                        >
                          {item.image}
                        </div>
                        <div style={{ flex: 1, minWidth: 0 }}>
                          <h4
                            style={{
                              fontSize: "clamp(16px, 2.5vw, 20px)",
                              color: "#333",
                              marginBottom: "8px",
                              fontWeight: "700",
                            }}
                          >
                            {item.name}
                          </h4>
                          <div
                            style={{
                              fontSize: "clamp(12px, 2vw, 14px)",
                              color: "#999",
                              marginBottom: "12px",
                            }}
                          >
                            {item.category}
                          </div>
                          <div
                            style={{
                              display: "flex",
                              gap: "15px",
                              flexWrap: "wrap",
                              alignItems: "center",
                              marginBottom: "15px",
                            }}
                          >
                            <div>
                              <label
                                style={{
                                  fontSize: "clamp(12px, 2vw, 13px)",
                                  color: "#666",
                                  display: "block",
                                  marginBottom: "5px",
                                }}
                              >
                                Rental Days:
                              </label>
                              <input
                                type="number"
                                min="1"
                                max="30"
                                value={item.days}
                                onChange={(e) => {
                                  const days = Math.max(
                                    1,
                                    Math.min(30, parseInt(e.target.value) || 1),
                                  );
                                  setCart(
                                    cart.map((i) =>
                                      i.bookingId === item.bookingId
                                        ? { ...i, days }
                                        : i,
                                    ),
                                  );
                                }}
                                style={{
                                  width: "80px",
                                  padding: "8px 12px",
                                  borderRadius: "10px",
                                  border: "2px solid #FFE5EC",
                                  fontSize: "clamp(14px, 2vw, 16px)",
                                  fontWeight: "600",
                                  color: "#FF6B9D",
                                }}
                              />
                            </div>
                            <div>
                              <div
                                style={{
                                  fontSize: "clamp(11px, 2vw, 12px)",
                                  color: "#666",
                                }}
                              >
                                Daily Rate
                              </div>
                              <div
                                style={{
                                  fontSize: "clamp(16px, 3vw, 20px)",
                                  color: "#FF6B9D",
                                  fontWeight: "700",
                                }}
                              >
                                ₹{item.dailyRate}/day
                              </div>
                            </div>
                            <div>
                              <div
                                style={{
                                  fontSize: "clamp(11px, 2vw, 12px)",
                                  color: "#666",
                                }}
                              >
                                Total Rental
                              </div>
                              <div
                                style={{
                                  fontSize: "clamp(16px, 3vw, 20px)",
                                  color: "#FF6B9D",
                                  fontWeight: "700",
                                }}
                              >
                                ₹{item.dailyRate * item.days}
                              </div>
                            </div>
                          </div>
                          <div
                            style={{
                              background: "#FFF9E6",
                              padding: "10px 15px",
                              borderRadius: "10px",
                              fontSize: "clamp(12px, 2vw, 13px)",
                              color: "#666",
                              marginBottom: "12px",
                            }}
                          >
                            💰 Refundable Deposit: ₹{item.deposit}
                          </div>
                          <button
                            onClick={() => removeFromCart(item.id)}
                            style={{
                              background: "#FFE5EC",
                              color: "#FF6B9D",
                              border: "none",
                              padding: "8px 15px",
                              borderRadius: "10px",
                              fontSize: "clamp(12px, 2vw, 14px)",
                              fontWeight: "600",
                              cursor: "pointer",
                            }}
                          >
                            Remove
                          </button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>

                {/* Order Summary */}
                <div
                  style={{
                    background: "white",
                    borderRadius: "20px",
                    padding: "clamp(20px, 3vw, 30px)",
                    border: "3px solid #FFE5EC",
                    height: "fit-content",
                    position: "sticky",
                    top: "100px",
                    boxShadow: "0 5px 20px rgba(255, 107, 157, 0.1)",
                  }}
                >
                  <h3
                    style={{
                      fontSize: "clamp(18px, 3vw, 24px)",
                      color: "#333",
                      marginBottom: "20px",
                      fontWeight: "700",
                    }}
                  >
                    Order Summary
                  </h3>

                  <div
                    style={{
                      borderBottom: "2px dashed #FFE5EC",
                      paddingBottom: "15px",
                      marginBottom: "15px",
                    }}
                  >
                    <div
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        marginBottom: "10px",
                        fontSize: "clamp(13px, 2vw, 14px)",
                        color: "#666",
                      }}
                    >
                      <span>Rental Charges</span>
                      <span style={{ fontWeight: "600" }}>₹{cartTotal}</span>
                    </div>
                    <div
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        marginBottom: "10px",
                        fontSize: "clamp(13px, 2vw, 14px)",
                        color: "#666",
                      }}
                    >
                      <span>Refundable Deposit</span>
                      <span style={{ fontWeight: "600" }}>₹{depositTotal}</span>
                    </div>
                    <div
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        fontSize: "clamp(13px, 2vw, 14px)",
                        color: "#4CAF50",
                        fontWeight: "600",
                      }}
                    >
                      <span>✓ Free Delivery</span>
                      <span>₹0</span>
                    </div>
                  </div>

                  <div
                    style={{
                      display: "flex",
                      justifyContent: "space-between",
                      marginBottom: "25px",
                      fontSize: "clamp(18px, 3vw, 24px)",
                      fontWeight: "800",
                      color: "#333",
                    }}
                  >
                    <span>Total Amount</span>
                    <span style={{ color: "#FF6B9D" }}>₹{totalAmount}</span>
                  </div>

                  <div
                    style={{
                      background: "#E8F5E9",
                      padding: "12px",
                      borderRadius: "12px",
                      marginBottom: "20px",
                      fontSize: "clamp(12px, 2vw, 13px)",
                      color: "#4CAF50",
                      textAlign: "center",
                      fontWeight: "600",
                    }}
                  >
                    🎁 You'll earn {Math.floor(totalAmount / 10)} reward points!
                  </div>

                  <button
                    onClick={() => setCurrentView("checkout")}
                    disabled={user.wallet < totalAmount}
                    style={{
                      width: "100%",
                      background:
                        user.wallet >= totalAmount
                          ? "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)"
                          : "#E0E0E0",
                      color: "white",
                      border: "none",
                      padding: "18px",
                      borderRadius: "15px",
                      fontSize: "clamp(15px, 2.5vw, 18px)",
                      fontWeight: "700",
                      cursor:
                        user.wallet >= totalAmount ? "pointer" : "not-allowed",
                      boxShadow:
                        user.wallet >= totalAmount
                          ? "0 8px 25px rgba(255, 107, 157, 0.4)"
                          : "none",
                      marginBottom: "15px",
                    }}
                  >
                    {user.wallet >= totalAmount
                      ? "Proceed to Checkout →"
                      : "⚠️ Insufficient Wallet Balance"}
                  </button>

                  {user.wallet < totalAmount && (
                    <div
                      style={{
                        background: "#FFE5EC",
                        padding: "12px",
                        borderRadius: "12px",
                        fontSize: "clamp(12px, 2vw, 13px)",
                        color: "#FF6B9D",
                        textAlign: "center",
                        fontWeight: "600",
                      }}
                    >
                      Please add ₹{totalAmount - user.wallet} to your wallet
                    </div>
                  )}

                  <div
                    style={{
                      marginTop: "20px",
                      padding: "15px",
                      background: "#FFF9E6",
                      borderRadius: "12px",
                      fontSize: "clamp(11px, 2vw, 12px)",
                      color: "#666",
                      lineHeight: "1.6",
                    }}
                  >
                    <strong>💡 Note:</strong> Deposit will be refunded within
                    2-3 business days after toy return in good condition.
                  </div>
                </div>
              </div>
            )}
          </div>
        )}

        {/* CHECKOUT VIEW */}
        {currentView === "checkout" && (
          <div className="checkout-view">
            <h2
              style={{
                fontSize: "clamp(24px, 5vw, 36px)",
                color: "#333",
                marginBottom: "25px",
                fontWeight: "800",
              }}
            >
              Checkout 💳
            </h2>

            <div
              style={{
                background: "white",
                borderRadius: "25px",
                padding: "clamp(25px, 4vw, 40px)",
                border: "3px solid #FFE5EC",
                width: "100%", // ← ADD THIS LINE
                boxShadow: "0 10px 40px rgba(255, 107, 157, 0.15)",
              }}
            >
              <div
                style={{
                  textAlign: "center",
                  marginBottom: "30px",
                }}
              >
                <div
                  style={{
                    fontSize: "clamp(60px, 10vw, 80px)",
                    marginBottom: "15px",
                  }}
                >
                  🎉
                </div>
                <h3
                  style={{
                    fontSize: "clamp(20px, 4vw, 28px)",
                    color: "#333",
                    marginBottom: "10px",
                    fontWeight: "700",
                  }}
                >
                  Confirm Your Booking
                </h3>
                <p
                  style={{
                    fontSize: "clamp(13px, 2vw, 15px)",
                    color: "#666",
                  }}
                >
                  Review your order before completing payment
                </p>
              </div>

              {/* Order Summary */}
              <div
                style={{
                  background: "#FFF9E6",
                  padding: "20px",
                  borderRadius: "15px",
                  marginBottom: "25px",
                }}
              >
                <h4
                  style={{
                    fontSize: "clamp(14px, 2.5vw, 16px)",
                    color: "#333",
                    marginBottom: "15px",
                    fontWeight: "700",
                  }}
                >
                  📦 Order Summary
                </h4>
                {cart.map((item) => (
                  <div
                    key={item.bookingId}
                    style={{
                      display: "flex",
                      justifyContent: "space-between",
                      marginBottom: "10px",
                      fontSize: "clamp(12px, 2vw, 14px)",
                      color: "#666",
                    }}
                  >
                    <span>
                      {item.name} ({item.days} days)
                    </span>
                    <span style={{ fontWeight: "600" }}>
                      ₹{item.dailyRate * item.days}
                    </span>
                  </div>
                ))}
                <div
                  style={{
                    borderTop: "2px dashed #E0E0E0",
                    marginTop: "15px",
                    paddingTop: "15px",
                  }}
                >
                  <div
                    style={{
                      display: "flex",
                      justifyContent: "space-between",
                      fontSize: "clamp(15px, 3vw, 18px)",
                      fontWeight: "700",
                      color: "#333",
                      marginBottom: "8px",
                    }}
                  >
                    <span>Total Amount</span>
                    <span style={{ color: "#FF6B9D" }}>₹{totalAmount}</span>
                  </div>
                  <div
                    style={{
                      fontSize: "clamp(11px, 2vw, 12px)",
                      color: "#999",
                      textAlign: "right",
                    }}
                  >
                    (Including ₹{depositTotal} refundable deposit)
                  </div>
                </div>
              </div>

              {/* Payment Info */}
              <div
                style={{
                  background: "#E3F2FD",
                  padding: "20px",
                  borderRadius: "15px",
                  marginBottom: "25px",
                }}
              >
                <h4
                  style={{
                    fontSize: "clamp(14px, 2.5vw, 16px)",
                    color: "#333",
                    marginBottom: "15px",
                    fontWeight: "700",
                  }}
                >
                  💰 Payment Method
                </h4>
                <div
                  style={{
                    background: "white",
                    padding: "15px",
                    borderRadius: "12px",
                    display: "flex",
                    justifyContent: "space-between",
                    alignItems: "center",
                    border: "2px solid #2196F3",
                  }}
                >
                  <div>
                    <div
                      style={{
                        fontSize: "clamp(13px, 2vw, 14px)",
                        color: "#666",
                        marginBottom: "5px",
                      }}
                    >
                      ToyShare Wallet
                    </div>
                    <div
                      style={{
                        fontSize: "clamp(16px, 3vw, 20px)",
                        color: "#2196F3",
                        fontWeight: "700",
                      }}
                    >
                      ₹{user.wallet}
                    </div>
                  </div>
                  <div
                    style={{
                      fontSize: "24px",
                    }}
                  >
                    ✓
                  </div>
                </div>
              </div>

              {/* Delivery Address */}
              <div
                style={{
                  background: "#FFE5EC",
                  padding: "20px",
                  borderRadius: "15px",
                  marginBottom: "30px",
                }}
              >
                <h4
                  style={{
                    fontSize: "clamp(14px, 2.5vw, 16px)",
                    color: "#333",
                    marginBottom: "15px",
                    fontWeight: "700",
                  }}
                >
                  📍 Delivery Address
                </h4>
                <div
                  style={{
                    fontSize: "clamp(13px, 2vw, 14px)",
                    color: "#666",
                    lineHeight: "1.6",
                  }}
                >
                  <div style={{ fontWeight: "600", marginBottom: "5px" }}>
                    {user.name}
                  </div>
                  <div>123, Green Park Society</div>
                  <div>Satellite Road, Ahmedabad</div>
                  <div>Gujarat - 380015</div>
                  <div
                    style={{
                      marginTop: "8px",
                      color: "#FF6B9D",
                      fontWeight: "600",
                    }}
                  >
                    📞 +91 98765 43210
                  </div>
                </div>
              </div>

              {/* Confirm Button */}
              <button
                onClick={processCheckout}
                style={{
                  width: "100%",
                  background:
                    "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)",
                  color: "white",
                  border: "none",
                  padding: "20px",
                  borderRadius: "15px",
                  fontSize: "clamp(16px, 3vw, 20px)",
                  fontWeight: "700",
                  cursor: "pointer",
                  boxShadow: "0 8px 25px rgba(255, 107, 157, 0.4)",
                  marginBottom: "15px",
                  transition: "transform 0.2s",
                }}
                onMouseEnter={(e) => (e.target.style.transform = "scale(1.02)")}
                onMouseLeave={(e) => (e.target.style.transform = "scale(1)")}
              >
                Confirm & Pay ₹{totalAmount} 🎉
              </button>

              <button
                onClick={() => setCurrentView("cart")}
                style={{
                  width: "100%",
                  background: "transparent",
                  color: "#666",
                  border: "2px solid #E0E0E0",
                  padding: "15px",
                  borderRadius: "15px",
                  fontSize: "clamp(14px, 2.5vw, 16px)",
                  fontWeight: "600",
                  cursor: "pointer",
                }}
              >
                ← Back to Cart
              </button>
            </div>
          </div>
        )}

        {/* MY BOOKINGS VIEW */}
        {currentView === "my-bookings" && (
          <div className="bookings-view">
            <h2
              style={{
                fontSize: "clamp(24px, 5vw, 36px)",
                color: "#333",
                marginBottom: "25px",
                fontWeight: "800",
                width: "1440px", // ← ADD THIS LINE
              }}
            >
              My Bookings 📋
            </h2>

            {bookings.length === 0 ? (
              <div
                style={{
                  background: "white",
                  borderRadius: "25px",
                  padding: "clamp(40px, 6vw, 80px)",
                  textAlign: "center",
                  border: "3px solid #FFE5EC",
                  minHeight: "500px",
                  display: "flex",
                  flexDirection: "column",
                  justifyContent: "center",
                  alignItems: "center",
                }}
              >
                <div
                  style={{
                    fontSize: "clamp(80px, 12vw, 120px)",
                    marginBottom: "20px",
                  }}
                >
                  📋
                </div>
                <h3
                  style={{
                    fontSize: "clamp(20px, 4vw, 28px)",
                    color: "#333",
                    marginBottom: "15px",
                    fontWeight: "700",
                  }}
                >
                  No bookings yet
                </h3>
                <p
                  style={{
                    fontSize: "clamp(14px, 2vw, 16px)",
                    color: "#666",
                    marginBottom: "25px",
                  }}
                >
                  Start renting toys to see your bookings here!
                </p>
                <button
                  onClick={() => setCurrentView("browse")}
                  style={{
                    background:
                      "linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%)",
                    color: "white",
                    border: "none",
                    padding: "15px 35px",
                    borderRadius: "15px",
                    fontSize: "clamp(14px, 2.5vw, 16px)",
                    fontWeight: "700",
                    cursor: "pointer",
                    boxShadow: "0 8px 25px rgba(255, 107, 157, 0.4)",
                  }}
                >
                  Browse Toys 🎮
                </button>
              </div>
            ) : (
              <div>
                {bookings.map((booking) => (
                  <div
                    key={booking.bookingId}
                    style={{
                      background: "white",
                      borderRadius: "20px",
                      padding: "clamp(20px, 3vw, 25px)",
                      marginBottom: "20px",
                      border: "3px solid #FFE5EC",
                      boxShadow: "0 5px 20px rgba(255, 107, 157, 0.1)",
                    }}
                  >
                    <div
                      style={{
                        display: "flex",
                        gap: "20px",
                        flexWrap: "wrap",
                      }}
                    >
                      <div
                        style={{
                          background:
                            "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                          borderRadius: "15px",
                          padding: "20px",
                          fontSize: "clamp(40px, 6vw, 60px)",
                          flexShrink: 0,
                          width: "fit-content",
                          height: "fit-content",
                        }}
                      >
                        {booking.image}
                      </div>
                      <div style={{ flex: 1, minWidth: "200px" }}>
                        <div
                          style={{
                            display: "flex",
                            justifyContent: "space-between",
                            alignItems: "flex-start",
                            marginBottom: "10px",
                            flexWrap: "wrap",
                            gap: "10px",
                          }}
                        >
                          <div>
                            <h4
                              style={{
                                fontSize: "clamp(16px, 3vw, 20px)",
                                color: "#333",
                                marginBottom: "5px",
                                fontWeight: "700",
                              }}
                            >
                              {booking.name}
                            </h4>
                            <div
                              style={{
                                fontSize: "clamp(12px, 2vw, 13px)",
                                color: "#999",
                              }}
                            >
                              Booking ID: #{String(booking.bookingId).slice(-8)}
                            </div>
                          </div>
                          <div
                            style={{
                              background: "#4CAF50",
                              color: "white",
                              padding: "6px 15px",
                              borderRadius: "10px",
                              fontSize: "clamp(12px, 2vw, 13px)",
                              fontWeight: "600",
                            }}
                          >
                            {booking.status}
                          </div>
                        </div>

                        <div
                          style={{
                            display: "grid",
                            gridTemplateColumns:
                              "repeat(auto-fit, minmax(140px, 1fr))",
                            gap: "15px",
                            marginBottom: "15px",
                          }}
                        >
                          <div>
                            <div
                              style={{
                                fontSize: "clamp(11px, 2vw, 12px)",
                                color: "#999",
                                marginBottom: "4px",
                              }}
                            >
                              Rental Period
                            </div>
                            <div
                              style={{
                                fontSize: "clamp(13px, 2vw, 14px)",
                                color: "#333",
                                fontWeight: "600",
                              }}
                            >
                              {booking.days} days
                            </div>
                          </div>
                          <div>
                            <div
                              style={{
                                fontSize: "clamp(11px, 2vw, 12px)",
                                color: "#999",
                                marginBottom: "4px",
                              }}
                            >
                              Start Date
                            </div>
                            <div
                              style={{
                                fontSize: "clamp(13px, 2vw, 14px)",
                                color: "#333",
                                fontWeight: "600",
                              }}
                            >
                              {booking.startDate}
                            </div>
                          </div>
                          <div>
                            <div
                              style={{
                                fontSize: "clamp(11px, 2vw, 12px)",
                                color: "#999",
                                marginBottom: "4px",
                              }}
                            >
                              Return Date
                            </div>
                            <div
                              style={{
                                fontSize: "clamp(13px, 2vw, 14px)",
                                color: "#333",
                                fontWeight: "600",
                              }}
                            >
                              {booking.endDate}
                            </div>
                          </div>
                          <div>
                            <div
                              style={{
                                fontSize: "clamp(11px, 2vw, 12px)",
                                color: "#999",
                                marginBottom: "4px",
                              }}
                            >
                              Amount Paid
                            </div>
                            <div
                              style={{
                                fontSize: "clamp(13px, 2vw, 14px)",
                                color: "#FF6B9D",
                                fontWeight: "700",
                              }}
                            >
                              ₹{booking.paidAmount}
                            </div>
                          </div>
                        </div>

                        <div
                          style={{
                            background: "#FFF9E6",
                            padding: "12px",
                            borderRadius: "10px",
                            fontSize: "clamp(12px, 2vw, 13px)",
                            color: "#666",
                          }}
                        >
                          💰 Deposit Held: ₹{booking.depositPaid} (Will be
                          refunded after return)
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        )}

        {/* PROFILE VIEW */}
        {currentView === "profile" && (
          <div className="profile-view">
            <h2
              style={{
                fontSize: "clamp(24px, 5vw, 36px)",
                color: "#333",
                marginBottom: "25px",
                fontWeight: "800",
              }}
            >
              My Profile 👤
            </h2>

            <div
              style={{
                background: "white",
                borderRadius: "25px",
                padding: "clamp(25px, 4vw, 40px)",
                border: "3px solid #FFE5EC",
                width: "100%", // ← ADD THIS LINE
                boxShadow: "0 10px 40px rgba(255, 107, 157, 0.15)",
              }}
            >
              <div
                style={{
                  textAlign: "center",
                  marginBottom: "30px",
                }}
              >
                <div
                  style={{
                    background:
                      "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                    width: "clamp(80px, 15vw, 120px)",
                    height: "clamp(80px, 15vw, 120px)",
                    borderRadius: "50%",
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "center",
                    fontSize: "clamp(40px, 8vw, 60px)",
                    margin: "0 auto 20px",
                    border: "4px solid #FF6B9D",
                  }}
                >
                  👤
                </div>
                <h3
                  style={{
                    fontSize: "clamp(20px, 4vw, 28px)",
                    color: "#333",
                    marginBottom: "5px",
                    fontWeight: "700",
                  }}
                >
                  {user.name}
                </h3>
                <p
                  style={{
                    fontSize: "clamp(13px, 2vw, 15px)",
                    color: "#666",
                  }}
                >
                  {user.email}
                </p>
              </div>

              {/* Wallet */}
              <div
                style={{
                  background:
                    "linear-gradient(135deg, #FFE5EC 0%, #FFF0F5 100%)",
                  padding: "25px",
                  borderRadius: "20px",
                  marginBottom: "25px",
                  border: "2px solid #FF6B9D",
                }}
              >
                <div
                  style={{
                    display: "flex",
                    justifyContent: "space-between",
                    alignItems: "center",
                    marginBottom: "15px",
                  }}
                >
                  <div>
                    <div
                      style={{
                        fontSize: "clamp(12px, 2vw, 14px)",
                        color: "#666",
                        marginBottom: "5px",
                      }}
                    >
                      Wallet Balance
                    </div>
                    <div
                      style={{
                        fontSize: "clamp(28px, 5vw, 40px)",
                        color: "#FF6B9D",
                        fontWeight: "800",
                      }}
                    >
                      ₹{user.wallet}
                    </div>
                  </div>
                  <div
                    style={{
                      fontSize: "clamp(40px, 6vw, 50px)",
                    }}
                  >
                    💰
                  </div>
                </div>
                <button
                  style={{
                    width: "100%",
                    background: "white",
                    color: "#FF6B9D",
                    border: "2px solid #FF6B9D",
                    padding: "12px",
                    borderRadius: "12px",
                    fontSize: "clamp(13px, 2vw, 15px)",
                    fontWeight: "600",
                    cursor: "pointer",
                  }}
                >
                  + Add Money
                </button>
              </div>

              {/* Reward Points */}
              <div
                style={{
                  background: "#FFF9E6",
                  padding: "20px",
                  borderRadius: "15px",
                  marginBottom: "25px",
                  border: "2px solid #FFD700",
                }}
              >
                <div
                  style={{
                    display: "flex",
                    justifyContent: "space-between",
                    alignItems: "center",
                  }}
                >
                  <div>
                    <div
                      style={{
                        fontSize: "clamp(12px, 2vw, 14px)",
                        color: "#666",
                        marginBottom: "5px",
                      }}
                    >
                      Reward Points
                    </div>
                    <div
                      style={{
                        fontSize: "clamp(24px, 4vw, 32px)",
                        color: "#FFA500",
                        fontWeight: "700",
                      }}
                    >
                      🌟 {user.points}
                    </div>
                  </div>
                  <div
                    style={{
                      fontSize: "clamp(12px, 2vw, 13px)",
                      color: "#666",
                      textAlign: "right",
                    }}
                  >
                    Earn 1 point for
                    <br />
                    every ₹10 spent
                  </div>
                </div>
              </div>

              {/* Stats */}
              <div
                style={{
                  display: "grid",
                  gridTemplateColumns: "repeat(2, 1fr)",
                  gap: "15px",
                }}
              >
                <div
                  style={{
                    background: "#E3F2FD",
                    padding: "20px",
                    borderRadius: "15px",
                    textAlign: "center",
                  }}
                >
                  <div
                    style={{
                      fontSize: "clamp(24px, 4vw, 32px)",
                      color: "#2196F3",
                      fontWeight: "700",
                      marginBottom: "5px",
                    }}
                  >
                    {bookings.length}
                  </div>
                  <div
                    style={{
                      fontSize: "clamp(11px, 2vw, 13px)",
                      color: "#666",
                    }}
                  >
                    Total Bookings
                  </div>
                </div>
                <div
                  style={{
                    background: "#E8F5E9",
                    padding: "20px",
                    borderRadius: "15px",
                    textAlign: "center",
                  }}
                >
                  <div
                    style={{
                      fontSize: "clamp(24px, 4vw, 32px)",
                      color: "#4CAF50",
                      fontWeight: "700",
                      marginBottom: "5px",
                    }}
                  >
                    {bookings.filter((b) => b.status === "Active").length}
                  </div>
                  <div
                    style={{
                      fontSize: "clamp(11px, 2vw, 13px)",
                      color: "#666",
                    }}
                  >
                    Active Rentals
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>

      <style>{`
        @keyframes slideDown {
          from {
            opacity: 0;
            transform: translateY(-20px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }

        * {
          box-sizing: border-box;
        }

        body {
          margin: 0;
          padding: 0;
          overflow-x: hidden;
        }

        /* Full width layouts with minimum height */
        .home-view,
        .browse-view,
        .toy-detail-view,
        .cart-view,
        .bookings-view,
        .checkout-view,
        .profile-view {
          width: 1440px;
          max-width: 100%;
          min-height: calc(100vh - 200px);
        }

        /* Centered content for specific views */
        .checkout-view,
        .profile-view {
          width: 1440px;
          max-width: 100%;
        }

        .checkout-view > div,
        .profile-view > div {
          max-width: min(800px, 100%);
        }

        /* Override any centered containers - FULL WIDTH */
.checkout-view > div,
.profile-view > div {
  max-width: 100% !important;
  margin-left: 0 !important;
  margin-right: 0 !important;
}

        /* Desktop Navigation */
        @media (max-width: 768px) {
          .desktop-menu {
            display: none !important;
          }
          .mobile-menu-btn {
            display: block !important;
          }
          .desktop-only {
            display: none !important;
          }
        }

        @media (min-width: 769px) {
          .mobile-menu-btn {
            display: none !important;
          }
        }

        /* Toy Detail Grid - Desktop */
        @media (min-width: 769px) {
          .toy-detail-grid {
            grid-template-columns: 1fr 1fr !important;
          }
        }

        /* Cart Grid - Desktop */
        @media (min-width: 969px) {
          .cart-grid {
            grid-template-columns: 2fr 1fr !important;
          }
        }

        /* Tablet adjustments */
        @media (min-width: 769px) and (max-width: 968px) {
          .cart-grid {
            grid-template-columns: 1fr !important;
          }
        }

        /* Scrollbar Styling */
        ::-webkit-scrollbar {
          width: 8px;
          height: 8px;
        }

        ::-webkit-scrollbar-track {
          background: #FFE5EC;
          border-radius: 10px;
        }

        ::-webkit-scrollbar-thumb {
          background: linear-gradient(135deg, #FF6B9D 0%, #C06C84 100%);
          border-radius: 10px;
        }

        ::-webkit-scrollbar-thumb:hover {
          background: linear-gradient(135deg, #C06C84 0%, #FF6B9D 100%);
        }

        /* Smooth transitions */
        button {
          transition: all 0.3s ease;
        }

        button:hover {
          transform: translateY(-2px);
        }

        button:active {
          transform: translateY(0);
        }

        /* Container widths for better desktop experience */
        @media (min-width: 1400px) {
          .browse-view .toys-grid,
          .home-view .features-grid,
          .home-view .popular-toys-grid {
            max-width: 1600px;
            margin: 0 auto;
          }
        }

        /* Grid optimizations for different screen sizes */
        @media (min-width: 1600px) {
          .toys-grid,
          .popular-toys-grid {
            grid-template-columns: repeat(auto-fill, minmax(240px, 1fr)) !important;
          }
        }

        @media (min-width: 1200px) and (max-width: 1599px) {
          .toys-grid,
          .popular-toys-grid {
            grid-template-columns: repeat(auto-fill, minmax(260px, 1fr)) !important;
          }
        }

        @media (min-width: 969px) and (max-width: 1199px) {
          .toys-grid,
          .popular-toys-grid {
            grid-template-columns: repeat(3, 1fr) !important;
          }
        }

        @media (min-width: 769px) and (max-width: 968px) {
          .toys-grid,
          .popular-toys-grid {
            grid-template-columns: repeat(2, 1fr) !important;
          }
        }

        @media (max-width: 768px) {
          .toys-grid,
          .popular-toys-grid {
            grid-template-columns: 1fr !important;
          }
        }
      `}</style>
    </div>
  );
}
