import { useState, useEffect, useRef } from "react";

/* ─── DATA ─────────────────────────────────────────── */
const SERVICES = [
  {
    id: "grooming",
    icon: "✂️",
    emoji: "🛁",
    title: "Pet Grooming",
    tagline: "Look & Feel Amazing",
    desc: "Full spa experience — bath, trim, nail care, ear cleaning, and luxurious blow-dry for your furry companion.",
    color: "#F97316",
    bg: "from-orange-50 to-amber-50",
    border: "border-orange-200",
    badge: "bg-orange-100 text-orange-700",
    packages: [
      {
        name: "Basic Bath",
        price: 499,
        duration: "45 min",
        includes: ["Bath & Dry", "Nail Trim", "Ear Clean"],
      },
      {
        name: "Full Groom",
        price: 899,
        duration: "90 min",
        includes: [
          "Bath & Dry",
          "Haircut",
          "Nail Trim",
          "Ear Clean",
          "Bow/Bandana",
        ],
      },
      {
        name: "Luxury Spa",
        price: 1499,
        duration: "2 hrs",
        includes: [
          "All Full Groom",
          "Teeth Brushing",
          "Paw Massage",
          "Cologne",
          "Premium Shampoo",
        ],
      },
    ],
  },
  {
    id: "vet",
    icon: "🏥",
    emoji: "💉",
    title: "Veterinary Care",
    tagline: "Health First, Always",
    desc: "Expert checkups, vaccinations, dental care, and preventive treatment from certified veterinarians.",
    color: "#0D9488",
    bg: "from-teal-50 to-cyan-50",
    border: "border-teal-200",
    badge: "bg-teal-100 text-teal-700",
    packages: [
      {
        name: "Wellness Check",
        price: 699,
        duration: "30 min",
        includes: ["Physical Exam", "Weight Check", "Health Report"],
      },
      {
        name: "Vaccination",
        price: 999,
        duration: "45 min",
        includes: ["Core Vaccines", "Health Check", "Certificate"],
      },
      {
        name: "Full Health Package",
        price: 2499,
        duration: "90 min",
        includes: [
          "Full Exam",
          "All Vaccines",
          "Blood Test",
          "Dental Check",
          "Deworming",
        ],
      },
    ],
  },
  {
    id: "boarding",
    icon: "🏠",
    emoji: "🛏️",
    title: "Pet Boarding",
    tagline: "Home Away From Home",
    desc: "Safe, cozy overnight stays with 24/7 supervision, play time, and lots of cuddles.",
    color: "#E11D48",
    bg: "from-rose-50 to-pink-50",
    border: "border-rose-200",
    badge: "bg-rose-100 text-rose-700",
    packages: [
      {
        name: "Standard Stay",
        price: 799,
        duration: "Per Night",
        includes: ["Cozy Kennel", "2x Meals", "Morning Walk"],
      },
      {
        name: "Deluxe Room",
        price: 1299,
        duration: "Per Night",
        includes: ["Private Suite", "3x Meals", "2x Walks", "Playtime"],
      },
      {
        name: "VIP Suite",
        price: 1999,
        duration: "Per Night",
        includes: [
          "Luxury Suite",
          "Premium Food",
          "3x Walks",
          "TV & Music",
          "Video Updates",
        ],
      },
    ],
  },
  {
    id: "training",
    icon: "🎾",
    emoji: "🏆",
    title: "Training",
    tagline: "Smart & Well-Behaved",
    desc: "Positive-reinforcement obedience, agility, and behavior correction classes for all breeds.",
    color: "#7C3AED",
    bg: "from-violet-50 to-purple-50",
    border: "border-violet-200",
    badge: "bg-violet-100 text-violet-700",
    packages: [
      {
        name: "Beginner Class",
        price: 1499,
        duration: "4 weeks",
        includes: ["Sit/Stay/Come", "Leash Training", "Group Class"],
      },
      {
        name: "Advanced Training",
        price: 2999,
        duration: "8 weeks",
        includes: ["All Basic", "Agility", "Off-Leash", "Trick Training"],
      },
      {
        name: "1-on-1 Sessions",
        price: 999,
        duration: "Per Session",
        includes: [
          "Private Trainer",
          "Custom Plan",
          "Behavior Fix",
          "Progress Report",
        ],
      },
    ],
  },
  {
    id: "taxi",
    icon: "🚐",
    emoji: "🗺️",
    title: "Pet Taxi",
    tagline: "Safe Rides for Pets",
    desc: "Comfortable, air-conditioned pickup & drop-off for vet visits, grooming, or boarding.",
    color: "#0284C7",
    bg: "from-sky-50 to-blue-50",
    border: "border-sky-200",
    badge: "bg-sky-100 text-sky-700",
    packages: [
      {
        name: "One Way",
        price: 299,
        duration: "Per Trip",
        includes: ["AC Vehicle", "Safe Carrier", "Up to 5km"],
      },
      {
        name: "Round Trip",
        price: 499,
        duration: "Per Day",
        includes: [
          "AC Vehicle",
          "Safe Carrier",
          "Up to 10km",
          "Wait Time Included",
        ],
      },
      {
        name: "Monthly Pass",
        price: 2999,
        duration: "Per Month",
        includes: [
          "Unlimited Rides",
          "Priority Booking",
          "Up to 10km/trip",
          "SMS Updates",
        ],
      },
    ],
  },
  {
    id: "nutrition",
    icon: "🌿",
    emoji: "🥗",
    title: "Nutrition Advice",
    tagline: "Eat Well, Live Long",
    desc: "Custom diet plans, supplement guidance, and weight management tailored for your pet.",
    color: "#16A34A",
    bg: "from-lime-50 to-green-50",
    border: "border-green-200",
    badge: "bg-green-100 text-green-700",
    packages: [
      {
        name: "Diet Consultation",
        price: 499,
        duration: "45 min",
        includes: ["Health Assessment", "Diet Plan", "Food List"],
      },
      {
        name: "Full Nutrition Plan",
        price: 999,
        duration: "1 month",
        includes: ["Custom Meal Plan", "Supplement Guide", "Weekly Check-in"],
      },
      {
        name: "Ongoing Support",
        price: 2499,
        duration: "3 months",
        includes: [
          "Monthly Plans",
          "Weight Tracking",
          "WhatsApp Support",
          "Recipe Book",
        ],
      },
    ],
  },
];

const TESTIMONIALS = [
  {
    name: "Priya Shah",
    pet: "Golden Retriever Mom",
    text: "PawHaven completely transformed how I care for Bruno. The groomers are incredibly gentle and he actually gets excited to visit now!",
    rating: 5,
    avatar: "🐶",
    location: "Ahmedabad",
  },
  {
    name: "Rahul Mehta",
    pet: "Cat Parent",
    text: "The vets here are so patient with my anxious cat Luna. The boarding suite kept her calm and I received daily video updates!",
    rating: 5,
    avatar: "🐱",
    location: "Surat",
  },
  {
    name: "Meera Patel",
    pet: "Bunny Owner",
    text: "I was nervous about leaving my rabbit, but the team treated him like royalty. The nutrition plan has made a huge difference!",
    rating: 5,
    avatar: "🐰",
    location: "Vadodara",
  },
  {
    name: "Arjun Desai",
    pet: "Dog Trainer Client",
    text: "After 4 weeks of training classes, my dog Max listens perfectly. The positive-only approach really works!",
    rating: 5,
    avatar: "🐕",
    location: "Rajkot",
  },
];

const TEAM = [
  {
    name: "Dr. Amelia Chase",
    role: "Chief Veterinarian",
    exp: "15 yrs",
    emoji: "👩‍⚕️",
    spec: "Surgery & Internal Medicine",
  },
  {
    name: "Ravi Kumar",
    role: "Head Groomer",
    exp: "10 yrs",
    emoji: "💇",
    spec: "All Breeds & Specialty Cuts",
  },
  {
    name: "Sneha Joshi",
    role: "Lead Trainer",
    exp: "8 yrs",
    emoji: "🎓",
    spec: "Behavioral & Agility",
  },
  {
    name: "Karan Shah",
    role: "Nutrition Specialist",
    exp: "7 yrs",
    emoji: "🥗",
    spec: "Canine & Feline Diets",
  },
];

const FAQS = [
  {
    q: "What pets do you care for?",
    a: "We care for dogs, cats, rabbits, birds, guinea pigs, and most small animals. Contact us for exotic pets.",
  },
  {
    q: "How do I book an appointment?",
    a: "Browse our services, select a package, fill in your details, and submit. We confirm within 2 hours via WhatsApp/email.",
  },
  {
    q: "Do you offer home visits?",
    a: "Yes! Our Pet Taxi and select grooming/vet services are available as home visits. Select 'Home Visit' during booking.",
  },
  {
    q: "What is your cancellation policy?",
    a: "Free cancellation up to 24 hours before. Late cancellations may incur a 20% fee. Emergency exceptions always considered.",
  },
  {
    q: "Are your vets certified?",
    a: "All our veterinarians are BVSc & AH certified with a minimum of 5 years clinical experience.",
  },
];

const GALLERY = [
  "🐶",
  "🐱",
  "🐰",
  "🐦",
  "🐹",
  "🐕",
  "🐈",
  "🦜",
  "🐩",
  "🐾",
  "🦮",
  "🐈‍⬛",
];

/* ─── UTILS ──────────────────────────────────────────── */
function Toast({ msg, type, onClose }) {
  useEffect(() => {
    const t = setTimeout(onClose, 3500);
    return () => clearTimeout(t);
  }, []);
  return (
    <div
      className={`fixed top-6 right-4 left-4 md:left-auto md:w-96 z-[999] px-5 py-4 rounded-2xl shadow-2xl flex items-start gap-3 animate-bounce-in ${type === "success" ? "bg-green-500" : "bg-red-500"} text-white`}
      style={{ animation: "slideIn 0.4s ease" }}
    >
      <span className="text-2xl">{type === "success" ? "✅" : "❌"}</span>
      <div className="flex-1">
        <div className="font-bold text-sm mb-0.5">
          {type === "success" ? "Booking Confirmed!" : "Error"}
        </div>
        <div className="text-xs opacity-90">{msg}</div>
      </div>
      <button
        onClick={onClose}
        className="opacity-70 hover:opacity-100 text-lg leading-none"
      >
        ✕
      </button>
    </div>
  );
}

function StarRating({ rating }) {
  return (
    <div className="flex gap-0.5">
      {Array.from({ length: 5 }).map((_, i) => (
        <span
          key={i}
          className={i < rating ? "text-yellow-400" : "text-gray-200"}
        >
          ★
        </span>
      ))}
    </div>
  );
}

/* ─── PAGES ──────────────────────────────────────────── */

// HOME PAGE
function HomePage({ onNavigate, onBookService }) {
  const [heroAnim, setHeroAnim] = useState(false);
  useEffect(() => {
    setTimeout(() => setHeroAnim(true), 100);
  }, []);
  const STATS = [
    { v: "5,000+", l: "Happy Pets", ic: "🐾" },
    { v: "12+", l: "Years Experience", ic: "🏆" },
    { v: "98%", l: "Satisfaction Rate", ic: "⭐" },
    { v: "30+", l: "Expert Staff", ic: "👩‍⚕️" },
  ];
  return (
    <div>
      {/* HERO */}
      <section
        className="relative min-h-screen flex items-center justify-center overflow-hidden"
        style={{
          background:
            "linear-gradient(135deg, #fff7ed 0%, #fdf4e3 50%, #fff1f2 100%)",
        }}
      >
        <div className="absolute inset-0 overflow-hidden pointer-events-none">
          {["🐾", "🐶", "🐱", "🐾", "🐰", "🐾", "🦮", "🐈"].map((e, i) => (
            <div
              key={i}
              className="absolute text-3xl opacity-10 select-none animate-float"
              style={{
                left: `${10 + i * 12}%`,
                top: `${15 + (i % 3) * 25}%`,
                animationDelay: `${i * 0.5}s`,
                animationDuration: `${3 + i * 0.4}s`,
              }}
            >
              {e}
            </div>
          ))}
        </div>
        <div className="relative z-10 max-w-6xl mx-auto px-4 sm:px-6 py-24 pt-32 grid md:grid-cols-2 gap-10 md:gap-16 items-center">
          <div
            className={`transition-all duration-700 ${heroAnim ? "opacity-100 translate-x-0" : "opacity-0 -translate-x-8"}`}
          >
            <div className="inline-flex items-center gap-2 bg-orange-100 text-orange-700 px-4 py-2 rounded-full text-sm font-semibold mb-5 shadow-sm">
              <span>🌟</span> Gujarat's #1 Trusted Pet Care
            </div>
            <h1 className="font-display text-4xl sm:text-5xl md:text-6xl font-black leading-tight text-gray-900 mb-5">
              Your Pet's
              <br />
              <span style={{ color: "#F97316" }}>Happiness</span>
              <br />
              is Our Mission
            </h1>
            <p className="text-gray-600 text-base sm:text-lg leading-relaxed mb-8 max-w-md font-body">
              Premium grooming, expert vet care, cozy boarding, and professional
              training — all under one roof. Because your furry family deserves
              the absolute best.
            </p>
            <div className="flex flex-wrap gap-3">
              <button
                onClick={() => onNavigate("services")}
                className="bg-orange-500 hover:bg-orange-600 text-white px-6 sm:px-8 py-3.5 rounded-full font-bold shadow-xl hover:shadow-orange-200 transition-all hover:-translate-y-0.5 text-sm sm:text-base"
              >
                Explore Services →
              </button>
              <button
                onClick={() => onNavigate("booking")}
                className="bg-white border-2 border-gray-200 hover:border-orange-300 text-gray-800 px-6 sm:px-8 py-3.5 rounded-full font-bold transition-all hover:-translate-y-0.5 text-sm sm:text-base"
              >
                Book Now 📅
              </button>
            </div>
          </div>
          <div
            className={`transition-all duration-700 delay-300 ${heroAnim ? "opacity-100 translate-x-0" : "opacity-0 translate-x-8"} flex justify-center`}
          >
            <div className="relative w-64 h-64 sm:w-80 sm:h-80 md:w-96 md:h-96">
              <div className="absolute inset-0 bg-orange-200 rounded-full opacity-30 animate-pulse" />
              <div className="absolute inset-4 sm:inset-6 bg-gradient-to-br from-orange-100 to-amber-100 rounded-full flex items-center justify-center text-8xl sm:text-[110px] md:text-[140px] shadow-2xl animate-float">
                🐶
              </div>
              <div className="absolute -top-2 -right-2 sm:-top-4 sm:-right-4 bg-white rounded-2xl shadow-xl px-3 sm:px-4 py-2 sm:py-3 text-xs sm:text-sm font-bold flex items-center gap-1 sm:gap-2">
                ⭐ 4.9 <span className="text-gray-400 font-normal">Rating</span>
              </div>
              <div className="absolute -bottom-2 -left-2 sm:-bottom-4 sm:-left-4 bg-orange-500 text-white rounded-2xl shadow-xl px-3 sm:px-4 py-2 sm:py-3 text-xs sm:text-sm font-bold">
                🐾 5,000+ Happy Pets
              </div>
            </div>
          </div>
        </div>
        <div className="absolute bottom-0 left-0 right-0">
          <svg viewBox="0 0 1440 60" className="w-full" fill="white">
            <path d="M0 30 Q360 60 720 30 Q1080 0 1440 30 L1440 60 L0 60 Z" />
          </svg>
        </div>
      </section>

      {/* STATS */}
      <section className="bg-white py-10 sm:py-14">
        <div className="max-w-5xl mx-auto px-4 sm:px-6 grid grid-cols-2 md:grid-cols-4 gap-6 text-center">
          {STATS.map((s) => (
            <div key={s.l} className="p-4">
              <div className="text-3xl mb-2">{s.ic}</div>
              <div
                className="font-display text-3xl sm:text-4xl font-black"
                style={{ color: "#F97316" }}
              >
                {s.v}
              </div>
              <div className="text-gray-500 text-xs sm:text-sm mt-1 font-body">
                {s.l}
              </div>
            </div>
          ))}
        </div>
      </section>

      {/* SERVICES PREVIEW */}
      <section className="py-16 sm:py-24" style={{ background: "#fdfaf6" }}>
        <div className="max-w-6xl mx-auto px-4 sm:px-6">
          <div className="text-center mb-10 sm:mb-16">
            <span className="text-orange-500 font-bold text-xs sm:text-sm uppercase tracking-widest">
              What We Offer
            </span>
            <h2 className="font-display text-3xl sm:text-4xl md:text-5xl font-bold mt-2 mb-3">
              Our Services
            </h2>
            <p className="text-gray-500 max-w-xl mx-auto text-sm sm:text-base font-body">
              Every service crafted with love, expertise, and your pet's comfort
              at heart.
            </p>
          </div>
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 sm:gap-6">
            {SERVICES.map((s) => (
              <div
                key={s.id}
                onClick={() => onNavigate("services")}
                className={`cursor-pointer bg-gradient-to-br ${s.bg} rounded-3xl p-5 sm:p-7 border ${s.border} hover:-translate-y-2 hover:shadow-xl transition-all duration-300 group`}
              >
                <div className="flex items-start justify-between mb-3 sm:mb-4">
                  <div className="text-3xl sm:text-4xl">{s.icon}</div>
                  <span
                    className={`${s.badge} text-xs font-bold px-2 sm:px-3 py-1 rounded-full`}
                  >
                    {s.tagline}
                  </span>
                </div>
                <h3
                  className="font-display text-lg sm:text-xl font-bold mb-2"
                  style={{ color: s.color }}
                >
                  {s.title}
                </h3>
                <p className="text-gray-600 text-xs sm:text-sm leading-relaxed mb-3 sm:mb-5 font-body">
                  {s.desc}
                </p>
                <div className="flex items-center justify-between">
                  <span className="text-xs text-gray-400 font-body">
                    From ₹{s.packages[0].price}
                  </span>
                  <button
                    className="text-xs sm:text-sm font-bold flex items-center gap-1 group-hover:gap-2 transition-all"
                    style={{ color: s.color }}
                  >
                    Book Now →
                  </button>
                </div>
              </div>
            ))}
          </div>
          <div className="text-center mt-8 sm:mt-10">
            <button
              onClick={() => onNavigate("services")}
              className="bg-orange-500 hover:bg-orange-600 text-white px-8 py-3 rounded-full font-bold shadow-lg transition-all"
            >
              View All Services & Packages
            </button>
          </div>
        </div>
      </section>

      {/* TESTIMONIALS */}
      <section className="py-16 sm:py-24 bg-white">
        <div className="max-w-6xl mx-auto px-4 sm:px-6">
          <div className="text-center mb-10 sm:mb-14">
            <span className="text-orange-500 font-bold text-xs sm:text-sm uppercase tracking-widest">
              Reviews
            </span>
            <h2 className="font-display text-3xl sm:text-4xl md:text-5xl font-bold mt-2">
              Happy Pet Parents
            </h2>
          </div>
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-5">
            {TESTIMONIALS.map((t) => (
              <div
                key={t.name}
                className="bg-gray-50 rounded-3xl p-5 sm:p-6 hover:-translate-y-1 hover:shadow-lg transition-all border border-gray-100"
              >
                <StarRating rating={t.rating} />
                <p className="text-gray-600 text-xs sm:text-sm italic leading-relaxed my-3 sm:my-4 font-body">
                  "{t.text}"
                </p>
                <div className="flex items-center gap-2 sm:gap-3 pt-3 border-t border-gray-100">
                  <div className="w-9 h-9 sm:w-10 sm:h-10 bg-orange-100 rounded-full flex items-center justify-center text-lg sm:text-xl">
                    {t.avatar}
                  </div>
                  <div>
                    <div className="font-bold text-xs sm:text-sm">{t.name}</div>
                    <div className="text-xs text-gray-400 font-body">
                      {t.pet} · {t.location}
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CTA BAND */}
      <section
        className="py-14 sm:py-20"
        style={{ background: "linear-gradient(135deg, #F97316, #ea580c)" }}
      >
        <div className="max-w-3xl mx-auto px-4 sm:px-6 text-center text-white">
          <div className="text-4xl sm:text-5xl mb-4">🐾</div>
          <h2 className="font-display text-3xl sm:text-4xl md:text-5xl font-black mb-4">
            Ready to Book?
          </h2>
          <p className="text-orange-100 mb-6 sm:mb-8 text-sm sm:text-base font-body">
            Join 5,000+ happy pet families. Book your first appointment today!
          </p>
          <button
            onClick={() => onNavigate("booking")}
            className="bg-white text-orange-600 px-8 sm:px-10 py-3 sm:py-4 rounded-full font-black text-base sm:text-lg hover:bg-orange-50 shadow-2xl transition-all hover:-translate-y-1"
          >
            Book an Appointment 🐶
          </button>
        </div>
      </section>
    </div>
  );
}

// SERVICES PAGE
function ServicesPage({ onBookService }) {
  const [selected, setSelected] = useState(null);
  return (
    <div className="min-h-screen" style={{ background: "#fdfaf6" }}>
      <div className="max-w-6xl mx-auto px-4 sm:px-6 py-24 pt-28 sm:pt-32">
        <div className="text-center mb-10 sm:mb-16">
          <span className="text-orange-500 font-bold text-xs sm:text-sm uppercase tracking-widest">
            All Services
          </span>
          <h1 className="font-display text-3xl sm:text-4xl md:text-5xl font-black mt-2 mb-3">
            Choose Your Service
          </h1>
          <p className="text-gray-500 max-w-xl mx-auto text-sm sm:text-base font-body">
            Click any service to see packages and pricing, then book instantly.
          </p>
        </div>
        <div className="space-y-4 sm:space-y-6">
          {SERVICES.map((s) => (
            <div
              key={s.id}
              className={`bg-white rounded-3xl border-2 overflow-hidden transition-all duration-300 ${selected === s.id ? s.border : "border-gray-100 hover:border-gray-200"} shadow-sm hover:shadow-md`}
            >
              <button
                className="w-full px-5 sm:px-8 py-5 sm:py-6 flex items-center gap-4 sm:gap-5 text-left"
                onClick={() => setSelected(selected === s.id ? null : s.id)}
              >
                <div
                  className={`w-12 h-12 sm:w-16 sm:h-16 rounded-2xl bg-gradient-to-br ${s.bg} flex items-center justify-center text-2xl sm:text-3xl border ${s.border} flex-shrink-0`}
                >
                  {s.icon}
                </div>
                <div className="flex-1 min-w-0">
                  <div className="flex flex-wrap items-center gap-2 mb-1">
                    <h3
                      className="font-display text-lg sm:text-2xl font-bold"
                      style={{ color: s.color }}
                    >
                      {s.title}
                    </h3>
                    <span
                      className={`${s.badge} text-xs font-bold px-2 sm:px-3 py-1 rounded-full hidden sm:inline`}
                    >
                      {s.tagline}
                    </span>
                  </div>
                  <p className="text-gray-500 text-xs sm:text-sm font-body line-clamp-1 sm:line-clamp-none">
                    {s.desc}
                  </p>
                </div>
                <div className="text-right flex-shrink-0 ml-2">
                  <div className="text-xs text-gray-400 font-body">From</div>
                  <div
                    className="font-black text-base sm:text-xl"
                    style={{ color: s.color }}
                  >
                    ₹{s.packages[0].price}
                  </div>
                  <div
                    className="text-lg sm:text-2xl mt-1 transition-transform duration-300"
                    style={{
                      transform:
                        selected === s.id ? "rotate(180deg)" : "rotate(0)",
                    }}
                  >
                    ⌄
                  </div>
                </div>
              </button>
              {selected === s.id && (
                <div
                  className={`px-4 sm:px-8 pb-6 sm:pb-8 border-t ${s.border} bg-gradient-to-br ${s.bg}`}
                >
                  <p className="text-gray-600 text-xs sm:text-sm mt-4 sm:mt-5 mb-4 sm:mb-6 font-body">
                    {s.desc}
                  </p>
                  <div className="grid grid-cols-1 sm:grid-cols-3 gap-3 sm:gap-4">
                    {s.packages.map((pkg) => (
                      <div
                        key={pkg.name}
                        className="bg-white rounded-2xl p-4 sm:p-5 border border-white shadow-sm hover:shadow-md transition-all"
                      >
                        <div className="flex justify-between items-start mb-3">
                          <h4 className="font-bold text-sm sm:text-base text-gray-800">
                            {pkg.name}
                          </h4>
                          <span className="text-xs text-gray-400 font-body bg-gray-100 px-2 py-0.5 rounded-full">
                            {pkg.duration}
                          </span>
                        </div>
                        <div
                          className="font-black text-xl sm:text-2xl mb-3 sm:mb-4"
                          style={{ color: s.color }}
                        >
                          ₹{pkg.price}
                        </div>
                        <ul className="space-y-1 sm:space-y-1.5 mb-4 sm:mb-5">
                          {pkg.includes.map((inc) => (
                            <li
                              key={inc}
                              className="flex items-center gap-2 text-xs sm:text-sm text-gray-600 font-body"
                            >
                              <span style={{ color: s.color }}>✓</span> {inc}
                            </li>
                          ))}
                        </ul>
                        <button
                          onClick={() => onBookService(s, pkg)}
                          className="w-full py-2 sm:py-2.5 rounded-xl font-bold text-xs sm:text-sm text-white transition-all hover:-translate-y-0.5 hover:shadow-lg"
                          style={{ background: s.color }}
                        >
                          Book This Package
                        </button>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}

// BOOKING PAGE
function BookingPage({ preSelected, onConfirm }) {
  const [step, setStep] = useState(1);
  const [selectedService, setSelectedService] = useState(
    preSelected?.service || null,
  );
  const [selectedPkg, setSelectedPkg] = useState(preSelected?.pkg || null);
  const [form, setForm] = useState({
    name: "",
    phone: "",
    email: "",
    petName: "",
    petType: "Dog",
    petAge: "",
    petWeight: "",
    date: "",
    time: "",
    notes: "",
    homeVisit: false,
  });
  const [errors, setErrors] = useState({});

  useEffect(() => {
    if (preSelected) {
      setSelectedService(preSelected.service);
      setSelectedPkg(preSelected.pkg);
      setStep(2);
    }
  }, [preSelected]);

  const times = [
    "09:00 AM",
    "10:00 AM",
    "11:00 AM",
    "12:00 PM",
    "02:00 PM",
    "03:00 PM",
    "04:00 PM",
    "05:00 PM",
    "06:00 PM",
  ];

  const validate = () => {
    const e = {};
    if (!form.name.trim()) e.name = "Name required";
    if (!form.phone.match(/^\d{10}$/))
      e.phone = "Valid 10-digit phone required";
    if (!form.email.match(/\S+@\S+\.\S+/)) e.email = "Valid email required";
    if (!form.petName.trim()) e.petName = "Pet name required";
    if (!form.date) e.date = "Date required";
    if (!form.time) e.time = "Time required";
    setErrors(e);
    return Object.keys(e).length === 0;
  };

  const handleSubmit = () => {
    if (validate()) {
      onConfirm({ service: selectedService, pkg: selectedPkg, form });
      setStep(4);
    }
  };

  const inp =
    "w-full bg-gray-50 border-2 border-gray-200 rounded-xl px-4 py-3 text-sm focus:border-orange-400 focus:bg-white outline-none transition-all font-body";
  const lbl =
    "block text-xs font-bold text-gray-600 mb-1.5 uppercase tracking-wide";
  const err = "text-xs text-red-500 mt-1 font-body";

  const today = new Date().toISOString().split("T")[0];

  return (
    <div className="min-h-screen bg-gray-50 pt-20 sm:pt-24 pb-16">
      <div className="max-w-3xl mx-auto px-4 sm:px-6">
        {/* Header */}
        <div className="text-center mb-8">
          <h1 className="font-display text-3xl sm:text-4xl font-black text-gray-900">
            Book an Appointment
          </h1>
          <p className="text-gray-500 mt-2 text-sm sm:text-base font-body">
            Quick & easy — takes less than 2 minutes!
          </p>
        </div>

        {/* Progress */}
        {step < 4 && (
          <div className="flex items-center justify-center gap-2 sm:gap-3 mb-8 sm:mb-10">
            {[1, 2, 3].map((s) => (
              <div key={s} className="flex items-center gap-2 sm:gap-3">
                <div
                  className={`w-8 h-8 sm:w-10 sm:h-10 rounded-full flex items-center justify-center font-black text-xs sm:text-sm transition-all ${step >= s ? "bg-orange-500 text-white shadow-lg shadow-orange-200" : "bg-gray-200 text-gray-500"}`}
                >
                  {s}
                </div>
                {s < 3 && (
                  <div
                    className={`w-8 sm:w-16 h-1 rounded-full transition-all ${step > s ? "bg-orange-500" : "bg-gray-200"}`}
                  />
                )}
              </div>
            ))}
          </div>
        )}
        <div className="flex justify-center gap-4 sm:gap-16 mb-8 text-xs font-bold text-gray-400 uppercase tracking-wider">
          <span className={step >= 1 ? "text-orange-500" : ""}>Service</span>
          <span className={step >= 2 ? "text-orange-500" : ""}>Details</span>
          <span className={step >= 3 ? "text-orange-500" : ""}>Confirm</span>
        </div>

        {/* STEP 1: Choose Service */}
        {step === 1 && (
          <div className="bg-white rounded-3xl shadow-sm border border-gray-100 p-5 sm:p-8">
            <h2 className="font-display text-xl sm:text-2xl font-bold mb-5 sm:mb-6">
              Choose a Service
            </h2>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
              {SERVICES.map((s) => (
                <button
                  key={s.id}
                  onClick={() => {
                    setSelectedService(s);
                    setSelectedPkg(null);
                  }}
                  className={`text-left p-4 rounded-2xl border-2 transition-all ${selectedService?.id === s.id ? `${s.border} bg-gradient-to-br ${s.bg}` : "border-gray-100 hover:border-gray-200 bg-gray-50"}`}
                >
                  <div className="flex items-center gap-3">
                    <span className="text-2xl">{s.icon}</span>
                    <div>
                      <div
                        className="font-bold text-sm"
                        style={{
                          color:
                            selectedService?.id === s.id ? s.color : "#374151",
                        }}
                      >
                        {s.title}
                      </div>
                      <div className="text-xs text-gray-400 font-body">
                        From ₹{s.packages[0].price}
                      </div>
                    </div>
                  </div>
                </button>
              ))}
            </div>
            {selectedService && (
              <div className="mt-5 sm:mt-6">
                <h3 className="font-bold text-sm mb-3 text-gray-700">
                  Choose Package
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-3 gap-3">
                  {selectedService.packages.map((pkg) => (
                    <button
                      key={pkg.name}
                      onClick={() => setSelectedPkg(pkg)}
                      className={`p-3 sm:p-4 rounded-xl border-2 text-left transition-all ${selectedPkg?.name === pkg.name ? `border-2 shadow-md` : "border-gray-100 hover:border-gray-200"}`}
                      style={
                        selectedPkg?.name === pkg.name
                          ? {
                              borderColor: selectedService.color,
                              background: `${selectedService.color}10`,
                            }
                          : {}
                      }
                    >
                      <div className="font-bold text-xs sm:text-sm text-gray-800">
                        {pkg.name}
                      </div>
                      <div
                        className="font-black text-base sm:text-lg mt-1"
                        style={{ color: selectedService.color }}
                      >
                        ₹{pkg.price}
                      </div>
                      <div className="text-xs text-gray-400 font-body">
                        {pkg.duration}
                      </div>
                    </button>
                  ))}
                </div>
              </div>
            )}
            <button
              onClick={() => selectedService && selectedPkg && setStep(2)}
              disabled={!selectedService || !selectedPkg}
              className={`w-full mt-6 py-3 sm:py-4 rounded-2xl font-black text-white transition-all text-sm sm:text-base ${selectedService && selectedPkg ? "bg-orange-500 hover:bg-orange-600 hover:-translate-y-0.5 shadow-lg" : "bg-gray-200 cursor-not-allowed text-gray-400"}`}
            >
              Continue →
            </button>
          </div>
        )}

        {/* STEP 2: Details */}
        {step === 2 && (
          <div className="bg-white rounded-3xl shadow-sm border border-gray-100 p-5 sm:p-8">
            <div className="flex items-center justify-between mb-5 sm:mb-6">
              <h2 className="font-display text-xl sm:text-2xl font-bold">
                Your Details
              </h2>
              <button
                onClick={() => setStep(1)}
                className="text-xs text-orange-500 font-bold"
              >
                ← Change Service
              </button>
            </div>
            {selectedService && selectedPkg && (
              <div
                className={`bg-gradient-to-br ${selectedService.bg} border ${selectedService.border} rounded-2xl p-3 sm:p-4 mb-5 sm:mb-6 flex items-center gap-3`}
              >
                <span className="text-2xl">{selectedService.icon}</span>
                <div>
                  <div
                    className="font-bold text-sm"
                    style={{ color: selectedService.color }}
                  >
                    {selectedService.title} — {selectedPkg.name}
                  </div>
                  <div className="text-xs text-gray-500 font-body">
                    ₹{selectedPkg.price} · {selectedPkg.duration}
                  </div>
                </div>
              </div>
            )}
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
              <div>
                <label className={lbl}>Your Name *</label>
                <input
                  className={inp}
                  placeholder="e.g. Rahul Shah"
                  value={form.name}
                  onChange={(e) => setForm({ ...form, name: e.target.value })}
                />
                {errors.name && <p className={err}>{errors.name}</p>}
              </div>
              <div>
                <label className={lbl}>Phone *</label>
                <input
                  className={inp}
                  placeholder="10-digit number"
                  value={form.phone}
                  onChange={(e) => setForm({ ...form, phone: e.target.value })}
                  maxLength={10}
                />
                {errors.phone && <p className={err}>{errors.phone}</p>}
              </div>
              <div className="sm:col-span-2">
                <label className={lbl}>Email *</label>
                <input
                  className={inp}
                  placeholder="you@example.com"
                  value={form.email}
                  onChange={(e) => setForm({ ...form, email: e.target.value })}
                  type="email"
                />
                {errors.email && <p className={err}>{errors.email}</p>}
              </div>
              <div>
                <label className={lbl}>Pet's Name *</label>
                <input
                  className={inp}
                  placeholder="e.g. Bruno"
                  value={form.petName}
                  onChange={(e) =>
                    setForm({ ...form, petName: e.target.value })
                  }
                />
                {errors.petName && <p className={err}>{errors.petName}</p>}
              </div>
              <div>
                <label className={lbl}>Pet Type</label>
                <select
                  className={inp}
                  value={form.petType}
                  onChange={(e) =>
                    setForm({ ...form, petType: e.target.value })
                  }
                >
                  {["Dog", "Cat", "Rabbit", "Bird", "Guinea Pig", "Other"].map(
                    (p) => (
                      <option key={p}>{p}</option>
                    ),
                  )}
                </select>
              </div>
              <div>
                <label className={lbl}>Pet's Age</label>
                <input
                  className={inp}
                  placeholder="e.g. 2 years"
                  value={form.petAge}
                  onChange={(e) => setForm({ ...form, petAge: e.target.value })}
                />
              </div>
              <div>
                <label className={lbl}>Pet's Weight (kg)</label>
                <input
                  className={inp}
                  placeholder="e.g. 15"
                  value={form.petWeight}
                  onChange={(e) =>
                    setForm({ ...form, petWeight: e.target.value })
                  }
                  type="number"
                />
              </div>
              <div>
                <label className={lbl}>Preferred Date *</label>
                <input
                  className={inp}
                  type="date"
                  min={today}
                  value={form.date}
                  onChange={(e) => setForm({ ...form, date: e.target.value })}
                />
                {errors.date && <p className={err}>{errors.date}</p>}
              </div>
              <div>
                <label className={lbl}>Preferred Time *</label>
                <select
                  className={inp}
                  value={form.time}
                  onChange={(e) => setForm({ ...form, time: e.target.value })}
                >
                  <option value="">Select time</option>
                  {times.map((t) => (
                    <option key={t}>{t}</option>
                  ))}
                </select>
                {errors.time && <p className={err}>{errors.time}</p>}
              </div>
              <div className="sm:col-span-2">
                <label className={lbl}>Special Notes</label>
                <textarea
                  className={inp}
                  rows={3}
                  placeholder="Any medical conditions, allergies, or special requests..."
                  value={form.notes}
                  onChange={(e) => setForm({ ...form, notes: e.target.value })}
                />
              </div>
              <div className="sm:col-span-2">
                <label className="flex items-center gap-3 cursor-pointer">
                  <input
                    type="checkbox"
                    checked={form.homeVisit}
                    onChange={(e) =>
                      setForm({ ...form, homeVisit: e.target.checked })
                    }
                    className="w-4 h-4 accent-orange-500"
                  />
                  <span className="text-sm font-body text-gray-700">
                    🏠 I'd like a Home Visit (+₹200)
                  </span>
                </label>
              </div>
            </div>
            <div className="flex gap-3 mt-6">
              <button
                onClick={() => setStep(1)}
                className="flex-1 py-3 rounded-2xl border-2 border-gray-200 font-bold text-gray-600 hover:bg-gray-50 transition-all text-sm"
              >
                ← Back
              </button>
              <button
                onClick={() => {
                  if (validate()) setStep(3);
                }}
                className="flex-[2] py-3 rounded-2xl bg-orange-500 hover:bg-orange-600 text-white font-black shadow-lg transition-all hover:-translate-y-0.5 text-sm sm:text-base"
              >
                Review Booking →
              </button>
            </div>
          </div>
        )}

        {/* STEP 3: Confirm */}
        {step === 3 && (
          <div className="bg-white rounded-3xl shadow-sm border border-gray-100 p-5 sm:p-8">
            <h2 className="font-display text-xl sm:text-2xl font-bold mb-5 sm:mb-6">
              Review & Confirm
            </h2>
            <div
              className={`bg-gradient-to-br ${selectedService?.bg} border ${selectedService?.border} rounded-2xl p-4 sm:p-5 mb-4 sm:mb-5`}
            >
              <div className="flex items-center gap-3 mb-3">
                <span className="text-3xl">{selectedService?.icon}</span>
                <div>
                  <div
                    className="font-black text-base sm:text-lg"
                    style={{ color: selectedService?.color }}
                  >
                    {selectedService?.title}
                  </div>
                  <div className="text-xs sm:text-sm text-gray-500 font-body">
                    {selectedPkg?.name} · {selectedPkg?.duration}
                  </div>
                </div>
              </div>
              <div className="text-xs text-gray-600 mb-2 font-body font-bold">
                Includes:
              </div>
              <div className="flex flex-wrap gap-1 sm:gap-2">
                {selectedPkg?.includes.map((inc) => (
                  <span
                    key={inc}
                    className="bg-white/70 text-gray-700 text-xs px-2 sm:px-3 py-1 rounded-full font-body"
                  >
                    ✓ {inc}
                  </span>
                ))}
              </div>
            </div>
            <div className="grid grid-cols-2 gap-3 sm:gap-4 mb-4 sm:mb-5">
              {[
                { l: "Your Name", v: form.name },
                { l: "Phone", v: form.phone },
                { l: "Email", v: form.email },
                { l: "Pet", v: `${form.petName} (${form.petType})` },
                { l: "Date", v: form.date },
                { l: "Time", v: form.time },
                { l: "Pet Age", v: form.petAge || "—" },
                {
                  l: "Pet Weight",
                  v: form.petWeight ? `${form.petWeight} kg` : "—",
                },
              ].map((r) => (
                <div key={r.l} className="bg-gray-50 rounded-xl p-3">
                  <div className="text-xs text-gray-400 font-body mb-0.5">
                    {r.l}
                  </div>
                  <div className="font-bold text-xs sm:text-sm text-gray-800 break-words">
                    {r.v}
                  </div>
                </div>
              ))}
            </div>
            {form.notes && (
              <div className="bg-yellow-50 border border-yellow-200 rounded-xl p-3 mb-4 sm:mb-5">
                <div className="text-xs text-yellow-700 font-bold mb-1">
                  Notes:
                </div>
                <div className="text-xs text-gray-600 font-body">
                  {form.notes}
                </div>
              </div>
            )}
            <div className="bg-gray-50 rounded-2xl p-3 sm:p-4 mb-5 sm:mb-6 flex items-center justify-between">
              <div>
                <div className="text-xs text-gray-400 font-body">
                  Total Amount
                </div>
                <div className="font-black text-2xl sm:text-3xl text-gray-900">
                  ₹{(selectedPkg?.price || 0) + (form.homeVisit ? 200 : 0)}
                </div>
                {form.homeVisit && (
                  <div className="text-xs text-gray-400 font-body">
                    Includes ₹200 home visit
                  </div>
                )}
              </div>
              <div className="text-right">
                <div className="text-xs text-gray-400 font-body mb-1">
                  Payment
                </div>
                <div className="bg-green-100 text-green-700 text-xs font-bold px-3 py-1 rounded-full">
                  Pay at Center
                </div>
              </div>
            </div>
            <div className="flex gap-3">
              <button
                onClick={() => setStep(2)}
                className="flex-1 py-3 rounded-2xl border-2 border-gray-200 font-bold text-gray-600 hover:bg-gray-50 transition-all text-sm"
              >
                ← Edit
              </button>
              <button
                onClick={handleSubmit}
                className="flex-[2] py-3 sm:py-4 rounded-2xl bg-orange-500 hover:bg-orange-600 text-white font-black shadow-xl transition-all hover:-translate-y-0.5 text-sm sm:text-base"
              >
                Confirm Booking 🐾
              </button>
            </div>
          </div>
        )}

        {/* STEP 4: Success */}
        {step === 4 && (
          <div className="bg-white rounded-3xl shadow-sm border border-gray-100 p-6 sm:p-12 text-center">
            <div className="text-6xl sm:text-8xl mb-5 sm:mb-6 animate-bounce">
              🎉
            </div>
            <h2 className="font-display text-2xl sm:text-3xl font-black text-gray-900 mb-3">
              Booking Confirmed!
            </h2>
            <p className="text-gray-500 mb-4 sm:mb-6 font-body text-sm sm:text-base">
              Thank you, <strong>{form.name}</strong>! Your appointment for{" "}
              <strong>{form.petName}</strong> has been successfully booked.
            </p>
            <div
              className={`bg-gradient-to-br ${selectedService?.bg} rounded-2xl p-4 sm:p-5 mb-4 sm:mb-6 text-left`}
            >
              <div className="grid grid-cols-2 gap-3">
                {[
                  {
                    l: "Service",
                    v: `${selectedService?.title} — ${selectedPkg?.name}`,
                  },
                  { l: "Date & Time", v: `${form.date} at ${form.time}` },
                  { l: "Pet", v: `${form.petName} (${form.petType})` },
                  {
                    l: "Amount",
                    v: `₹${(selectedPkg?.price || 0) + (form.homeVisit ? 200 : 0)}`,
                  },
                ].map((r) => (
                  <div key={r.l}>
                    <div className="text-xs text-gray-400 font-body">{r.l}</div>
                    <div className="font-bold text-xs sm:text-sm text-gray-800">
                      {r.v}
                    </div>
                  </div>
                ))}
              </div>
            </div>
            <div className="bg-blue-50 border border-blue-200 rounded-2xl p-3 sm:p-4 mb-5 sm:mb-6 text-left">
              <div className="flex items-start gap-2 sm:gap-3">
                <span className="text-xl sm:text-2xl">📱</span>
                <div>
                  <div className="font-bold text-xs sm:text-sm text-blue-800">
                    What's Next?
                  </div>
                  <div className="text-xs text-blue-600 font-body mt-1">
                    You'll receive a confirmation on{" "}
                    <strong>{form.phone}</strong> and{" "}
                    <strong>{form.email}</strong>. Our team will call 1 hour
                    before your appointment.
                  </div>
                </div>
              </div>
            </div>
            <div className="bg-gray-50 rounded-2xl p-3 text-xs text-gray-500 font-body mb-5 sm:mb-6">
              📍 PawHaven · 42 Paw Street, Pet City, Ahmedabad · 📞 +91 80001
              23456
            </div>
            <button
              onClick={() => {
                setStep(1);
                setSelectedService(null);
                setSelectedPkg(null);
                setForm({
                  name: "",
                  phone: "",
                  email: "",
                  petName: "",
                  petType: "Dog",
                  petAge: "",
                  petWeight: "",
                  date: "",
                  time: "",
                  notes: "",
                  homeVisit: false,
                });
              }}
              className="bg-orange-500 hover:bg-orange-600 text-white px-8 py-3 rounded-2xl font-black transition-all shadow-lg text-sm sm:text-base"
            >
              Book Another Service 🐾
            </button>
          </div>
        )}
      </div>
    </div>
  );
}

// ABOUT PAGE
function AboutPage() {
  return (
    <div className="min-h-screen" style={{ background: "#fdfaf6" }}>
      <div className="max-w-6xl mx-auto px-4 sm:px-6 py-24 pt-28 sm:pt-32">
        <div className="text-center mb-12 sm:mb-16">
          <span className="text-orange-500 font-bold text-xs sm:text-sm uppercase tracking-widest">
            Our Story
          </span>
          <h1 className="font-display text-3xl sm:text-4xl md:text-5xl font-black mt-2 mb-4">
            About PawHaven
          </h1>
          <p className="text-gray-500 max-w-2xl mx-auto text-sm sm:text-base font-body">
            Founded with love for animals, built on trust and expertise.
          </p>
        </div>
        <div className="grid md:grid-cols-2 gap-10 sm:gap-16 items-center mb-16 sm:mb-24">
          <div className="relative">
            <div className="bg-gradient-to-br from-orange-100 to-amber-100 rounded-3xl p-8 sm:p-12 text-center text-7xl sm:text-9xl shadow-xl">
              🏥
            </div>
            <div className="absolute -top-3 -left-3 sm:-top-5 sm:-left-5 bg-orange-500 text-white rounded-2xl p-3 sm:p-5 text-3xl sm:text-4xl shadow-xl">
              🐕
            </div>
            <div className="absolute -bottom-3 -right-3 sm:-bottom-5 sm:-right-5 bg-teal-500 text-white rounded-2xl p-3 sm:p-5 text-3xl sm:text-4xl shadow-xl">
              🐈
            </div>
          </div>
          <div>
            <h2 className="font-display text-2xl sm:text-3xl md:text-4xl font-bold mb-4 sm:mb-6">
              We Treat Every Pet
              <br />
              <span style={{ color: "#F97316" }}>Like Family</span>
            </h2>
            <p className="text-gray-600 leading-relaxed mb-4 font-body text-sm sm:text-base">
              PawHaven was founded in 2012 by Dr. Amelia Chase, a passionate
              veterinarian who believed pets deserved the same quality of care
              as any family member. What started as a small clinic in Ahmedabad
              has grown into Gujarat's most trusted full-service pet care
              center.
            </p>
            <p className="text-gray-600 leading-relaxed mb-6 font-body text-sm sm:text-base">
              Our team of certified vets, licensed groomers, and professional
              trainers combine scientific expertise with genuine love for
              animals to create an environment where pets feel safe,
              comfortable, and genuinely cared for.
            </p>
            <div className="flex flex-wrap gap-2 sm:gap-3">
              {[
                "Certified Vets",
                "Fear-Free Environment",
                "24/7 Monitoring",
                "Licensed Groomers",
                "Positive Training",
                "Trauma-Informed Care",
              ].map((t) => (
                <span
                  key={t}
                  className="bg-orange-50 text-orange-700 text-xs font-bold px-3 sm:px-4 py-1.5 sm:py-2 rounded-full border border-orange-200"
                >
                  ✓ {t}
                </span>
              ))}
            </div>
          </div>
        </div>

        {/* Team */}
        <div className="text-center mb-8 sm:mb-12">
          <h2 className="font-display text-2xl sm:text-3xl md:text-4xl font-bold">
            Meet Our Team
          </h2>
        </div>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 sm:gap-6 mb-16 sm:mb-24">
          {TEAM.map((m) => (
            <div
              key={m.name}
              className="bg-white rounded-3xl p-4 sm:p-6 text-center shadow-sm border border-gray-100 hover:-translate-y-1 hover:shadow-md transition-all"
            >
              <div className="w-14 h-14 sm:w-20 sm:h-20 bg-gradient-to-br from-orange-100 to-amber-100 rounded-full flex items-center justify-center text-2xl sm:text-4xl mx-auto mb-3 sm:mb-4">
                {m.emoji}
              </div>
              <h3 className="font-bold text-xs sm:text-sm text-gray-900">
                {m.name}
              </h3>
              <div className="text-orange-500 font-bold text-xs mt-1">
                {m.role}
              </div>
              <div className="text-gray-400 text-xs mt-1 font-body">
                {m.exp} exp.
              </div>
              <div className="text-gray-500 text-xs mt-1 sm:mt-2 font-body">
                {m.spec}
              </div>
            </div>
          ))}
        </div>

        {/* FAQs */}
        <div className="text-center mb-8 sm:mb-12">
          <h2 className="font-display text-2xl sm:text-3xl md:text-4xl font-bold">
            Frequently Asked Questions
          </h2>
        </div>
        <div className="space-y-3 sm:space-y-4 max-w-3xl mx-auto">
          {FAQS.map((f, i) => (
            <FaqItem key={i} q={f.q} a={f.a} />
          ))}
        </div>
      </div>
    </div>
  );
}

function FaqItem({ q, a }) {
  const [open, setOpen] = useState(false);
  return (
    <div className="bg-white rounded-2xl border border-gray-100 overflow-hidden shadow-sm">
      <button
        className="w-full px-5 sm:px-6 py-4 flex items-center justify-between text-left gap-3"
        onClick={() => setOpen(!open)}
      >
        <span className="font-bold text-sm sm:text-base text-gray-800">
          {q}
        </span>
        <span
          className="text-orange-500 text-lg sm:text-xl transition-transform flex-shrink-0"
          style={{ transform: open ? "rotate(180deg)" : "rotate(0)" }}
        >
          ⌄
        </span>
      </button>
      {open && (
        <div className="px-5 sm:px-6 pb-4 text-gray-600 text-xs sm:text-sm font-body leading-relaxed border-t border-gray-50 pt-3">
          {a}
        </div>
      )}
    </div>
  );
}

// CONTACT PAGE
function ContactPage({ onToast }) {
  const [form, setForm] = useState({
    name: "",
    email: "",
    phone: "",
    subject: "",
    message: "",
  });
  const [errors, setErrors] = useState({});
  const [sent, setSent] = useState(false);
  const inp =
    "w-full bg-gray-50 border-2 border-gray-200 rounded-xl px-4 py-3 text-sm focus:border-orange-400 focus:bg-white outline-none transition-all font-body";
  const lbl =
    "block text-xs font-bold text-gray-600 mb-1.5 uppercase tracking-wide";
  const validate = () => {
    const e = {};
    if (!form.name.trim()) e.name = "Required";
    if (!form.email.match(/\S+@\S+\.\S+/)) e.email = "Valid email required";
    if (!form.message.trim()) e.message = "Required";
    setErrors(e);
    return !Object.keys(e).length;
  };
  const handleSubmit = (e) => {
    e.preventDefault();
    if (validate()) {
      setSent(true);
      onToast(
        "Your message has been sent! We'll reply within 2 hours.",
        "success",
      );
      setForm({ name: "", email: "", phone: "", subject: "", message: "" });
      setTimeout(() => setSent(false), 4000);
    }
  };
  return (
    <div className="min-h-screen bg-gray-50 pt-24 sm:pt-28 pb-16">
      <div className="max-w-5xl mx-auto px-4 sm:px-6">
        <div className="text-center mb-10 sm:mb-14">
          <span className="text-orange-500 font-bold text-xs sm:text-sm uppercase tracking-widest">
            Get in Touch
          </span>
          <h1 className="font-display text-3xl sm:text-4xl md:text-5xl font-black mt-2 mb-3">
            Contact Us
          </h1>
          <p className="text-gray-500 max-w-xl mx-auto text-sm sm:text-base font-body">
            We'd love to hear from you. Reach out anytime!
          </p>
        </div>
        <div className="grid md:grid-cols-2 gap-6 sm:gap-10">
          <div>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 mb-6 sm:mb-8">
              {[
                {
                  ic: "📍",
                  t: "Visit Us",
                  v: "42 Paw Street,\nPet City, Ahmedabad\nGujarat - 380001",
                },
                {
                  ic: "📞",
                  t: "Call Us",
                  v: "+91 80001 23456\n+91 90001 56789",
                },
                {
                  ic: "📧",
                  t: "Email Us",
                  v: "hello@pawhaven.com\nsupport@pawhaven.com",
                },
                {
                  ic: "🕐",
                  t: "Hours",
                  v: "Mon–Sat: 8am – 7pm\nSun: 10am – 4pm",
                },
              ].map((c) => (
                <div
                  key={c.t}
                  className="bg-white rounded-2xl p-4 sm:p-5 border border-gray-100 shadow-sm"
                >
                  <div className="text-2xl mb-2">{c.ic}</div>
                  <div className="font-bold text-xs sm:text-sm text-gray-800 mb-1">
                    {c.t}
                  </div>
                  <div className="text-xs text-gray-500 font-body whitespace-pre-line">
                    {c.v}
                  </div>
                </div>
              ))}
            </div>
            <div className="bg-orange-500 text-white rounded-3xl p-5 sm:p-6">
              <div className="text-3xl mb-3">🚨</div>
              <h3 className="font-bold text-base sm:text-lg mb-2">
                Pet Emergency?
              </h3>
              <p className="text-orange-100 text-xs sm:text-sm font-body mb-3">
                We have 24/7 emergency vet services. Call immediately if your
                pet needs urgent care.
              </p>
              <div className="bg-white/20 rounded-xl px-4 py-2 font-black text-lg sm:text-xl">
                📞 +91 99999 00001
              </div>
            </div>
          </div>
          <div className="bg-white rounded-3xl shadow-sm border border-gray-100 p-5 sm:p-8">
            <h2 className="font-display text-xl sm:text-2xl font-bold mb-5 sm:mb-6">
              Send a Message
            </h2>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div>
                  <label className={lbl}>Name *</label>
                  <input
                    className={inp}
                    placeholder="Your Name"
                    value={form.name}
                    onChange={(e) => setForm({ ...form, name: e.target.value })}
                  />
                  {errors.name && (
                    <p className="text-xs text-red-500 mt-1">{errors.name}</p>
                  )}
                </div>
                <div>
                  <label className={lbl}>Phone</label>
                  <input
                    className={inp}
                    placeholder="Optional"
                    value={form.phone}
                    onChange={(e) =>
                      setForm({ ...form, phone: e.target.value })
                    }
                  />
                </div>
              </div>
              <div>
                <label className={lbl}>Email *</label>
                <input
                  className={inp}
                  type="email"
                  placeholder="you@example.com"
                  value={form.email}
                  onChange={(e) => setForm({ ...form, email: e.target.value })}
                />
                {errors.email && (
                  <p className="text-xs text-red-500 mt-1">{errors.email}</p>
                )}
              </div>
              <div>
                <label className={lbl}>Subject</label>
                <select
                  className={inp}
                  value={form.subject}
                  onChange={(e) =>
                    setForm({ ...form, subject: e.target.value })
                  }
                >
                  <option value="">Select a topic</option>
                  {[
                    "General Inquiry",
                    "Booking Question",
                    "Service Info",
                    "Complaint",
                    "Partnership",
                    "Other",
                  ].map((o) => (
                    <option key={o}>{o}</option>
                  ))}
                </select>
              </div>
              <div>
                <label className={lbl}>Message *</label>
                <textarea
                  className={inp}
                  rows={4}
                  placeholder="How can we help?"
                  value={form.message}
                  onChange={(e) =>
                    setForm({ ...form, message: e.target.value })
                  }
                />
                {errors.message && (
                  <p className="text-xs text-red-500 mt-1">{errors.message}</p>
                )}
              </div>
              <button
                type="submit"
                className="w-full bg-orange-500 hover:bg-orange-600 text-white py-3 sm:py-4 rounded-2xl font-black shadow-lg transition-all hover:-translate-y-0.5 text-sm sm:text-base"
              >
                Send Message 🐾
              </button>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}

/* ─── APP SHELL ──────────────────────────────────────── */
export default function App() {
  const [page, setPage] = useState("home");
  const [menuOpen, setMenuOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);
  const [toast, setToast] = useState(null);
  const [preSelected, setPreSelected] = useState(null);

  useEffect(() => {
    const h = () => setScrolled(window.scrollY > 50);
    window.addEventListener("scroll", h);
    return () => window.removeEventListener("scroll", h);
  }, []);

  useEffect(() => {
    window.scrollTo(0, 0);
  }, [page]);

  const navigate = (p) => {
    setPage(p);
    setMenuOpen(false);
  };

  const bookService = (service, pkg) => {
    setPreSelected({ service, pkg });
    navigate("booking");
  };

  const showToast = (msg, type) => setToast({ msg, type });

  const NAV = [
    { id: "home", label: "Home", icon: "🏠" },
    { id: "services", label: "Services", icon: "🛎️" },
    { id: "about", label: "About", icon: "🐾" },
    { id: "booking", label: "Book Now", icon: "📅" },
    { id: "contact", label: "Contact", icon: "📞" },
  ];

  return (
    <div
      className="font-body text-gray-900"
      style={{ fontFamily: "'DM Sans', sans-serif" }}
    >
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@700;900&family=DM+Sans:wght@300;400;500;700&display=swap');
        .font-display { font-family: 'Playfair Display', Georgia, serif; }
        .font-body { font-family: 'DM Sans', sans-serif; }
        @keyframes float { 0%,100%{transform:translateY(0)} 50%{transform:translateY(-12px)} }
        .animate-float { animation: float 3s ease-in-out infinite; }
        @keyframes slideIn { from{opacity:0;transform:translateX(40px)} to{opacity:1;transform:translateX(0)} }
        * { box-sizing: border-box; }
        .line-clamp-1 { overflow:hidden; display:-webkit-box; -webkit-line-clamp:1; -webkit-box-orient:vertical; }
      `}</style>

      {toast && (
        <Toast
          msg={toast.msg}
          type={toast.type}
          onClose={() => setToast(null)}
        />
      )}

      {/* NAV */}
      <nav
        className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${scrolled || page !== "home" ? "bg-white/95 backdrop-blur-md shadow-md py-3" : "bg-transparent py-4 sm:py-5"}`}
      >
        <div className="max-w-6xl mx-auto px-4 sm:px-6 flex items-center justify-between">
          <button
            onClick={() => navigate("home")}
            className="flex items-center gap-2 group"
          >
            <span className="text-2xl sm:text-3xl group-hover:animate-bounce">
              🐾
            </span>
            <span
              className="font-display text-xl sm:text-2xl font-black"
              style={{ color: "#F97316" }}
            >
              PawHaven
            </span>
          </button>
          {/* Desktop */}
          <div className="hidden md:flex items-center gap-1">
            {NAV.filter((n) => n.id !== "booking").map((n) => (
              <button
                key={n.id}
                onClick={() => navigate(n.id)}
                className={`px-4 py-2 rounded-full text-sm font-semibold transition-all ${page === n.id ? "bg-orange-50 text-orange-600" : "text-gray-600 hover:text-orange-500 hover:bg-orange-50"}`}
              >
                {n.label}
              </button>
            ))}
            <button
              onClick={() => navigate("booking")}
              className="ml-2 bg-orange-500 hover:bg-orange-600 text-white px-5 py-2 rounded-full text-sm font-bold shadow-lg hover:shadow-orange-200 transition-all hover:-translate-y-0.5"
            >
              📅 Book Now
            </button>
          </div>
          {/* Mobile */}
          <button
            className="md:hidden w-10 h-10 flex items-center justify-center text-gray-700 text-xl"
            onClick={() => setMenuOpen(!menuOpen)}
          >
            {menuOpen ? "✕" : "☰"}
          </button>
        </div>
        {/* Mobile Menu */}
        {menuOpen && (
          <div className="md:hidden bg-white border-t border-gray-100 shadow-xl px-4 py-4">
            <div className="grid grid-cols-2 gap-2">
              {NAV.map((n) => (
                <button
                  key={n.id}
                  onClick={() => navigate(n.id)}
                  className={`flex items-center gap-2 p-3 rounded-xl text-sm font-semibold transition-all ${page === n.id ? "bg-orange-50 text-orange-600" : "bg-gray-50 text-gray-700 hover:bg-orange-50"} ${n.id === "booking" ? "col-span-2 bg-orange-500 text-white hover:bg-orange-600" : ""}`}
                >
                  <span>{n.icon}</span> {n.label}
                </button>
              ))}
            </div>
          </div>
        )}
      </nav>

      {/* PAGES */}
      <main>
        {page === "home" && (
          <HomePage onNavigate={navigate} onBookService={bookService} />
        )}
        {page === "services" && <ServicesPage onBookService={bookService} />}
        {page === "booking" && (
          <BookingPage
            preSelected={preSelected}
            onConfirm={(data) =>
              showToast(
                `${data.form.petName}'s appointment confirmed for ${data.form.date} at ${data.form.time}!`,
                "success",
              )
            }
          />
        )}
        {page === "about" && <AboutPage />}
        {page === "contact" && <ContactPage onToast={showToast} />}
      </main>

      {/* FOOTER */}
      <footer className="bg-gray-900 text-white pt-12 sm:pt-16 pb-6">
        <div className="max-w-6xl mx-auto px-4 sm:px-6">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8 sm:gap-10 mb-10 sm:mb-12">
            <div className="col-span-2 md:col-span-1">
              <div className="flex items-center gap-2 mb-3 sm:mb-4">
                <span className="text-2xl">🐾</span>
                <span
                  className="font-display text-xl font-black"
                  style={{ color: "#F97316" }}
                >
                  PawHaven
                </span>
              </div>
              <p className="text-gray-400 text-xs sm:text-sm font-body leading-relaxed mb-4">
                Gujarat's #1 trusted pet care center. Every animal deserves to
                be loved and well-cared for.
              </p>
              <div className="flex gap-2 sm:gap-3">
                {["📘", "📸", "🐦", "▶️"].map((ic) => (
                  <button
                    key={ic}
                    className="w-8 h-8 sm:w-9 sm:h-9 bg-gray-800 hover:bg-orange-500 rounded-full flex items-center justify-center text-sm transition-all"
                  >
                    {ic}
                  </button>
                ))}
              </div>
            </div>
            {[
              { t: "Services", links: SERVICES.map((s) => s.title) },
              {
                t: "Company",
                links: ["About Us", "Our Team", "Careers", "Blog", "Press"],
              },
              {
                t: "Support",
                links: [
                  "Contact",
                  "FAQs",
                  "Booking Policy",
                  "Privacy",
                  "Terms",
                ],
              },
            ].map((col) => (
              <div key={col.t}>
                <h4 className="font-bold text-xs uppercase tracking-widest text-gray-400 mb-3 sm:mb-4">
                  {col.t}
                </h4>
                <ul className="space-y-2">
                  {col.links.map((l) => (
                    <li key={l}>
                      <button className="text-gray-500 hover:text-orange-400 text-xs sm:text-sm transition-colors font-body">
                        {l}
                      </button>
                    </li>
                  ))}
                </ul>
              </div>
            ))}
          </div>
          <div className="border-t border-gray-800 pt-5 sm:pt-6 flex flex-col sm:flex-row items-center justify-between gap-3">
            <p className="text-gray-600 text-xs font-body">
              © 2026 PawHaven. All rights reserved. Made with for pets.
            </p>
            <div className="flex items-center gap-2">
              <div className="w-2 h-2 rounded-full bg-green-400 animate-pulse" />
              <span className="text-gray-500 text-xs font-body">
                All services operational
              </span>
            </div>
          </div>
        </div>
      </footer>
    </div>
  );
}
