import Task from '../models/Task.js';
import Project from '../models/Project.js';

// @desc    Create a new task
// @route   POST /api/tasks
// @access  Private
const createTask = async (req, res) => {
    const { title, description, projectId, assignedTo, priority, dueDate } = req.body;

    // Verify project existence and user permissions
    const project = await Project.findById(projectId);
    if (!project) {
        res.status(404).json({ message: 'Project not found' });
        return;
    }

    // Only Admin or Project Creator or Manager can create tasks
    if (req.user.role !== 'Admin' && project.createdBy.toString() !== req.user._id.toString()) {
        res.status(401).json({ message: 'Not authorized to add tasks to this project' });
        return;
    }

    const task = new Task({
        title,
        description,
        project: projectId,
        assignedTo,
        priority,
        dueDate,
        status: 'To Do'
    });

    const createdTask = await task.save();
    res.status(201).json(createdTask);
};

// @desc    Get tasks for a project
// @route   GET /api/tasks/project/:projectId
// @access  Private
const getTasks = async (req, res) => {
    const tasks = await Task.find({ project: req.params.projectId })
        .populate('assignedTo', 'name email')
        .populate('project', 'name');
    res.json(tasks);
};

// @desc    Update task status
// @route   PUT /api/tasks/:id
// @access  Private
const updateTask = async (req, res) => {
    const { status } = req.body;
    const task = await Task.findById(req.params.id);

    if (task) {
        task.status = status || task.status;
        const updatedTask = await task.save();
        res.json(updatedTask);
    } else {
        res.status(404).json({ message: 'Task not found' });
    }
};

export { createTask, getTasks, updateTask };
