import Project from '../models/Project.js';
import User from '../models/User.js';

// @desc    Create a new project
// @route   POST /api/projects
// @access  Private/ProjectManager/Admin
const createProject = async (req, res) => {
    const { name, description, deadline, members } = req.body;

    // Admin projects are Active by default, others are Pending
    const initialStatus = req.user.role === 'Admin' ? 'Active' : 'Pending';

    const project = new Project({
        name,
        description,
        deadline,
        createdBy: req.user._id,
        members: members || [],
        status: initialStatus
    });

    const createdProject = await project.save();
    res.status(201).json(createdProject);
};

// @desc    Get all projects
// @route   GET /api/projects
// @access  Private
const getProjects = async (req, res) => {
    let projects;
    if (req.user.role === 'Admin') {
        // Admin sees all projects
        projects = await Project.find({})
            .populate('createdBy', 'name email')
            .populate('members', 'name email')
            .sort({ createdAt: -1 });
    } else {
        // Managers/Members see projects they created OR are members of.
        // AND they should see their own Pending projects. 
        // Requirement Update: "pending show on both user and manager side" -> Assume transparency for Pending? 
        // For now, sticking to standard RBAC: You see what you are involved in.
        // If a Manager creates a project, they must see it to track it (even if Pending).

        projects = await Project.find({
            $or: [{ createdBy: req.user._id }, { members: req.user._id }]
        })
            .populate('createdBy', 'name email')
            .populate('members', 'name email')
            .sort({ createdAt: -1 });
    }
    res.json(projects);
};

// @desc    Get project by ID
// @route   GET /api/projects/:id
// @access  Private
const getProjectById = async (req, res) => {
    const project = await Project.findById(req.params.id)
        .populate('createdBy', 'name email')
        .populate('members', 'name email');

    if (project) {
        const isMember = project.members.some(member => member._id.toString() === req.user._id.toString());
        const isCreator = project.createdBy._id.toString() === req.user._id.toString();
        const isAdmin = req.user.role === 'Admin';

        if (isAdmin || isCreator || isMember) {
            res.json(project);
        } else {
            res.status(401).json({ message: 'Not authorized to view this project' });
        }
    } else {
        res.status(404).json({ message: 'Project not found' });
    }
};

// @desc    Update project (Status change for Admin, Details for Manager)
// @route   PUT /api/projects/:id
// @access  Private
const updateProject = async (req, res) => {
    const { status, name, description, deadline, members } = req.body;
    const project = await Project.findById(req.params.id);

    if (!project) {
        res.status(404).json({ message: 'Project not found' });
        return;
    }

    // Authorization
    const isAdmin = req.user.role === 'Admin';
    const isCreator = project.createdBy.toString() === req.user._id.toString();

    if (!isAdmin && !isCreator) {
        res.status(401).json({ message: 'Not authorized to update this project' });
        return;
    }

    // Only Admin can change status (Approve/Finish)
    if (status) {
        if (isAdmin) {
            project.status = status;
        } else {
            // Managers cannot change status unless implementation specific logic allowed it (e.g. Active -> Completed?)
            // For now, strict: Only Admin approves or changes status.
            res.status(401).json({ message: 'Only Admin can change project status' });
            return;
        }
    }

    // Update details if provided
    if (name) project.name = name;
    if (description) project.description = description;
    if (deadline) project.deadline = deadline;
    if (members) project.members = members;

    const updatedProject = await project.save();
    res.json(updatedProject);
};

export { createProject, getProjects, getProjectById, updateProject };
