import React, { useEffect, useState } from "react";
import { useParams } from "react-router-dom";
import api from "../api/axios";
import { DndContext, closestCorners } from "@dnd-kit/core";
import {
  SortableContext,
  verticalListSortingStrategy,
} from "@dnd-kit/sortable";
import { useSortable } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { Plus } from "lucide-react";
import CreateTaskModal from "../components/CreateTaskModal";

const TaskCard = ({ task }) => {
  const { attributes, listeners, setNodeRef, transform, transition } =
    useSortable({ id: task._id });
  const style = { transform: CSS.Transform.toString(transform), transition };

  return (
    <div
      ref={setNodeRef}
      style={style}
      {...attributes}
      {...listeners}
      className="bg-white p-3 rounded shadow-sm mb-2 border hover:shadow-md cursor-grab active:cursor-grabbing"
    >
      <h4 className="font-medium text-gray-800">{task.title}</h4>
      <div className="flex justify-between items-center mt-2 text-xs text-gray-500">
        <span
          className={`px-2 py-0.5 rounded ${
            task.priority === "High"
              ? "bg-red-100 text-red-700"
              : task.priority === "Medium"
                ? "bg-yellow-100 text-yellow-700"
                : "bg-green-100 text-green-700"
          }`}
        >
          {task.priority}
        </span>
        {task.assignedTo && <span>{task.assignedTo.name}</span>}
      </div>
    </div>
  );
};

const Column = ({ title, tasks, id }) => {
  const { setNodeRef } = useSortable({ id });

  return (
    <div
      ref={setNodeRef}
      className="bg-gray-100 p-4 rounded-lg w-full md:w-1/3 min-h-[500px]"
    >
      <div className="flex justify-between items-center mb-4">
        <h3 className="font-bold text-gray-700">{title}</h3>
        <span className="bg-gray-200 px-2 py-1 rounded text-sm font-semibold">
          {tasks.length}
        </span>
      </div>
      <SortableContext
        items={tasks.map((t) => t._id)}
        strategy={verticalListSortingStrategy}
      >
        <div className="space-y-3">
          {tasks.map((task) => (
            <TaskCard key={task._id} task={task} />
          ))}
        </div>
      </SortableContext>
    </div>
  );
};

const ProjectBoard = () => {
  const { id } = useParams();
  const [project, setProject] = useState(null);
  const [tasks, setTasks] = useState([]);
  const [loading, setLoading] = useState(true);
  const [isTaskModalOpen, setIsTaskModalOpen] = useState(false);

  useEffect(() => {
    fetchProjectData();
  }, [id]);

  const fetchProjectData = async () => {
    try {
      const [projRes, tasksRes] = await Promise.all([
        api.get(`/projects/${id}`),
        api.get(`/tasks/project/${id}`),
      ]);
      setProject(projRes.data);
      setTasks(tasksRes.data);
    } catch (error) {
      console.error("Error fetching board data", error);
    } finally {
      setLoading(false);
    }
  };

  const handleTaskCreated = (newTask) => {
    // Need to populate assignedTo properly or just refetch.
    // Refetch is safer to get populated fields.
    fetchProjectData();
  };

  if (loading) return <div>Loading Board...</div>;

  return (
    <div className="h-full flex flex-col">
      <div className="mb-6 flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold text-gray-800">{project?.name}</h1>
          <p className="text-gray-500">{project?.description}</p>
          <span
            className={`inline-block mt-2 px-2 py-1 rounded text-xs font-medium ${
              project.status === "Active"
                ? "bg-green-100 text-green-700"
                : project.status === "Completed"
                  ? "bg-blue-100 text-blue-700"
                  : project.status === "Pending"
                    ? "bg-yellow-200 text-yellow-800"
                    : "bg-gray-100 text-gray-700"
            }`}
          >
            {project.status}
          </span>
        </div>
        {/* Limit Task Creation to Active Projects? Or let it be open? Usually Active projects have tasks. */}
        {project.status === "Active" && (
          <button
            onClick={() => setIsTaskModalOpen(true)}
            className="bg-blue-600 text-white px-4 py-2 rounded flex items-center gap-2 hover:bg-blue-700 transition"
          >
            <Plus size={20} /> New Task
          </button>
        )}
      </div>

      <div className="flex flex-col md:flex-row gap-6 overflow-x-auto pb-4">
        {["To Do", "In Progress", "Done"].map((status) => (
          <div
            key={status}
            className="bg-gray-100 p-4 rounded-lg w-full md:w-1/3 min-w-[300px]"
          >
            <h3 className="font-bold text-gray-700 mb-4 flex justify-between">
              {status}
              <span className="bg-gray-200 px-2 rounded text-sm">
                {tasks.filter((t) => t.status === status).length}
              </span>
            </h3>
            <div className="space-y-3">
              {tasks
                .filter((t) => t.status === status)
                .map((task) => (
                  <div
                    key={task._id}
                    className="bg-white p-4 rounded shadow border hover:shadow-md transition"
                  >
                    <h4 className="font-medium">{task.title}</h4>
                    <div className="mt-2 text-xs flex justify-between text-gray-500">
                      <span
                        className={`px-2 py-0.5 rounded ${
                          task.priority === "High"
                            ? "bg-red-100 text-red-700"
                            : task.priority === "Medium"
                              ? "bg-yellow-100 text-yellow-700"
                              : "bg-green-100 text-green-700"
                        }`}
                      >
                        {task.priority}
                      </span>
                      <span>{new Date(task.dueDate).toLocaleDateString()}</span>
                    </div>
                    {task.assignedTo && (
                      <div className="mt-2 text-xs text-blue-600 bg-blue-50 px-2 py-1 rounded inline-block">
                        {task.assignedTo.name}
                      </div>
                    )}
                    <div className="mt-3 flex gap-2">
                      {status !== "To Do" && (
                        <button
                          onClick={async () => {
                            const newStatus =
                              status === "Done" ? "In Progress" : "To Do";
                            await api.put(`/tasks/${task._id}`, {
                              status: newStatus,
                            });
                            fetchProjectData();
                          }}
                          className="text-xs bg-gray-200 px-2 py-1 rounded hover:bg-gray-300"
                        >
                          ←
                        </button>
                      )}
                      {status !== "Done" && (
                        <button
                          onClick={async () => {
                            const newStatus =
                              status === "To Do" ? "In Progress" : "Done";
                            await api.put(`/tasks/${task._id}`, {
                              status: newStatus,
                            });
                            fetchProjectData();
                          }}
                          className="text-xs bg-gray-200 px-2 py-1 rounded hover:bg-gray-300 ml-auto"
                        >
                          →
                        </button>
                      )}
                    </div>
                  </div>
                ))}
            </div>
          </div>
        ))}
      </div>

      {isTaskModalOpen && (
        <CreateTaskModal
          projectId={id}
          onClose={() => setIsTaskModalOpen(false)}
          onTaskCreated={handleTaskCreated}
        />
      )}
    </div>
  );
};

export default ProjectBoard;
