import React, { useEffect, useState } from "react";
import api from "../api/axios";
import { Link } from "react-router-dom";
import { Plus, CheckCircle, XCircle } from "lucide-react";
import { useAuth } from "../context/AuthContext";
import CreateProjectModal from "../components/CreateProjectModal";

const Dashboard = () => {
  const [projects, setProjects] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const { user } = useAuth();

  useEffect(() => {
    fetchProjects();
  }, []);

  const fetchProjects = async () => {
    try {
      const { data } = await api.get("/projects");
      setProjects(data);
    } catch (error) {
      console.error("Failed to fetch projects", error);
    }
  };

  const handleProjectCreated = (newProject) => {
    // If admin, it's active. If manager, it's pending.
    // We just refetch or add to list.
    // If pending, it might show up depending on logic.
    // But since backend returns what we are allowed to see, we just add it.
    setProjects([newProject, ...projects]);
  };

  const handleStatusChange = async (e, projectId, newStatus) => {
    e.preventDefault(); // Prevent link navigation
    e.stopPropagation();
    try {
      const { data } = await api.put(`/projects/${projectId}`, {
        status: newStatus,
      });
      setProjects(projects.map((p) => (p._id === projectId ? data : p)));
    } catch (error) {
      console.error("Failed to update status", error);
      alert("Failed to update status");
    }
  };

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h2 className="text-2xl font-bold text-gray-800">Projects</h2>
        {(user?.role === "Admin" || user?.role === "ProjectManager") && (
          <button
            onClick={() => setIsModalOpen(true)}
            className="flex items-center gap-2 bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition"
          >
            <Plus size={20} />
            New Project
          </button>
        )}
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {projects.map((project) => (
          <Link
            to={`/projects/${project._id}`}
            key={project._id}
            className="block group"
          >
            <div
              className={`bg-white p-6 rounded-xl shadow-sm border group-hover:shadow-md transition relative ${
                project.status === "Pending"
                  ? "border-yellow-200 bg-yellow-50"
                  : "border-gray-100"
              }`}
            >
              <div className="flex justify-between items-start mb-4">
                <h3 className="text-lg font-semibold text-gray-800 group-hover:text-blue-600 transition">
                  {project.name}
                </h3>
                <span
                  className={`px-2 py-1 rounded text-xs font-medium ${
                    project.status === "Active"
                      ? "bg-green-100 text-green-700"
                      : project.status === "Completed"
                        ? "bg-blue-100 text-blue-700"
                        : project.status === "Pending"
                          ? "bg-yellow-200 text-yellow-800"
                          : "bg-gray-100 text-gray-700"
                  }`}
                >
                  {project.status}
                </span>
              </div>
              <p className="text-gray-600 text-sm mb-4 line-clamp-2">
                {project.description}
              </p>
              <div className="flex items-center justify-between text-sm text-gray-500 mb-4">
                <span>{new Date(project.deadline).toLocaleDateString()}</span>
                <span>{project.members.length} Members</span>
              </div>

              {/* Admin Action Buttons for Pending Projects */}
              {user.role === "Admin" && project.status === "Pending" && (
                <div className="flex gap-2 mt-2 border-t pt-3">
                  <button
                    onClick={(e) =>
                      handleStatusChange(e, project._id, "Active")
                    }
                    className="flex-1 flex items-center justify-center gap-1 bg-green-500 text-white py-1 px-2 rounded text-sm hover:bg-green-600"
                  >
                    <CheckCircle size={14} /> Approve
                  </button>
                  <button
                    onClick={(e) =>
                      handleStatusChange(e, project._id, "Archived")
                    } // Or Rejected? Assuming Archive for now
                    className="flex-1 flex items-center justify-center gap-1 bg-gray-500 text-white py-1 px-2 rounded text-sm hover:bg-gray-600"
                  >
                    <XCircle size={14} /> Reject
                  </button>
                </div>
              )}

              {/* Admin Action Buttons for Active Projects */}
              {user.role === "Admin" && project.status === "Active" && (
                <div className="mt-2 border-t pt-3">
                  <button
                    onClick={(e) =>
                      handleStatusChange(e, project._id, "Completed")
                    }
                    className="w-full flex items-center justify-center gap-1 bg-blue-500 text-white py-1 px-2 rounded text-sm hover:bg-blue-600"
                  >
                    <CheckCircle size={14} /> Mark Finished
                  </button>
                </div>
              )}
            </div>
          </Link>
        ))}
        {projects.length === 0 && (
          <div className="col-span-full text-center py-12 text-gray-500">
            No projects found.
          </div>
        )}
      </div>

      {isModalOpen && (
        <CreateProjectModal
          onClose={() => setIsModalOpen(false)}
          onProjectCreated={handleProjectCreated}
        />
      )}
    </div>
  );
};

export default Dashboard;
