import React, { useState, useEffect } from "react";
import api from "../api/axios";
import { X } from "lucide-react";

const CreateTaskModal = ({ onClose, projectId, onTaskCreated }) => {
  const [title, setTitle] = useState("");
  const [description, setDescription] = useState("");
  const [priority, setPriority] = useState("Medium");
  const [dueDate, setDueDate] = useState("");
  const [assignedTo, setAssignedTo] = useState("");
  const [users, setUsers] = useState([]);

  useEffect(() => {
    // Fetch project members or all users?
    // Ideally only project members. But our project object has members.
    // For simplicity, let's just fetch all users for now or fetch project details to get members
    // We can fetch project again here or pass members as props.
    // Let's pass members as props for efficiency.
    // But since I didn't update parent yet, I'll fetch project members here.
    const fetchMembers = async () => {
      try {
        const { data } = await api.get(`/projects/${projectId}`);
        // Include creator and members
        const allMembers = [data.createdBy, ...data.members];
        // Remove duplicates by ID
        const uniqueMembers = Array.from(
          new Map(allMembers.map((item) => [item._id, item])).values(),
        );
        setUsers(uniqueMembers);
      } catch (error) {
        console.error(error);
      }
    };
    fetchMembers();
  }, [projectId]);

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      const { data } = await api.post("/tasks", {
        title,
        description,
        projectId,
        priority,
        dueDate,
        assignedTo: assignedTo || null,
      });
      onTaskCreated(data);
      onClose();
    } catch (error) {
      console.error(error);
      alert(error.response?.data?.message || "Failed to create task");
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white p-6 rounded-lg w-full max-w-md">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-xl font-bold">Add New Task</h2>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
          >
            <X size={24} />
          </button>
        </div>
        <form onSubmit={handleSubmit}>
          <div className="mb-4">
            <label className="block text-sm font-medium mb-1 text-gray-700">
              Task Title
            </label>
            <input
              className="w-full border rounded p-2 focus:ring-2 focus:ring-blue-500 outline-none"
              value={title}
              onChange={(e) => setTitle(e.target.value)}
              required
            />
          </div>
          <div className="mb-4">
            <label className="block text-sm font-medium mb-1 text-gray-700">
              Description
            </label>
            <textarea
              className="w-full border rounded p-2 focus:ring-2 focus:ring-blue-500 outline-none h-20"
              value={description}
              onChange={(e) => setDescription(e.target.value)}
            />
          </div>
          <div className="grid grid-cols-2 gap-4 mb-4">
            <div>
              <label className="block text-sm font-medium mb-1 text-gray-700">
                Priority
              </label>
              <select
                className="w-full border rounded p-2 focus:ring-2 focus:ring-blue-500"
                value={priority}
                onChange={(e) => setPriority(e.target.value)}
              >
                <option value="Low">Low</option>
                <option value="Medium">Medium</option>
                <option value="High">High</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium mb-1 text-gray-700">
                Due Date
              </label>
              <input
                type="date"
                className="w-full border rounded p-2 focus:ring-2 focus:ring-blue-500"
                value={dueDate}
                onChange={(e) => setDueDate(e.target.value)}
              />
            </div>
          </div>
          <div className="mb-6">
            <label className="block text-sm font-medium mb-1 text-gray-700">
              Assign To
            </label>
            <select
              className="w-full border rounded p-2 focus:ring-2 focus:ring-blue-500"
              value={assignedTo}
              onChange={(e) => setAssignedTo(e.target.value)}
            >
              <option value="">-- Unassigned --</option>
              {users.map((u) => (
                <option key={u._id} value={u._id}>
                  {u.name} ({u.email})
                </option>
              ))}
            </select>
          </div>
          <button
            type="submit"
            className="w-full bg-blue-600 text-white p-2 rounded hover:bg-blue-700 transition"
          >
            Create Task
          </button>
        </form>
      </div>
    </div>
  );
};
export default CreateTaskModal;
