import React, { useState, useEffect, useRef } from "react";

export default function DiyaCatcher() {
  const [gameState, setGameState] = useState("menu"); // menu, playing, gameOver
  const [score, setScore] = useState(0);
  const [level, setLevel] = useState(1);
  const [lives, setLives] = useState(3);
  const [basketPos, setBasketPos] = useState(50);
  const [diyas, setDiyas] = useState([]);
  const [highScore, setHighScore] = useState(0);
  const gameLoopRef = useRef(null);
  const diyaSpawnRef = useRef(null);

  // Game dimensions - responsive
  const gameWidth =
    typeof window !== "undefined" ? Math.min(window.innerWidth - 40, 600) : 600;
  const gameHeight =
    typeof window !== "undefined"
      ? Math.min(window.innerHeight * 0.65, 450)
      : 450;

  // Start game
  const startGame = () => {
    setGameState("playing");
    setScore(0);
    setLevel(1);
    setLives(3);
    setDiyas([]);
    setBasketPos(50);
  };

  // Move basket with mouse/touch
  const handleMove = (e) => {
    if (gameState !== "playing") return;

    const rect = e.currentTarget.getBoundingClientRect();
    let clientX;

    if (e.type === "touchmove") {
      clientX = e.touches[0].clientX;
    } else {
      clientX = e.clientX;
    }

    const x = ((clientX - rect.left) / rect.width) * 100;
    setBasketPos(Math.max(5, Math.min(95, x)));
  };

  // Spawn diyas
  useEffect(() => {
    if (gameState === "playing") {
      const spawnInterval = Math.max(800 - level * 50, 400);

      diyaSpawnRef.current = setInterval(() => {
        const newDiya = {
          id: Date.now() + Math.random(),
          x: Math.random() * 90 + 5,
          y: 0,
          speed: 1 + level * 0.3,
          type: Math.random() > 0.8 ? "golden" : "normal",
        };
        setDiyas((prev) => [...prev, newDiya]);
      }, spawnInterval);

      return () => clearInterval(diyaSpawnRef.current);
    }
  }, [gameState, level]);

  // Game loop
  useEffect(() => {
    if (gameState === "playing") {
      gameLoopRef.current = setInterval(() => {
        setDiyas((prev) => {
          const updated = prev.map((diya) => ({
            ...diya,
            y: diya.y + diya.speed,
          }));

          // Check collisions and misses
          const remaining = updated.filter((diya) => {
            if (diya.y > 100) {
              // Missed diya
              if (diya.type !== "golden") {
                setLives((l) => {
                  const newLives = Math.max(0, l - 1);
                  if (newLives <= 0) {
                    setGameState("gameOver");
                    setHighScore((hs) => Math.max(hs, score));
                  }
                  return newLives;
                });
              }
              return false;
            }

            // Check if caught
            if (diya.y > 85 && diya.y < 95) {
              const caught = Math.abs(diya.x - basketPos) < 8;
              if (caught) {
                const points = diya.type === "golden" ? 10 : 5;
                setScore((s) => {
                  const newScore = s + points;
                  if (newScore % 50 === 0 && newScore > 0) {
                    setLevel((l) => l + 1);
                  }
                  return newScore;
                });
                return false;
              }
            }

            return true;
          });

          return remaining;
        });
      }, 50);

      return () => clearInterval(gameLoopRef.current);
    }
  }, [gameState, basketPos, score, level]);

  // Keyboard controls
  useEffect(() => {
    const handleKeyDown = (e) => {
      if (gameState !== "playing") return;

      if (e.key === "ArrowLeft") {
        setBasketPos((pos) => Math.max(5, pos - 3));
      } else if (e.key === "ArrowRight") {
        setBasketPos((pos) => Math.min(95, pos + 3));
      }
    };

    window.addEventListener("keydown", handleKeyDown);
    return () => window.removeEventListener("keydown", handleKeyDown);
  }, [gameState]);

  return (
    <div
      style={{
        position: "fixed",
        top: 0,
        left: 0,
        right: 0,
        bottom: 0,
        height: "100vh",
        width: "100vw",
        background:
          "linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%)",
        display: "flex",
        alignItems: "center",
        justifyContent: "center",
        padding: "20px",
        fontFamily: "system-ui, -apple-system, sans-serif",
        overflow: "hidden",
      }}
    >
      <div
        style={{
          maxWidth: "600px",
          width: "100%",
          textAlign: "center",
          margin: "auto",
          display: "flex",
          flexDirection: "column",
          justifyContent: "center",
          alignItems: "center",
        }}
      >
        {/* Header */}
        <div
          style={{
            background: "linear-gradient(135deg, #ff6b6b 0%, #feca57 100%)",
            padding: "15px 20px",
            borderRadius: "15px 15px 0 0",
            color: "white",
            boxShadow: "0 4px 15px rgba(0,0,0,0.3)",
            width: "100%",
          }}
        >
          <h1
            style={{
              margin: "0 0 10px 0",
              fontSize: "clamp(24px, 5vw, 36px)",
              textShadow: "2px 2px 4px rgba(0,0,0,0.3)",
            }}
          >
            🪔 Diya Catcher 🪔
          </h1>
          <div
            style={{
              display: "flex",
              justifyContent: "space-around",
              fontSize: "clamp(14px, 3vw, 18px)",
              fontWeight: "bold",
            }}
          >
            <div>Score: {score}</div>
            <div>Level: {level}</div>
            <div>Lives: {"❤️".repeat(Math.max(0, lives))}</div>
            <div>Best: {highScore}</div>
          </div>
        </div>

        {/* Game Area */}
        <div
          onMouseMove={handleMove}
          onTouchMove={handleMove}
          style={{
            width: "100%",
            height: `${gameHeight}px`,
            background:
              "linear-gradient(180deg, #0f0c29 0%, #302b63 50%, #24243e 100%)",
            position: "relative",
            overflow: "hidden",
            cursor: gameState === "playing" ? "none" : "default",
            borderRadius: "0 0 15px 15px",
            boxShadow: "0 4px 15px rgba(0,0,0,0.3)",
          }}
        >
          {/* Menu */}
          {gameState === "menu" && (
            <div
              style={{
                position: "absolute",
                top: "50%",
                left: "50%",
                transform: "translate(-50%, -50%)",
                background: "rgba(0,0,0,0.8)",
                padding: "40px",
                borderRadius: "20px",
                color: "white",
                minWidth: "80%",
                maxWidth: "400px",
              }}
            >
              <div style={{ fontSize: "60px", marginBottom: "20px" }}>🪔</div>
              <h2 style={{ margin: "0 0 20px 0" }}>Welcome!</h2>
              <p style={{ margin: "0 0 30px 0", fontSize: "14px" }}>
                Catch the falling diyas in your basket!
                <br />
                🪔 Normal Diya = 5 points
                <br />
                ✨ Golden Diya = 10 points
                <br />
                Miss 3 diyas and game over!
              </p>
              <button
                onClick={startGame}
                style={{
                  background: "linear-gradient(135deg, #ff6b6b, #feca57)",
                  border: "none",
                  padding: "15px 40px",
                  fontSize: "18px",
                  fontWeight: "bold",
                  borderRadius: "30px",
                  cursor: "pointer",
                  color: "white",
                  boxShadow: "0 4px 15px rgba(255,107,107,0.4)",
                  transition: "transform 0.2s",
                }}
                onMouseEnter={(e) => (e.target.style.transform = "scale(1.05)")}
                onMouseLeave={(e) => (e.target.style.transform = "scale(1)")}
              >
                Start Game
              </button>
              <p style={{ marginTop: "20px", fontSize: "12px", opacity: 0.7 }}>
                Use mouse/touch or ← → keys
              </p>
            </div>
          )}

          {/* Game Over */}
          {gameState === "gameOver" && (
            <div
              style={{
                position: "absolute",
                top: "50%",
                left: "50%",
                transform: "translate(-50%, -50%)",
                background: "rgba(0,0,0,0.9)",
                padding: "40px",
                borderRadius: "20px",
                color: "white",
                minWidth: "80%",
                maxWidth: "400px",
              }}
            >
              <div style={{ fontSize: "60px", marginBottom: "20px" }}>💔</div>
              <h2 style={{ margin: "0 0 20px 0" }}>Game Over!</h2>
              <div
                style={{ fontSize: "24px", margin: "20px 0", color: "#feca57" }}
              >
                Final Score: {score}
              </div>
              {score === highScore && score > 0 && (
                <div
                  style={{
                    color: "#ff6b6b",
                    marginBottom: "20px",
                    fontSize: "18px",
                  }}
                >
                  🎉 New High Score! 🎉
                </div>
              )}
              <button
                onClick={startGame}
                style={{
                  background: "linear-gradient(135deg, #ff6b6b, #feca57)",
                  border: "none",
                  padding: "15px 40px",
                  fontSize: "18px",
                  fontWeight: "bold",
                  borderRadius: "30px",
                  cursor: "pointer",
                  color: "white",
                  boxShadow: "0 4px 15px rgba(255,107,107,0.4)",
                  transition: "transform 0.2s",
                }}
                onMouseEnter={(e) => (e.target.style.transform = "scale(1.05)")}
                onMouseLeave={(e) => (e.target.style.transform = "scale(1)")}
              >
                Play Again
              </button>
            </div>
          )}

          {/* Falling Diyas */}
          {diyas.map((diya) => (
            <div
              key={diya.id}
              style={{
                position: "absolute",
                left: `${diya.x}%`,
                top: `${diya.y}%`,
                fontSize: "30px",
                transform: "translate(-50%, -50%)",
                animation: "glow 1s ease-in-out infinite alternate",
                filter:
                  diya.type === "golden"
                    ? "brightness(1.5) drop-shadow(0 0 10px gold)"
                    : "drop-shadow(0 0 5px orange)",
              }}
            >
              {diya.type === "golden" ? "✨" : "🪔"}
            </div>
          ))}

          {/* Basket */}
          {gameState === "playing" && (
            <div
              style={{
                position: "absolute",
                left: `${basketPos}%`,
                bottom: "30px",
                transform: "translateX(-50%)",
                fontSize: "50px",
                filter: "drop-shadow(0 0 10px rgba(255,107,107,0.5))",
                transition: "left 0.1s ease-out",
              }}
            >
              🧺
            </div>
          )}
        </div>

        {/* Instructions */}
        {gameState === "playing" && (
          <div
            style={{
              marginTop: "20px",
              color: "white",
              fontSize: "14px",
              opacity: 0.7,
            }}
          >
            Move with mouse/touch or use ← → arrow keys
          </div>
        )}
      </div>

      <style>{`
        @keyframes glow {
          from {
            filter: brightness(1) drop-shadow(0 0 5px orange);
          }
          to {
            filter: brightness(1.3) drop-shadow(0 0 15px orange);
          }
        }
      `}</style>
    </div>
  );
}
