import React, { useState, useEffect } from "react";

// Storage with persistence
const Storage = {
  save(key, value) {
    try {
      window.storage.set(key, JSON.stringify(value));
    } catch (e) {
      localStorage.setItem(key, JSON.stringify(value));
    }
  },

  load(key, defaultValue) {
    try {
      window.storage.get(key).then((result) => {
        if (result) return JSON.parse(result.value);
      });
    } catch (e) {
      const item = localStorage.getItem(key);
      if (item) return JSON.parse(item);
    }
    return defaultValue;
  },
};

// Initial data
let USERS = Storage.load("users", [
  {
    id: 1,
    name: "Test User",
    email: "test@test.com",
    password: "123",
    phone: "9876543210",
    location: "Mumbai",
    isCook: false,
  },
]);

let CURRENT_USER = Storage.load("currentUser", null);

let BOOKINGS = Storage.load("bookings", []);

let COOKS = Storage.load("cooks", [
  {
    id: 1,
    name: "Chef Maria Rodriguez",
    specialty: "Italian & Mediterranean",
    cuisine: "italian",
    rating: 5.0,
    reviews: 156,
    experience: "8 years",
    verified: true,
    price: 1299,
    avatar: "👩‍🍳",
    description:
      "Authentic Italian cuisine with a modern twist. Specializing in fresh pasta, risotto, and classic Mediterranean dishes.",
  },
  {
    id: 2,
    name: "Chef Rajesh Kumar",
    specialty: "North Indian Specialist",
    cuisine: "indian",
    rating: 4.9,
    reviews: 203,
    experience: "12 years",
    verified: true,
    price: 899,
    avatar: "👨‍🍳",
    description:
      "Traditional North Indian flavors with recipes passed down through generations. Expert in tandoori and curries.",
  },
  {
    id: 3,
    name: "Chef Emily Chen",
    specialty: "Chinese & Asian Fusion",
    cuisine: "chinese",
    rating: 4.8,
    reviews: 142,
    experience: "6 years",
    verified: true,
    price: 1099,
    avatar: "👩‍🍳",
    description:
      "Contemporary Asian fusion cuisine blending traditional Chinese techniques with modern presentation.",
  },
  {
    id: 4,
    name: "Chef Antonio Garcia",
    specialty: "Mexican & Latin American",
    cuisine: "mexican",
    rating: 4.9,
    reviews: 178,
    experience: "10 years",
    verified: true,
    price: 1199,
    avatar: "👨‍🍳",
    description:
      "Vibrant Mexican and Latin flavors. Authentic tacos, enchiladas, and traditional family recipes.",
  },
  {
    id: 5,
    name: "Chef Priya Sharma",
    specialty: "South Indian & Coastal",
    cuisine: "indian",
    rating: 5.0,
    reviews: 189,
    experience: "7 years",
    verified: true,
    price: 999,
    avatar: "👩‍🍳",
    description:
      "Coastal delicacies and South Indian classics. Specializing in dosas, idlis, and seafood preparations.",
  },
  {
    id: 6,
    name: "Chef Thai Nguyen",
    specialty: "Thai & Vietnamese",
    cuisine: "thai",
    rating: 4.7,
    reviews: 134,
    experience: "9 years",
    verified: true,
    price: 1149,
    avatar: "👨‍🍳",
    description:
      "Authentic Southeast Asian cuisine with balanced flavors. Expert in pad thai, pho, and curries.",
  },
]);

// Auth Modal
function AuthModal({ show, onClose, onSuccess }) {
  const [isLogin, setIsLogin] = useState(true);
  const [formData, setFormData] = useState({
    email: "",
    password: "",
    name: "",
    phone: "",
    location: "",
  });
  const [error, setError] = useState("");

  const handleSubmit = (e) => {
    e.preventDefault();
    setError("");

    if (isLogin) {
      // LOGIN
      const user = USERS.find(
        (u) => u.email === formData.email && u.password === formData.password,
      );
      if (user) {
        CURRENT_USER = user;
        Storage.save("currentUser", user);
        alert("✅ Welcome back, " + user.name + "!");
        onSuccess(user);
        onClose();
        setFormData({
          email: "",
          password: "",
          name: "",
          phone: "",
          location: "",
        });
      } else {
        setError("Invalid email or password!");
      }
    } else {
      // SIGNUP
      if (
        !formData.name ||
        !formData.email ||
        !formData.password ||
        !formData.phone
      ) {
        setError("Please fill all required fields!");
        return;
      }

      const exists = USERS.find((u) => u.email === formData.email);
      if (exists) {
        setError("Email already registered! Please login.");
        return;
      }

      const newUser = {
        id: Date.now(),
        name: formData.name,
        email: formData.email,
        password: formData.password,
        phone: formData.phone,
        location: formData.location || "India",
        isCook: false,
      };

      USERS.push(newUser);
      Storage.save("users", USERS);

      CURRENT_USER = newUser;
      Storage.save("currentUser", newUser);

      alert("✅ Account created successfully! Welcome, " + newUser.name + "!");
      onSuccess(newUser);
      onClose();
      setFormData({
        email: "",
        password: "",
        name: "",
        phone: "",
        location: "",
      });
    }
  };

  if (!show) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-60 z-50 flex items-center justify-center p-4">
      <div className="bg-white rounded-2xl max-w-md w-full p-8 shadow-2xl">
        <div className="flex justify-between items-center mb-6">
          <h2 className="text-3xl font-bold text-gray-900">
            {isLogin ? "Welcome Back!" : "Create Account"}
          </h2>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-orange-600 text-4xl leading-none"
          >
            ×
          </button>
        </div>

        {isLogin && (
          <div className="mb-6 p-4 bg-blue-50 border-l-4 border-blue-500 rounded">
            <p className="font-bold text-blue-900 mb-1">📝 Test Account:</p>
            <p className="text-blue-700 text-sm">Email: test@test.com</p>
            <p className="text-blue-700 text-sm">Password: 123</p>
          </div>
        )}

        {error && (
          <div className="mb-4 p-4 bg-red-50 border-l-4 border-red-500 rounded">
            <p className="text-red-700 font-semibold">❌ {error}</p>
          </div>
        )}

        <form onSubmit={handleSubmit} className="space-y-4">
          {!isLogin && (
            <div>
              <label className="block text-gray-700 font-bold mb-2">
                Full Name *
              </label>
              <input
                type="text"
                value={formData.name}
                onChange={(e) =>
                  setFormData({ ...formData, name: e.target.value })
                }
                className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none text-lg"
                placeholder="Enter your name"
              />
            </div>
          )}

          <div>
            <label className="block text-gray-700 font-bold mb-2">
              Email Address *
            </label>
            <input
              type="email"
              value={formData.email}
              onChange={(e) =>
                setFormData({ ...formData, email: e.target.value })
              }
              className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none text-lg"
              placeholder="your@email.com"
              required
            />
          </div>

          <div>
            <label className="block text-gray-700 font-bold mb-2">
              Password *
            </label>
            <input
              type="password"
              value={formData.password}
              onChange={(e) =>
                setFormData({ ...formData, password: e.target.value })
              }
              className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none text-lg"
              placeholder="Enter password"
              required
            />
          </div>

          {!isLogin && (
            <>
              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Phone Number *
                </label>
                <input
                  type="tel"
                  value={formData.phone}
                  onChange={(e) =>
                    setFormData({ ...formData, phone: e.target.value })
                  }
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none text-lg"
                  placeholder="9876543210"
                />
              </div>
              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Location (Optional)
                </label>
                <input
                  type="text"
                  value={formData.location}
                  onChange={(e) =>
                    setFormData({ ...formData, location: e.target.value })
                  }
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none text-lg"
                  placeholder="City, State"
                />
              </div>
            </>
          )}

          <button
            type="submit"
            className="w-full px-6 py-4 bg-gradient-to-r from-orange-600 to-orange-500 text-white rounded-lg font-bold text-lg hover:from-orange-700 hover:to-orange-600 transform hover:scale-105 transition-all shadow-lg"
          >
            {isLogin ? "SIGN IN NOW" : "CREATE ACCOUNT"}
          </button>
        </form>

        <div className="mt-6 text-center">
          <button
            onClick={() => {
              setIsLogin(!isLogin);
              setError("");
              setFormData({
                email: "",
                password: "",
                name: "",
                phone: "",
                location: "",
              });
            }}
            className="text-orange-600 hover:text-orange-700 font-bold text-lg underline"
          >
            {isLogin
              ? "Don't have an account? Sign Up"
              : "Already have an account? Sign In"}
          </button>
        </div>
      </div>
    </div>
  );
}

// Become Cook Modal
function BecomeCookModal({ show, onClose, user, onSuccess }) {
  const [formData, setFormData] = useState({
    specialty: "",
    cuisine: "",
    experience: "",
    price: "",
    description: "",
    avatar: "👨‍🍳",
  });

  const handleSubmit = (e) => {
    e.preventDefault();

    if (!formData.specialty || !formData.cuisine || !formData.price) {
      alert("❌ Please fill all required fields!");
      return;
    }

    const newCook = {
      id: Date.now(),
      name: user.name,
      specialty: formData.specialty,
      cuisine: formData.cuisine,
      rating: 5.0,
      reviews: 0,
      experience: formData.experience || "New Cook",
      verified: false,
      price: parseInt(formData.price),
      avatar: formData.avatar,
      description:
        formData.description ||
        `Delicious ${formData.cuisine} cuisine prepared with love and expertise.`,
    };

    COOKS.push(newCook);
    Storage.save("cooks", COOKS);

    user.isCook = true;
    user.cookId = newCook.id;
    const userIndex = USERS.findIndex((u) => u.id === user.id);
    USERS[userIndex] = user;
    Storage.save("users", USERS);
    Storage.save("currentUser", user);

    alert("🎉 Congratulations! You're now a registered cook on ServeMyMeal!");
    onSuccess(user);
    onClose();
  };

  if (!show) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-60 z-50 flex items-center justify-center p-4 overflow-y-auto">
      <div className="bg-white rounded-2xl max-w-2xl w-full p-8 my-8 shadow-2xl">
        <div className="flex justify-between items-center mb-6">
          <h2 className="text-3xl font-bold text-gray-900">Become a Cook</h2>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-orange-600 text-4xl leading-none"
          >
            ×
          </button>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div>
            <label className="block text-gray-700 font-bold mb-3">
              Choose Your Avatar
            </label>
            <div className="flex gap-4">
              {["👨‍🍳", "👩‍🍳", "🧑‍🍳"].map((a) => (
                <button
                  key={a}
                  type="button"
                  onClick={() => setFormData({ ...formData, avatar: a })}
                  className={`text-7xl p-6 border-4 rounded-xl transition-all ${
                    formData.avatar === a
                      ? "border-orange-600 bg-orange-50 shadow-lg scale-110"
                      : "border-gray-300 hover:border-orange-300"
                  }`}
                >
                  {a}
                </button>
              ))}
            </div>
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-gray-700 font-bold mb-2">
                Specialty *
              </label>
              <input
                type="text"
                value={formData.specialty}
                onChange={(e) =>
                  setFormData({ ...formData, specialty: e.target.value })
                }
                placeholder="e.g., Italian & Mediterranean"
                className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                required
              />
            </div>

            <div>
              <label className="block text-gray-700 font-bold mb-2">
                Cuisine Type *
              </label>
              <select
                value={formData.cuisine}
                onChange={(e) =>
                  setFormData({ ...formData, cuisine: e.target.value })
                }
                className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                required
              >
                <option value="">Select Cuisine</option>
                <option value="indian">Indian</option>
                <option value="italian">Italian</option>
                <option value="chinese">Chinese</option>
                <option value="mexican">Mexican</option>
                <option value="thai">Thai</option>
                <option value="mediterranean">Mediterranean</option>
                <option value="fusion">Fusion</option>
              </select>
            </div>
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-gray-700 font-bold mb-2">
                Experience
              </label>
              <input
                type="text"
                value={formData.experience}
                onChange={(e) =>
                  setFormData({ ...formData, experience: e.target.value })
                }
                placeholder="e.g., 5 years"
                className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
              />
            </div>

            <div>
              <label className="block text-gray-700 font-bold mb-2">
                Price per Meal (₹) *
              </label>
              <input
                type="number"
                value={formData.price}
                onChange={(e) =>
                  setFormData({ ...formData, price: e.target.value })
                }
                placeholder="e.g., 999"
                min="100"
                className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                required
              />
            </div>
          </div>

          <div>
            <label className="block text-gray-700 font-bold mb-2">
              Description
            </label>
            <textarea
              value={formData.description}
              onChange={(e) =>
                setFormData({ ...formData, description: e.target.value })
              }
              placeholder="Tell customers about your cooking style, specialties, and what makes your food special..."
              rows="4"
              className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
            />
          </div>

          <button
            type="submit"
            className="w-full px-6 py-4 bg-gradient-to-r from-orange-600 to-orange-500 text-white rounded-lg font-bold text-lg hover:from-orange-700 hover:to-orange-600 shadow-lg transform hover:scale-105 transition-all"
          >
            BECOME A COOK
          </button>
        </form>
      </div>
    </div>
  );
}

// Header
function Header({ user, onSignIn, onLogout, onBecomeCook, onDashboard }) {
  const [mobileMenu, setMobileMenu] = useState(false);

  return (
    <header className="bg-white border-b-2 border-orange-200 sticky top-0 z-50 shadow-md">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center h-20">
          <div className="flex items-center">
            <h1
              className="text-3xl sm:text-4xl font-bold italic bg-gradient-to-r from-orange-600 to-orange-500 bg-clip-text text-transparent cursor-pointer"
              onClick={() => window.location.reload()}
            >
              ServeMyMeal
            </h1>
          </div>

          {/* Desktop Menu */}
          <div className="hidden md:flex items-center space-x-8">
            <a
              href="#home"
              className="text-gray-700 hover:text-orange-600 font-semibold transition-colors"
            >
              Home
            </a>
            <a
              href="#features"
              className="text-gray-700 hover:text-orange-600 font-semibold transition-colors"
            >
              How It Works
            </a>
            <a
              href="#cooks"
              className="text-gray-700 hover:text-orange-600 font-semibold transition-colors"
            >
              Browse Cooks
            </a>
            {user && (
              <button
                onClick={onDashboard}
                className="text-gray-700 hover:text-orange-600 font-semibold transition-colors"
              >
                My Bookings
              </button>
            )}
          </div>

          <div className="hidden md:flex items-center gap-4">
            {user ? (
              <>
                {!user.isCook && (
                  <button
                    onClick={onBecomeCook}
                    className="px-5 py-2.5 border-2 border-orange-600 text-orange-600 rounded-lg font-bold hover:bg-orange-600 hover:text-white transition-all"
                  >
                    Become a Cook
                  </button>
                )}
                <span className="text-gray-700 font-semibold">
                  👋 Hi, {user.name}!
                </span>
                <button
                  onClick={onLogout}
                  className="px-5 py-2.5 bg-gradient-to-r from-orange-600 to-orange-500 text-white rounded-lg font-bold hover:from-orange-700 hover:to-orange-600 shadow-md transition-all"
                >
                  Logout
                </button>
              </>
            ) : (
              <button
                onClick={onSignIn}
                className="px-6 py-3 bg-gradient-to-r from-orange-600 to-orange-500 text-white rounded-lg font-bold text-lg hover:from-orange-700 hover:to-orange-600 shadow-lg transform hover:scale-105 transition-all"
              >
                SIGN IN
              </button>
            )}
          </div>

          {/* Mobile Menu Button */}
          <button
            onClick={() => setMobileMenu(!mobileMenu)}
            className="md:hidden text-3xl text-gray-700"
          >
            ☰
          </button>
        </div>

        {/* Mobile Menu */}
        {mobileMenu && (
          <div className="md:hidden py-4 space-y-3 border-t border-gray-200">
            <a
              href="#home"
              className="block text-gray-700 hover:text-orange-600 font-semibold py-2"
            >
              Home
            </a>
            <a
              href="#features"
              className="block text-gray-700 hover:text-orange-600 font-semibold py-2"
            >
              How It Works
            </a>
            <a
              href="#cooks"
              className="block text-gray-700 hover:text-orange-600 font-semibold py-2"
            >
              Browse Cooks
            </a>
            {user ? (
              <>
                <button
                  onClick={() => {
                    onDashboard();
                    setMobileMenu(false);
                  }}
                  className="block w-full text-left text-gray-700 hover:text-orange-600 font-semibold py-2"
                >
                  My Bookings
                </button>
                {!user.isCook && (
                  <button
                    onClick={() => {
                      onBecomeCook();
                      setMobileMenu(false);
                    }}
                    className="w-full px-4 py-2 border-2 border-orange-600 text-orange-600 rounded-lg font-bold"
                  >
                    Become a Cook
                  </button>
                )}
                <button
                  onClick={() => {
                    onLogout();
                    setMobileMenu(false);
                  }}
                  className="w-full px-4 py-2 bg-orange-600 text-white rounded-lg font-bold"
                >
                  Logout
                </button>
              </>
            ) : (
              <button
                onClick={() => {
                  onSignIn();
                  setMobileMenu(false);
                }}
                className="w-full px-6 py-3 bg-orange-600 text-white rounded-lg font-bold text-lg"
              >
                SIGN IN
              </button>
            )}
          </div>
        )}
      </div>
    </header>
  );
}

// Hero Section
function Hero() {
  return (
    <section
      className="relative bg-gradient-to-br from-orange-600 via-orange-500 to-orange-400 text-white py-20 sm:py-32"
      id="home"
    >
      <div className="absolute inset-0 overflow-hidden opacity-10">
        <div className="absolute -top-40 -right-40 w-96 h-96 bg-white rounded-full blur-3xl"></div>
        <div className="absolute -bottom-40 -left-40 w-96 h-96 bg-white rounded-full blur-3xl"></div>
      </div>

      <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center">
          <h1 className="text-5xl sm:text-6xl lg:text-7xl font-extrabold mb-6 leading-tight">
            Delicious Home-Cooked
            <br />
            Meals, Made Just for You
          </h1>
          <p className="text-xl sm:text-2xl mb-8 max-w-3xl mx-auto leading-relaxed opacity-95">
            Book professional home cooks for authentic, personalized dining
            experiences. From family dinners to special occasions, we bring
            restaurant-quality food to your home.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
            <button
              onClick={() =>
                document
                  .getElementById("cooks")
                  ?.scrollIntoView({ behavior: "smooth" })
              }
              className="px-8 py-4 bg-white text-orange-600 rounded-lg font-bold text-xl hover:bg-gray-100 shadow-xl transform hover:scale-105 transition-all"
            >
              🔍 Browse Cooks
            </button>
            <button
              onClick={() =>
                document
                  .getElementById("features")
                  ?.scrollIntoView({ behavior: "smooth" })
              }
              className="px-8 py-4 border-2 border-white text-white rounded-lg font-bold text-xl hover:bg-white hover:text-orange-600 shadow-xl transform hover:scale-105 transition-all"
            >
              📖 How It Works
            </button>
          </div>

          {/* Stats */}
          <div className="mt-16 grid grid-cols-3 gap-8 max-w-4xl mx-auto">
            <div className="text-center">
              <div className="text-4xl sm:text-5xl font-bold mb-2">500+</div>
              <div className="text-lg opacity-90">Happy Customers</div>
            </div>
            <div className="text-center">
              <div className="text-4xl sm:text-5xl font-bold mb-2">50+</div>
              <div className="text-lg opacity-90">Expert Cooks</div>
            </div>
            <div className="text-center">
              <div className="text-4xl sm:text-5xl font-bold mb-2">4.9★</div>
              <div className="text-lg opacity-90">Average Rating</div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}

// Features Section
function Features() {
  const features = [
    {
      icon: "🔍",
      title: "Browse & Select",
      description:
        "Explore our curated selection of verified professional cooks. View detailed profiles, read reviews, and choose the perfect chef for your occasion.",
    },
    {
      icon: "📅",
      title: "Schedule & Customize",
      description:
        "Pick your preferred date, time, and meal type. Share dietary preferences and special requests to personalize your dining experience.",
    },
    {
      icon: "💳",
      title: "Secure Payment",
      description:
        "Pay safely through our secure platform with multiple payment options. Transparent pricing with no hidden charges.",
    },
    {
      icon: "🍳",
      title: "Enjoy Your Meal",
      description:
        "Relax while our professional cook prepares fresh, delicious meals right in your kitchen. Quality ingredients, authentic taste.",
    },
  ];

  return (
    <section className="py-20 sm:py-24 bg-white" id="features">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-16">
          <h2 className="text-4xl sm:text-5xl font-bold text-gray-900 mb-4">
            How ServeMyMeal Works
          </h2>
          <p className="text-xl text-gray-600 max-w-2xl mx-auto">
            Book professional home cooks in four simple steps
          </p>
        </div>

        <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-8">
          {features.map((feature, index) => (
            <div
              key={index}
              className="bg-gradient-to-br from-orange-50 to-amber-50 rounded-2xl p-8 text-center hover:shadow-2xl hover:-translate-y-2 transition-all duration-300 border-2 border-orange-100"
            >
              <div className="w-24 h-24 bg-gradient-to-br from-orange-600 to-orange-500 rounded-full flex items-center justify-center text-5xl mx-auto mb-6 shadow-lg">
                {feature.icon}
              </div>
              <h3 className="text-2xl font-bold text-gray-900 mb-4">
                {feature.title}
              </h3>
              <p className="text-gray-600 leading-relaxed">
                {feature.description}
              </p>
            </div>
          ))}
        </div>
      </div>
    </section>
  );
}

// Cook Card
function CookCard({ cook, onBook, user, onSignIn }) {
  return (
    <div className="bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl hover:-translate-y-2 transition-all duration-300 border-2 border-gray-100">
      <div className="relative h-64 bg-gradient-to-br from-orange-400 to-orange-500 flex items-center justify-center">
        <div className="text-9xl">{cook.avatar}</div>
        {cook.verified && (
          <div className="absolute top-4 right-4 bg-green-600 text-white px-4 py-2 rounded-full text-sm font-bold flex items-center gap-1 shadow-lg">
            <span>✓</span> Verified
          </div>
        )}
      </div>

      <div className="p-6">
        <div className="flex justify-between items-start mb-4">
          <div className="flex-1">
            <h3 className="text-2xl font-bold text-gray-900 mb-1">
              {cook.name}
            </h3>
            <p className="text-orange-600 font-semibold mb-2">
              {cook.specialty}
            </p>
          </div>
          <div className="text-right">
            <div className="text-yellow-500 text-xl mb-1">
              {"★".repeat(Math.floor(cook.rating))}
            </div>
            <p className="text-sm text-gray-600 font-semibold">
              {cook.rating} ({cook.reviews} reviews)
            </p>
          </div>
        </div>

        <div className="flex items-center gap-4 mb-4 text-sm text-gray-600">
          <div className="flex items-center gap-1">
            <span>⏱️</span>
            <span className="font-medium">{cook.experience}</span>
          </div>
          <div className="flex items-center gap-1">
            <span>🍽️</span>
            <span className="capitalize font-medium">{cook.cuisine}</span>
          </div>
        </div>

        <p className="text-gray-600 mb-6 leading-relaxed">{cook.description}</p>

        <div className="flex justify-between items-center pt-4 border-t-2 border-gray-100">
          <div>
            <span className="text-3xl font-bold bg-gradient-to-r from-orange-600 to-orange-500 bg-clip-text text-transparent">
              ₹{cook.price}
            </span>
            <span className="text-gray-600 font-medium"> /meal</span>
          </div>
          <button
            onClick={() => (user ? onBook(cook) : onSignIn())}
            className="px-6 py-3 bg-gradient-to-r from-orange-600 to-orange-500 text-white rounded-lg font-bold hover:from-orange-700 hover:to-orange-600 shadow-md hover:shadow-lg transform hover:scale-105 transition-all"
          >
            Book Now
          </button>
        </div>
      </div>
    </div>
  );
}

// Booking Modal
function BookingModal({ show, onClose, cook, user }) {
  const [step, setStep] = useState(1);
  const [formData, setFormData] = useState({
    date: "",
    time: "",
    mealTypes: [],
    people: 4,
    address: "",
    city: "",
    pincode: "",
    paymentMethod: "",
  });

  const toggleMealType = (type) => {
    setFormData((prev) => ({
      ...prev,
      mealTypes: prev.mealTypes.includes(type)
        ? prev.mealTypes.filter((t) => t !== type)
        : [...prev.mealTypes, type],
    }));
  };

  const handleBook = () => {
    const booking = {
      id: Date.now(),
      userId: user.id,
      userName: user.name,
      cook: cook,
      date: formData.date,
      time: formData.time,
      mealTypes: formData.mealTypes,
      people: formData.people,
      address: formData.address,
      city: formData.city,
      pincode: formData.pincode,
      paymentMethod: formData.paymentMethod,
      total: cook.price * formData.people,
      status: "confirmed",
      createdAt: new Date().toISOString(),
    };

    BOOKINGS.push(booking);
    Storage.save("bookings", BOOKINGS);

    setStep(4);
  };

  const resetAndClose = () => {
    setStep(1);
    setFormData({
      date: "",
      time: "",
      mealTypes: [],
      people: 4,
      address: "",
      city: "",
      pincode: "",
      paymentMethod: "",
    });
    onClose();
  };

  if (!show || !cook) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-60 z-50 flex items-center justify-center p-4 overflow-y-auto">
      <div className="bg-white rounded-2xl max-w-3xl w-full max-h-[90vh] overflow-y-auto shadow-2xl">
        <div className="sticky top-0 bg-white border-b-2 border-gray-200 p-6 flex justify-between items-center rounded-t-2xl z-10">
          <h2 className="text-3xl font-bold text-gray-900">Book Your Cook</h2>
          <button
            onClick={resetAndClose}
            className="text-gray-500 hover:text-orange-600 text-4xl leading-none"
          >
            ×
          </button>
        </div>

        <div className="p-6">
          {/* Step Indicator */}
          <div className="flex justify-between mb-10 relative">
            <div className="absolute top-5 left-0 right-0 h-1 bg-gray-200 -z-10"></div>
            <div
              className="absolute top-5 left-0 h-1 bg-orange-600 -z-10 transition-all"
              style={{ width: `${((step - 1) / 3) * 100}%` }}
            ></div>

            {[1, 2, 3, 4].map((s) => (
              <div key={s} className="flex flex-col items-center">
                <div
                  className={`w-12 h-12 rounded-full flex items-center justify-center font-bold text-lg transition-all ${
                    s < step
                      ? "bg-green-600 text-white"
                      : s === step
                        ? "bg-orange-600 text-white shadow-lg scale-110"
                        : "bg-gray-200 text-gray-600"
                  }`}
                >
                  {s < step ? "✓" : s}
                </div>
                <div
                  className={`mt-2 text-sm font-semibold text-center ${s === step ? "text-gray-900" : "text-gray-500"}`}
                >
                  {s === 1 && "Details"}
                  {s === 2 && "Address"}
                  {s === 3 && "Payment"}
                  {s === 4 && "Done"}
                </div>
              </div>
            ))}
          </div>

          {/* Step 1: Service Details */}
          {step === 1 && (
            <div className="space-y-6">
              <div className="grid sm:grid-cols-2 gap-4">
                <div>
                  <label className="block text-gray-700 font-bold mb-2">
                    Date *
                  </label>
                  <input
                    type="date"
                    value={formData.date}
                    onChange={(e) =>
                      setFormData({ ...formData, date: e.target.value })
                    }
                    min={new Date().toISOString().split("T")[0]}
                    className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                  />
                </div>
                <div>
                  <label className="block text-gray-700 font-bold mb-2">
                    Time *
                  </label>
                  <input
                    type="time"
                    value={formData.time}
                    onChange={(e) =>
                      setFormData({ ...formData, time: e.target.value })
                    }
                    className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                  />
                </div>
              </div>

              <div>
                <label className="block text-gray-700 font-bold mb-3">
                  Meal Type *
                </label>
                <div className="grid grid-cols-2 sm:grid-cols-4 gap-3">
                  {[
                    { type: "breakfast", icon: "🌅" },
                    { type: "lunch", icon: "🌞" },
                    { type: "dinner", icon: "🌙" },
                    { type: "snacks", icon: "🍪" },
                  ].map(({ type, icon }) => (
                    <button
                      key={type}
                      type="button"
                      onClick={() => toggleMealType(type)}
                      className={`p-4 border-2 rounded-lg text-center transition-all ${
                        formData.mealTypes.includes(type)
                          ? "border-orange-600 bg-orange-50 shadow-md scale-105"
                          : "border-gray-300 hover:border-orange-300"
                      }`}
                    >
                      <div className="text-3xl mb-2">{icon}</div>
                      <div className="capitalize font-semibold text-gray-900">
                        {type}
                      </div>
                    </button>
                  ))}
                </div>
              </div>

              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Number of People *
                </label>
                <input
                  type="number"
                  min="1"
                  max="50"
                  value={formData.people}
                  onChange={(e) =>
                    setFormData({
                      ...formData,
                      people: parseInt(e.target.value) || 1,
                    })
                  }
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none text-lg"
                />
              </div>

              <div className="flex justify-end pt-4">
                <button
                  onClick={() => {
                    if (
                      !formData.date ||
                      !formData.time ||
                      formData.mealTypes.length === 0
                    ) {
                      alert("❌ Please fill all required fields!");
                    } else {
                      setStep(2);
                    }
                  }}
                  className="px-8 py-3 bg-orange-600 text-white rounded-lg font-bold hover:bg-orange-700 shadow-md"
                >
                  Continue →
                </button>
              </div>
            </div>
          )}

          {/* Step 2: Address */}
          {step === 2 && (
            <div className="space-y-6">
              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Full Address *
                </label>
                <textarea
                  value={formData.address}
                  onChange={(e) =>
                    setFormData({ ...formData, address: e.target.value })
                  }
                  rows="3"
                  placeholder="House/Flat No., Building Name, Street..."
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                />
              </div>

              <div className="grid sm:grid-cols-2 gap-4">
                <div>
                  <label className="block text-gray-700 font-bold mb-2">
                    City *
                  </label>
                  <input
                    type="text"
                    value={formData.city}
                    onChange={(e) =>
                      setFormData({ ...formData, city: e.target.value })
                    }
                    placeholder="Enter city"
                    className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                  />
                </div>
                <div>
                  <label className="block text-gray-700 font-bold mb-2">
                    Pincode *
                  </label>
                  <input
                    type="text"
                    value={formData.pincode}
                    onChange={(e) =>
                      setFormData({ ...formData, pincode: e.target.value })
                    }
                    placeholder="6-digit pincode"
                    className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-orange-500 focus:outline-none"
                  />
                </div>
              </div>

              <div className="flex justify-between pt-4">
                <button
                  onClick={() => setStep(1)}
                  className="px-8 py-3 border-2 border-orange-600 text-orange-600 rounded-lg font-bold hover:bg-orange-50"
                >
                  ← Back
                </button>
                <button
                  onClick={() => {
                    if (
                      !formData.address ||
                      !formData.city ||
                      !formData.pincode
                    ) {
                      alert("❌ Please fill all address fields!");
                    } else {
                      setStep(3);
                    }
                  }}
                  className="px-8 py-3 bg-orange-600 text-white rounded-lg font-bold hover:bg-orange-700 shadow-md"
                >
                  Continue →
                </button>
              </div>
            </div>
          )}

          {/* Step 3: Payment */}
          {step === 3 && (
            <div className="space-y-6">
              <div className="bg-gradient-to-br from-orange-50 to-amber-50 rounded-xl p-6 border-2 border-orange-200">
                <h3 className="text-2xl font-bold text-gray-900 mb-4">
                  Booking Summary
                </h3>
                <div className="space-y-3">
                  <div className="flex justify-between py-2 border-b border-orange-200">
                    <span className="text-gray-700 font-semibold">Cook:</span>
                    <span className="font-bold text-gray-900">{cook.name}</span>
                  </div>
                  <div className="flex justify-between py-2 border-b border-orange-200">
                    <span className="text-gray-700 font-semibold">
                      Date & Time:
                    </span>
                    <span className="font-bold text-gray-900">
                      {formData.date} at {formData.time}
                    </span>
                  </div>
                  <div className="flex justify-between py-2 border-b border-orange-200">
                    <span className="text-gray-700 font-semibold">
                      Meal Types:
                    </span>
                    <span className="font-bold text-gray-900 capitalize">
                      {formData.mealTypes.join(", ")}
                    </span>
                  </div>
                  <div className="flex justify-between py-2 border-b border-orange-200">
                    <span className="text-gray-700 font-semibold">
                      Number of People:
                    </span>
                    <span className="font-bold text-gray-900">
                      {formData.people}
                    </span>
                  </div>
                  <div className="flex justify-between py-2 border-b border-orange-200">
                    <span className="text-gray-700 font-semibold">
                      Price per Person:
                    </span>
                    <span className="font-bold text-gray-900">
                      ₹{cook.price}
                    </span>
                  </div>
                  <div className="flex justify-between pt-3 text-xl">
                    <span className="font-bold text-gray-900">
                      Total Amount:
                    </span>
                    <span className="font-bold text-3xl bg-gradient-to-r from-orange-600 to-orange-500 bg-clip-text text-transparent">
                      ₹{cook.price * formData.people}
                    </span>
                  </div>
                </div>
              </div>

              <div>
                <label className="block text-gray-700 font-bold mb-3">
                  Select Payment Method *
                </label>
                <div className="grid sm:grid-cols-3 gap-4">
                  {[
                    { value: "card", icon: "💳", label: "Card" },
                    { value: "upi", icon: "📱", label: "UPI" },
                    { value: "wallet", icon: "👛", label: "Wallet" },
                  ].map((method) => (
                    <button
                      key={method.value}
                      type="button"
                      onClick={() =>
                        setFormData({
                          ...formData,
                          paymentMethod: method.value,
                        })
                      }
                      className={`p-6 border-2 rounded-lg text-center transition-all ${
                        formData.paymentMethod === method.value
                          ? "border-orange-600 bg-orange-50 shadow-lg scale-105"
                          : "border-gray-300 hover:border-orange-300"
                      }`}
                    >
                      <div className="text-5xl mb-2">{method.icon}</div>
                      <div className="font-bold text-gray-900">
                        {method.label}
                      </div>
                    </button>
                  ))}
                </div>
              </div>

              <div className="flex justify-between pt-4">
                <button
                  onClick={() => setStep(2)}
                  className="px-8 py-3 border-2 border-orange-600 text-orange-600 rounded-lg font-bold hover:bg-orange-50"
                >
                  ← Back
                </button>
                <button
                  onClick={() => {
                    if (!formData.paymentMethod) {
                      alert("❌ Please select a payment method!");
                    } else {
                      handleBook();
                    }
                  }}
                  className="px-8 py-3 bg-gradient-to-r from-green-600 to-green-500 text-white rounded-lg font-bold hover:from-green-700 hover:to-green-600 shadow-lg"
                >
                  Confirm Booking ✓
                </button>
              </div>
            </div>
          )}

          {/* Step 4: Confirmation */}
          {step === 4 && (
            <div className="text-center py-8">
              <div className="text-8xl mb-6 animate-bounce">✅</div>
              <h3 className="text-4xl font-bold text-gray-900 mb-4">
                Booking Confirmed!
              </h3>
              <p className="text-xl text-gray-600 mb-8">
                Your booking has been successfully confirmed. You will receive a
                confirmation email shortly.
              </p>

              <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl p-8 max-w-2xl mx-auto mb-8 border-2 border-green-200">
                <h4 className="text-2xl font-bold text-gray-900 mb-6">
                  Booking Details
                </h4>
                <div className="space-y-3 text-left">
                  <p className="text-lg">
                    <strong>Cook:</strong> {cook.name} {cook.avatar}
                  </p>
                  <p className="text-lg">
                    <strong>Date & Time:</strong> {formData.date} at{" "}
                    {formData.time}
                  </p>
                  <p className="text-lg">
                    <strong>Meals:</strong> {formData.mealTypes.join(", ")}
                  </p>
                  <p className="text-lg">
                    <strong>People:</strong> {formData.people}
                  </p>
                  <p className="text-lg">
                    <strong>Address:</strong> {formData.address},{" "}
                    {formData.city} - {formData.pincode}
                  </p>
                  <p className="text-xl pt-4 border-t-2 border-green-300">
                    <strong>Total Paid:</strong>
                    <span className="text-2xl font-bold text-green-600 ml-2">
                      ₹{cook.price * formData.people}
                    </span>
                  </p>
                </div>
              </div>

              <button
                onClick={resetAndClose}
                className="px-10 py-4 bg-gradient-to-r from-orange-600 to-orange-500 text-white rounded-lg font-bold text-lg hover:from-orange-700 hover:to-orange-600 shadow-lg transform hover:scale-105 transition-all"
              >
                Done
              </button>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}

// Dashboard
function Dashboard({ user, onBack }) {
  const [tab, setTab] = useState("upcoming");
  const userBookings = BOOKINGS.filter((b) => b.userId === user.id);
  const today = new Date().toISOString().split("T")[0];
  const upcoming = userBookings
    .filter((b) => b.date >= today)
    .sort((a, b) => new Date(a.date) - new Date(b.date));
  const past = userBookings
    .filter((b) => b.date < today)
    .sort((a, b) => new Date(b.date) - new Date(a.date));

  return (
    <div className="py-12 min-h-screen bg-gradient-to-br from-orange-50 to-amber-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-12">
          <h2 className="text-5xl font-bold text-gray-900 mb-4">
            My Dashboard
          </h2>
          <p className="text-xl text-gray-600">Welcome back, {user.name}! 👋</p>
        </div>

        <div className="bg-white rounded-2xl shadow-xl p-6 mb-8">
          <div className="flex flex-wrap gap-4">
            {["upcoming", "past", "profile"].map((t) => (
              <button
                key={t}
                onClick={() => setTab(t)}
                className={`px-6 py-3 rounded-lg font-bold capitalize transition-all ${
                  tab === t
                    ? "bg-gradient-to-r from-orange-600 to-orange-500 text-white shadow-lg scale-105"
                    : "text-gray-700 hover:bg-gray-100"
                }`}
              >
                {t === "upcoming" && "📅 Upcoming"}
                {t === "past" && "📜 Past"}
                {t === "profile" && "👤 Profile"}
              </button>
            ))}
          </div>
        </div>

        {tab === "upcoming" && (
          <div className="space-y-6">
            {upcoming.length === 0 ? (
              <div className="bg-white rounded-2xl shadow-xl p-12 text-center">
                <div className="text-8xl mb-6">📅</div>
                <h3 className="text-3xl font-bold text-gray-900 mb-4">
                  No Upcoming Bookings
                </h3>
                <p className="text-xl text-gray-600 mb-8">
                  Book a cook to get started!
                </p>
                <button
                  onClick={onBack}
                  className="px-8 py-4 bg-gradient-to-r from-orange-600 to-orange-500 text-white rounded-lg font-bold text-lg hover:from-orange-700 hover:to-orange-600 shadow-lg"
                >
                  Browse Cooks
                </button>
              </div>
            ) : (
              upcoming.map((booking) => (
                <div
                  key={booking.id}
                  className="bg-white rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all"
                >
                  <div className="grid sm:grid-cols-3 gap-6 items-center">
                    <div className="flex items-center gap-4">
                      <div className="text-6xl">{booking.cook.avatar}</div>
                      <div>
                        <h3 className="text-2xl font-bold text-gray-900">
                          {booking.cook.name}
                        </h3>
                        <p className="text-orange-600 font-semibold">
                          {booking.date} at {booking.time}
                        </p>
                        <p className="text-gray-600">
                          {booking.people} people •{" "}
                          {booking.mealTypes.join(", ")}
                        </p>
                      </div>
                    </div>
                    <div className="text-center">
                      <span className="inline-block px-6 py-3 rounded-full text-sm font-bold bg-green-100 text-green-700 border-2 border-green-300">
                        ✓ Confirmed
                      </span>
                    </div>
                    <div className="text-right">
                      <p className="text-4xl font-bold bg-gradient-to-r from-orange-600 to-orange-500 bg-clip-text text-transparent">
                        ₹{booking.total}
                      </p>
                    </div>
                  </div>
                </div>
              ))
            )}
          </div>
        )}

        {tab === "past" && (
          <div className="space-y-6">
            {past.length === 0 ? (
              <div className="bg-white rounded-2xl shadow-xl p-12 text-center">
                <div className="text-8xl mb-6">📜</div>
                <h3 className="text-3xl font-bold text-gray-900 mb-4">
                  No Past Bookings
                </h3>
                <p className="text-xl text-gray-600">
                  Your booking history will appear here
                </p>
              </div>
            ) : (
              past.map((booking) => (
                <div
                  key={booking.id}
                  className="bg-white rounded-2xl shadow-lg p-6"
                >
                  <div className="grid sm:grid-cols-3 gap-6 items-center">
                    <div className="flex items-center gap-4">
                      <div className="text-6xl">{booking.cook.avatar}</div>
                      <div>
                        <h3 className="text-2xl font-bold text-gray-900">
                          {booking.cook.name}
                        </h3>
                        <p className="text-gray-600">{booking.date}</p>
                        <p className="text-sm text-gray-500">
                          {booking.mealTypes.join(", ")}
                        </p>
                      </div>
                    </div>
                    <div className="text-center">
                      <span className="inline-block px-6 py-3 rounded-full text-sm font-bold bg-gray-100 text-gray-700 mb-3">
                        Completed
                      </span>
                      {booking.rating ? (
                        <div className="text-center">
                          <div className="text-yellow-500 text-2xl mb-1">
                            {"★".repeat(booking.rating)}
                            {"☆".repeat(5 - booking.rating)}
                          </div>
                          <p className="text-sm text-gray-600 font-semibold">
                            Your Rating
                          </p>
                        </div>
                      ) : (
                        <button
                          onClick={() => {
                            const rating = prompt(
                              "⭐ Rate this cook (1-5 stars):",
                            );
                            if (rating && rating >= 1 && rating <= 5) {
                              booking.rating = parseInt(rating);
                              const review = prompt(
                                "📝 Add a review (optional):",
                              );
                              if (review) booking.review = review;

                              // Update cook's average rating
                              const cook = COOKS.find(
                                (c) => c.id === booking.cook.id,
                              );
                              if (cook) {
                                const totalRating = cook.rating * cook.reviews;
                                cook.reviews += 1;
                                cook.rating = (
                                  (totalRating + parseInt(rating)) /
                                  cook.reviews
                                ).toFixed(1);
                                Storage.save("cooks", COOKS);
                              }

                              Storage.save("bookings", BOOKINGS);
                              alert("✅ Thank you for your rating!");
                              window.location.reload();
                            }
                          }}
                          className="px-4 py-2 bg-orange-600 text-white rounded-lg font-semibold hover:bg-orange-700 text-sm"
                        >
                          ⭐ Rate Cook
                        </button>
                      )}
                    </div>
                    <div className="text-right">
                      <p className="text-3xl font-bold text-gray-600">
                        ₹{booking.total}
                      </p>
                    </div>
                  </div>
                  {booking.review && (
                    <div className="mt-4 p-4 bg-gray-50 rounded-lg border-l-4 border-orange-500">
                      <p className="text-sm font-semibold text-gray-700 mb-1">
                        Your Review:
                      </p>
                      <p className="text-gray-600 italic">"{booking.review}"</p>
                    </div>
                  )}
                </div>
              ))
            )}
          </div>
        )}

        {tab === "profile" && (
          <div className="bg-white rounded-2xl shadow-xl p-8">
            <h3 className="text-3xl font-bold text-gray-900 mb-8">
              Profile Information
            </h3>
            <div className="grid sm:grid-cols-2 gap-6">
              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Full Name
                </label>
                <input
                  type="text"
                  value={user.name}
                  readOnly
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg bg-gray-50 text-lg"
                />
              </div>
              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Email
                </label>
                <input
                  type="email"
                  value={user.email}
                  readOnly
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg bg-gray-50 text-lg"
                />
              </div>
              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Phone
                </label>
                <input
                  type="tel"
                  value={user.phone}
                  readOnly
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg bg-gray-50 text-lg"
                />
              </div>
              <div>
                <label className="block text-gray-700 font-bold mb-2">
                  Location
                </label>
                <input
                  type="text"
                  value={user.location}
                  readOnly
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg bg-gray-50 text-lg"
                />
              </div>
            </div>
            <div className="mt-8 p-6 bg-gradient-to-r from-orange-50 to-amber-50 rounded-xl border-2 border-orange-200">
              <p className="text-lg font-semibold text-gray-700">
                <strong>Account Type:</strong>{" "}
                {user.isCook ? "🍳 Cook & Customer" : "👤 Customer"}
              </p>
              <p className="text-lg font-semibold text-gray-700 mt-2">
                <strong>Total Bookings:</strong> {userBookings.length}
              </p>
            </div>
          </div>
        )}

        <div className="mt-8 text-center">
          <button
            onClick={onBack}
            className="px-8 py-4 border-2 border-orange-600 text-orange-600 rounded-lg font-bold text-lg hover:bg-orange-600 hover:text-white transition-all"
          >
            ← Back to Home
          </button>
        </div>
      </div>
    </div>
  );
}

// Footer
function Footer() {
  return (
    <footer className="bg-gray-900 text-white py-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-8">
          <div>
            <h3 className="text-2xl font-bold italic text-orange-500 mb-4">
              ServeMyMeal
            </h3>
            <p className="text-gray-400">
              Bringing restaurant-quality home-cooked meals to your doorstep.
            </p>
          </div>
          <div>
            <h4 className="font-bold text-lg mb-4">Quick Links</h4>
            <ul className="space-y-2 text-gray-400">
              <li>
                <a href="#home" className="hover:text-orange-500">
                  Home
                </a>
              </li>
              <li>
                <a href="#features" className="hover:text-orange-500">
                  How It Works
                </a>
              </li>
              <li>
                <a href="#cooks" className="hover:text-orange-500">
                  Browse Cooks
                </a>
              </li>
            </ul>
          </div>
          <div>
            <h4 className="font-bold text-lg mb-4">Support</h4>
            <ul className="space-y-2 text-gray-400">
              <li>
                <a href="#" className="hover:text-orange-500">
                  Help Center
                </a>
              </li>
              <li>
                <a href="#" className="hover:text-orange-500">
                  Terms of Service
                </a>
              </li>
              <li>
                <a href="#" className="hover:text-orange-500">
                  Privacy Policy
                </a>
              </li>
            </ul>
          </div>
          <div>
            <h4 className="font-bold text-lg mb-4">Contact</h4>
            <ul className="space-y-2 text-gray-400">
              <li>📧 support@servemymeal.com</li>
              <li>📞 +91 9876543210</li>
              <li>📍 Mumbai, India</li>
            </ul>
          </div>
        </div>
        <div className="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400">
          <p>© 2024 ServeMyMeal. All rights reserved. Made with in India.</p>
        </div>
      </div>
    </footer>
  );
}

// MAIN APP
function App() {
  const [user, setUser] = useState(CURRENT_USER);
  const [cooks, setCooks] = useState(COOKS);
  const [filteredCooks, setFilteredCooks] = useState(COOKS);
  const [showAuth, setShowAuth] = useState(false);
  const [showBecomeCook, setShowBecomeCook] = useState(false);
  const [showBooking, setShowBooking] = useState(false);
  const [selectedCook, setSelectedCook] = useState(null);
  const [showDashboard, setShowDashboard] = useState(false);

  const handleLogin = (u) => {
    setUser(u);
    CURRENT_USER = u;
  };

  const handleLogout = () => {
    setUser(null);
    CURRENT_USER = null;
    Storage.save("currentUser", null);
    setShowDashboard(false);
    alert("👋 Logged out successfully!");
  };

  const handleBecomeCookSuccess = (updatedUser) => {
    setUser(updatedUser);
    setCooks([...COOKS]);
    setFilteredCooks([...COOKS]);
  };

  return (
    <div className="min-h-screen bg-amber-50">
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800;900&display=swap');
        * { font-family: 'Inter', sans-serif; }
        @keyframes bounce { 0%, 100% { transform: translateY(0); } 50% { transform: translateY(-20px); } }
        .animate-bounce { animation: bounce 2s infinite; }
      `}</style>

      <Header
        user={user}
        onSignIn={() => setShowAuth(true)}
        onLogout={handleLogout}
        onBecomeCook={() => setShowBecomeCook(true)}
        onDashboard={() => setShowDashboard(true)}
      />

      {!showDashboard ? (
        <>
          <Hero />
          <Features />

          <section
            className="py-20 bg-gradient-to-br from-orange-50 to-amber-50"
            id="cooks"
          >
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
              <div className="text-center mb-16">
                <h2 className="text-5xl font-bold text-gray-900 mb-4">
                  Meet Our Expert Cooks
                </h2>
                <p className="text-xl text-gray-600 max-w-2xl mx-auto">
                  Browse our selection of verified professional home cooks ready
                  to serve you
                </p>
              </div>

              <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-8">
                {filteredCooks.map((cook) => (
                  <CookCard
                    key={cook.id}
                    cook={cook}
                    onBook={(c) => {
                      setSelectedCook(c);
                      setShowBooking(true);
                    }}
                    user={user}
                    onSignIn={() => setShowAuth(true)}
                  />
                ))}
              </div>
            </div>
          </section>

          <Footer />
        </>
      ) : (
        <Dashboard user={user} onBack={() => setShowDashboard(false)} />
      )}

      <AuthModal
        show={showAuth}
        onClose={() => setShowAuth(false)}
        onSuccess={handleLogin}
      />

      <BecomeCookModal
        show={showBecomeCook}
        onClose={() => setShowBecomeCook(false)}
        user={user}
        onSuccess={handleBecomeCookSuccess}
      />

      <BookingModal
        show={showBooking}
        onClose={() => setShowBooking(false)}
        cook={selectedCook}
        user={user}
      />
    </div>
  );
}

export default App;
