// --- Setup and Initialization ---

// Create animated stars (100 stars)
const starsContainer = document.getElementById("stars");
for (let i = 0; i < 100; i++) {
  const star = document.createElement("div");
  star.className = "star";
  star.style.left = Math.random() * 100 + "%";
  star.style.top = Math.random() * 100 + "%";
  star.style.animationDelay = Math.random() * 3 + "s";
  starsContainer.appendChild(star);
}

// Canvas setup
const canvas = document.getElementById("game-canvas");
const ctx = canvas.getContext("2d");

// Stat elements
const scoreEl = document.getElementById("score");
const livesEl = document.getElementById("lives");
const waveEl = document.getElementById("wave");
const startBtn = document.getElementById("start-btn");
const messageEl = document.getElementById("message");

// --- Game Constants ---
const PLAYER_SPEED = 8;
const BULLET_SPEED = 10;
const ENEMY_ROWS = 5;
const ENEMY_COLS = 10;
const ENEMY_SIZE = 30;
const ENEMY_GAP = 15;
const ENEMY_START_Y = 50;
const ENEMY_MOVE_FREQUENCY = 50; // Frames between enemy movements

// --- Game State ---
let score = 0;
let lives = 3;
let wave = 1;
let isGameRunning = false;
let gameLoopId = null;
let enemyDirection = 1; // 1 = right, -1 = left
let enemyMoveTimer = 0;
let enemySpeed = 5;

// Player object
let player = {
  x: canvas.width / 2,
  y: canvas.height - 50,
  width: 40,
  height: 20,
};

// Arrays for game objects
let bullets = [];
let enemies = [];

// --- Core Functions ---

function initGame() {
  score = 0;
  lives = 3;
  wave = 1;
  enemySpeed = 5; // Reset speed

  // Update UI stats
  scoreEl.textContent = score;
  livesEl.textContent = lives;
  waveEl.textContent = wave;

  isGameRunning = true;
  enemyDirection = 1;
  enemyMoveTimer = 0;
  messageEl.textContent = "";
  messageEl.className = "message";

  // Reset player and objects
  player.x = canvas.width / 2 - player.width / 2;
  bullets = [];
  enemies = [];
  createEnemies();

  startBtn.textContent = "PAUSE";
  startBtn.className = "paused";
}

function createEnemies() {
  enemies = []; // Clear existing enemies for new wave
  for (let r = 0; r < ENEMY_ROWS; r++) {
    for (let c = 0; c < ENEMY_COLS; c++) {
      enemies.push({
        x: 50 + c * (ENEMY_SIZE + ENEMY_GAP),
        y: ENEMY_START_Y + r * (ENEMY_SIZE + ENEMY_GAP),
        width: ENEMY_SIZE,
        height: ENEMY_SIZE,
        row: r,
        isAlive: true,
      });
    }
  }
}

// --- Drawing Functions ---

function drawPlayer() {
  ctx.fillStyle = "#00ff88"; // Player body color

  // Draw Player Ship (Simple block with cannon)
  ctx.fillRect(player.x, player.y, player.width, player.height);
  ctx.fillRect(player.x + player.width / 2 - 2, player.y - 10, 4, 10);

  // Neon Glow Effect
  ctx.shadowBlur = 15;
  ctx.shadowColor = "#00ff88";
  ctx.fillRect(player.x, player.y, player.width, player.height);
  ctx.shadowBlur = 0; // Reset shadow
}

function drawEnemies() {
  enemies.forEach((enemy) => {
    if (enemy.isAlive) {
      // Color based on row
      let color;
      if (enemy.row < 1) color = "#ff0055"; // Top row
      else if (enemy.row < 3) color = "#ff8800"; // Middle rows
      else color = "#ffaa00"; // Bottom rows

      ctx.fillStyle = color;
      ctx.fillRect(enemy.x, enemy.y, enemy.width, enemy.height);

      // Simple pixel 'eyes' for retro look
      ctx.fillStyle = "#000";
      ctx.fillRect(enemy.x + 8, enemy.y + 8, 5, 5);
      ctx.fillRect(enemy.x + 17, enemy.y + 8, 5, 5);
    }
  });
}

function drawBullets() {
  ctx.fillStyle = "#00ccff"; // Bullet color
  ctx.shadowBlur = 10;
  ctx.shadowColor = "#00ccff";
  bullets.forEach((bullet) => {
    ctx.fillRect(bullet.x, bullet.y, bullet.width, bullet.height);
  });
  ctx.shadowBlur = 0;
}

function draw() {
  // Clear canvas
  ctx.clearRect(0, 0, canvas.width, canvas.height);
  ctx.fillStyle = "#000";
  ctx.fillRect(0, 0, canvas.width, canvas.height);

  drawPlayer();
  drawEnemies();
  drawBullets();
}

// --- Movement and Game Logic ---

function movePlayer(direction) {
  if (!isGameRunning) return;
  player.x += direction * PLAYER_SPEED;
  // Boundary checks
  if (player.x < 0) player.x = 0;
  if (player.x > canvas.width - player.width)
    player.x = canvas.width - player.width;
}

function moveBullets() {
  bullets.forEach((bullet) => (bullet.y -= BULLET_SPEED));
  // Remove bullets that leave the screen
  bullets = bullets.filter((bullet) => bullet.y > 0);
}

function fireBullet() {
  if (!isGameRunning) return;
  // Allow up to 3 bullets on screen at a time
  if (bullets.length < 3) {
    bullets.push({
      x: player.x + player.width / 2 - 2,
      y: player.y - 10,
      width: 4,
      height: 10,
    });
  }
}

function moveEnemies() {
  let shouldDrop = false;
  const aliveEnemies = enemies.filter((e) => e.isAlive);

  if (aliveEnemies.length === 0) return; // Should be handled by checkWinCondition

  // Calculate boundaries for edge detection
  const minX = Math.min(...aliveEnemies.map((e) => e.x));
  const maxX = Math.max(...aliveEnemies.map((e) => e.x + e.width));

  // Check boundaries
  if (enemyDirection === 1 && maxX >= canvas.width - 20) {
    enemyDirection = -1;
    shouldDrop = true;
  } else if (enemyDirection === -1 && minX <= 20) {
    enemyDirection = 1;
    shouldDrop = true;
  }

  enemies.forEach((enemy) => {
    if (enemy.isAlive) {
      if (shouldDrop) {
        enemy.y += 20; // Drop distance
      } else {
        // Move horizontally using current speed
        enemy.x += enemyDirection * enemySpeed;
      }

      // Check for Game Over condition (enemies reach the bottom)
      if (enemy.y + enemy.height >= player.y) {
        gameOver(false);
      }
    }
  });
}

function checkCollisions() {
  // Iterate backwards for safe removal of bullets
  for (let i = bullets.length - 1; i >= 0; i--) {
    const bullet = bullets[i];
    for (let j = 0; j < enemies.length; j++) {
      const enemy = enemies[j];

      if (enemy.isAlive) {
        // Simple AABB collision check
        if (
          bullet.x < enemy.x + enemy.width &&
          bullet.x + bullet.width > enemy.x &&
          bullet.y < enemy.y + enemy.height &&
          bullet.y + bullet.height > enemy.y
        ) {
          // Hit!
          enemy.isAlive = false;
          bullets.splice(i, 1); // Remove bullet
          score += 10;
          scoreEl.textContent = score;
          break;
        }
      }
    }
  }
}

function checkWinCondition() {
  if (enemies.every((e) => !e.isAlive)) {
    // Advance to next wave
    wave++;
    waveEl.textContent = wave;
    enemySpeed += 1; // Increase difficulty
    createEnemies();

    // Display wave message
    messageEl.textContent = `Wave ${wave}!`;
    messageEl.className = "message win";
    setTimeout(() => {
      messageEl.textContent = "";
      messageEl.className = "message";
    }, 2000);
  }
}

function update() {
  if (!isGameRunning) return;

  moveBullets();
  checkCollisions();
  checkWinCondition();

  enemyMoveTimer++;
  // Move enemies only when the timer hits the frequency limit
  if (enemyMoveTimer >= ENEMY_MOVE_FREQUENCY) {
    moveEnemies();
    enemyMoveTimer = 0;
  }

  draw();
  gameLoopId = requestAnimationFrame(update);
}

function gameOver(isWin) {
  isGameRunning = false;
  cancelAnimationFrame(gameLoopId);
  startBtn.textContent = "RESTART";
  startBtn.className = "ended";

  if (isWin) {
    messageEl.textContent = `VICTORY! Score: ${score}`;
    messageEl.className = "message win";
  } else {
    messageEl.textContent = `GAME OVER! Score: ${score}`;
    messageEl.className = "message lose";
  }
}

function toggleGame() {
  if (isGameRunning) {
    // Pause
    isGameRunning = false;
    cancelAnimationFrame(gameLoopId);
    startBtn.textContent = "RESUME";
    startBtn.className = "";
  } else {
    // Resume or Start
    if (startBtn.textContent === "RESTART" || enemies.length === 0) {
      initGame();
    }
    isGameRunning = true;
    startBtn.textContent = "PAUSE";
    startBtn.className = "paused";
    gameLoopId = requestAnimationFrame(update);
  }
}

// --- Event Listeners ---

startBtn.addEventListener("click", toggleGame);

// Touch/Button controls (using mousedown/up for simulated continuous movement)
const leftBtn = document.getElementById("btn-left");
const rightBtn = document.getElementById("btn-right");
const fireBtn = document.getElementById("btn-fire");

fireBtn.addEventListener("click", fireBullet);

// Simple move on press/tap
leftBtn.addEventListener("click", () => movePlayer(-1));
rightBtn.addEventListener("click", () => movePlayer(1));

// Keyboard controls
document.addEventListener("keydown", (e) => {
  if (!isGameRunning) return;
  switch (e.key) {
    case "ArrowLeft":
    case "a":
    case "A":
      e.preventDefault();
      movePlayer(-1);
      break;
    case "ArrowRight":
    case "d":
    case "D":
      e.preventDefault();
      movePlayer(1);
      break;
    case " ": // Space bar to fire
      e.preventDefault();
      fireBullet();
      break;
  }
});

// --- Initial Load ---
window.onload = () => {
  initGame();
  isGameRunning = false; // Set to false so the user has to click start
  startBtn.textContent = "START GAME";
  startBtn.className = "";
  messageEl.textContent = "Press Start Game";
  draw();
};
