import React from "react";

export default function EventCard({ event, onRSVP, onViewDetails }) {
  const seatsLeft = event.seats - event.booked;
  const percentBooked = (event.booked / event.seats) * 100;

  const getStatusColor = () => {
    if (percentBooked === 100) return "bg-red-500";
    if (percentBooked >= 80) return "bg-orange-500";
    if (percentBooked >= 50) return "bg-yellow-500";
    return "bg-green-500";
  };

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("en-US", {
      weekday: "short",
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  return (
    <div className="bg-white rounded-2xl shadow-lg overflow-hidden hover:shadow-2xl transition-all duration-300 hover:-translate-y-1 animate-slide-up">
      {/* Header with gradient */}
      <div className="bg-gradient-to-r from-indigo-500 via-purple-500 to-pink-500 p-6 relative">
        <div className="absolute top-4 right-4">
          <span
            className={`${getStatusColor()} text-white text-xs font-bold px-3 py-1 rounded-full shadow-lg`}
          >
            {seatsLeft === 0 ? "SOLD OUT" : `${seatsLeft} left`}
          </span>
        </div>
        <h3 className="text-2xl font-bold text-white mb-2 pr-20">
          {event.title}
        </h3>
        <div className="flex items-center text-white/90 text-sm">
          <svg
            className="w-4 h-4 mr-1"
            fill="none"
            stroke="currentColor"
            viewBox="0 0 24 24"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"
            />
          </svg>
          {formatDate(event.date)}
        </div>
      </div>

      {/* Body */}
      <div className="p-6">
        <div className="space-y-4">
          {/* Location */}
          <div className="flex items-start">
            <svg
              className="w-5 h-5 text-indigo-600 mr-3 mt-0.5 flex-shrink-0"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"
              />
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"
              />
            </svg>
            <div>
              <p className="text-sm text-gray-500 font-medium">Location</p>
              <p className="text-gray-900 font-semibold">{event.location}</p>
            </div>
          </div>

          {/* Seats Progress */}
          <div>
            <div className="flex justify-between items-center mb-2">
              <span className="text-sm text-gray-500 font-medium">
                Bookings
              </span>
              <span className="text-sm font-bold text-gray-900">
                {event.booked} / {event.seats}
              </span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
              <div
                className={`h-full ${getStatusColor()} transition-all duration-500 rounded-full`}
                style={{ width: `${percentBooked}%` }}
              ></div>
            </div>
          </div>

          {/* Stats */}
          <div className="grid grid-cols-2 gap-4 pt-4 border-t border-gray-100">
            <div className="text-center">
              <p className="text-2xl font-bold text-indigo-600">
                {event.booked}
              </p>
              <p className="text-xs text-gray-500 uppercase tracking-wide">
                Registered
              </p>
            </div>
            <div className="text-center">
              <p className="text-2xl font-bold text-purple-600">{seatsLeft}</p>
              <p className="text-xs text-gray-500 uppercase tracking-wide">
                Available
              </p>
            </div>
          </div>
        </div>

        {/* Actions */}
        <div className="mt-6 space-y-2">
          <button
            onClick={onRSVP}
            disabled={seatsLeft === 0}
            className={`w-full py-3 rounded-xl font-bold text-white transition-all duration-300 ${
              seatsLeft === 0
                ? "bg-gray-400 cursor-not-allowed"
                : "bg-gradient-to-r from-indigo-600 to-purple-600 hover:from-indigo-700 hover:to-purple-700 shadow-lg hover:shadow-xl transform hover:scale-105"
            }`}
          >
            {seatsLeft === 0 ? "Fully Booked" : "RSVP Now"}
          </button>

          {onViewDetails && (
            <button
              onClick={onViewDetails}
              className="w-full py-3 rounded-xl font-semibold text-indigo-600 border-2 border-indigo-600 hover:bg-indigo-50 transition-all duration-300"
            >
              View Details
            </button>
          )}
        </div>
      </div>
    </div>
  );
}
