import { useState, useEffect } from "react";
import EventCard from "./components/EventCard";

export default function App() {
  const [events, setEvents] = useState([]);
  const [form, setForm] = useState({
    title: "",
    date: "",
    location: "",
    seats: "",
    description: "",
    category: "Conference",
  });
  const [showModal, setShowModal] = useState(false);
  const [selectedEvent, setSelectedEvent] = useState(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [filterCategory, setFilterCategory] = useState("All");
  const [sortBy, setSortBy] = useState("date");
  const [userName, setUserName] = useState("");
  const [showUserModal, setShowUserModal] = useState(false);
  const [myBookings, setMyBookings] = useState([]);

  const categories = [
    "All",
    "Conference",
    "Workshop",
    "Meetup",
    "Seminar",
    "Concert",
    "Sports",
  ];

  // Load data from localStorage
  useEffect(() => {
    const savedEvents = localStorage.getItem("events");
    const savedBookings = localStorage.getItem("myBookings");
    const savedUserName = localStorage.getItem("userName");

    if (savedEvents) setEvents(JSON.parse(savedEvents));
    if (savedBookings) setMyBookings(JSON.parse(savedBookings));
    if (savedUserName) setUserName(savedUserName);
  }, []);

  // Save events to localStorage
  useEffect(() => {
    if (events.length > 0) {
      localStorage.setItem("events", JSON.stringify(events));
    }
  }, [events]);

  // Save bookings to localStorage
  useEffect(() => {
    localStorage.setItem("myBookings", JSON.stringify(myBookings));
  }, [myBookings]);

  const createEvent = () => {
    if (!form.title || !form.date || !form.location || !form.seats) {
      alert("Please fill all required fields!");
      return;
    }

    const newEvent = {
      ...form,
      id: Date.now(),
      booked: 0,
      createdAt: new Date().toISOString(),
      attendees: [],
    };

    setEvents([...events, newEvent]);
    setForm({
      title: "",
      date: "",
      location: "",
      seats: "",
      description: "",
      category: "Conference",
    });
    setShowModal(false);
  };

  const rsvp = (id) => {
    if (!userName) {
      setShowUserModal(true);
      return;
    }

    const event = events.find((e) => e.id === id);
    if (event.booked >= event.seats) {
      alert("Sorry, this event is fully booked!");
      return;
    }

    // Check if user already booked this event
    if (myBookings.some((booking) => booking.eventId === id)) {
      alert("You've already booked this event!");
      return;
    }

    setEvents(
      events.map((e) =>
        e.id === id && e.booked < e.seats
          ? {
              ...e,
              booked: e.booked + 1,
              attendees: [...e.attendees, userName],
            }
          : e
      )
    );

    // Add to user's bookings
    setMyBookings([
      ...myBookings,
      {
        eventId: id,
        eventTitle: event.title,
        eventDate: event.date,
        eventLocation: event.location,
        bookedAt: new Date().toISOString(),
        ticketId: `TKT-${Date.now()}`,
      },
    ]);

    alert(`Successfully booked! Your ticket ID: TKT-${Date.now()}`);
  };

  const deleteEvent = (id) => {
    if (confirm("Are you sure you want to delete this event?")) {
      setEvents(events.filter((e) => e.id !== id));
      setMyBookings(myBookings.filter((b) => b.eventId !== id));
    }
  };

  const saveUserName = () => {
    if (userName.trim()) {
      localStorage.setItem("userName", userName);
      setShowUserModal(false);
    } else {
      alert("Please enter your name!");
    }
  };

  // Filter and sort events
  const filteredEvents = events
    .filter((event) => {
      const matchesSearch =
        event.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        event.location.toLowerCase().includes(searchTerm.toLowerCase());
      const matchesCategory =
        filterCategory === "All" || event.category === filterCategory;
      return matchesSearch && matchesCategory;
    })
    .sort((a, b) => {
      if (sortBy === "date") return new Date(a.date) - new Date(b.date);
      if (sortBy === "seats") return b.seats - b.booked - (a.seats - a.booked);
      if (sortBy === "popularity") return b.booked - a.booked;
      return 0;
    });

  const stats = {
    total: events.length,
    upcoming: events.filter((e) => new Date(e.date) >= new Date()).length,
    myBookings: myBookings.length,
    totalSeats: events.reduce((sum, e) => sum + parseInt(e.seats), 0),
    bookedSeats: events.reduce((sum, e) => sum + e.booked, 0),
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50">
      {/* Header */}
      <header className="bg-white shadow-lg sticky top-0 z-50 border-b-4 border-indigo-500">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex flex-col md:flex-row justify-between items-center gap-4">
            <div className="flex items-center space-x-3">
              <div className="w-14 h-14 bg-gradient-to-r from-indigo-600 to-purple-600 rounded-2xl flex items-center justify-center shadow-lg">
                <svg
                  className="w-8 h-8 text-white"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"
                  />
                </svg>
              </div>
              <div>
                <h1 className="text-3xl font-bold bg-gradient-to-r from-indigo-600 to-purple-600 bg-clip-text text-transparent">
                  EventHub
                </h1>
                <p className="text-sm text-gray-600">
                  Manage Your Events Seamlessly
                </p>
              </div>
            </div>

            <div className="flex items-center gap-3">
              {userName && (
                <span className="hidden sm:block text-sm font-medium text-gray-700 bg-indigo-100 px-4 py-2 rounded-full">
                  👤 {userName}
                </span>
              )}
              <button
                onClick={() => setShowModal(true)}
                className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-6 py-3 rounded-xl font-bold shadow-lg hover:shadow-xl transition-all duration-300 hover:scale-105 flex items-center space-x-2"
              >
                <svg
                  className="w-5 h-5"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M12 4v16m8-8H4"
                  />
                </svg>
                <span>Create Event</span>
              </button>
            </div>
          </div>
        </div>
      </header>

      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Stats Dashboard */}
        <div className="grid grid-cols-2 md:grid-cols-5 gap-4 mb-8 animate-fade-in">
          <div className="bg-white rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300 border-l-4 border-indigo-500">
            <div className="text-3xl font-bold text-indigo-600">
              {stats.total}
            </div>
            <div className="text-sm text-gray-600 font-medium">
              Total Events
            </div>
          </div>
          <div className="bg-white rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300 border-l-4 border-green-500">
            <div className="text-3xl font-bold text-green-600">
              {stats.upcoming}
            </div>
            <div className="text-sm text-gray-600 font-medium">Upcoming</div>
          </div>
          <div className="bg-white rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300 border-l-4 border-purple-500">
            <div className="text-3xl font-bold text-purple-600">
              {stats.myBookings}
            </div>
            <div className="text-sm text-gray-600 font-medium">My Bookings</div>
          </div>
          <div className="bg-white rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300 border-l-4 border-yellow-500">
            <div className="text-3xl font-bold text-yellow-600">
              {stats.totalSeats}
            </div>
            <div className="text-sm text-gray-600 font-medium">Total Seats</div>
          </div>
          <div className="bg-white rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300 border-l-4 border-red-500">
            <div className="text-3xl font-bold text-red-600">
              {stats.bookedSeats}
            </div>
            <div className="text-sm text-gray-600 font-medium">Booked</div>
          </div>
        </div>

        {/* Search and Filter Section */}
        <div className="bg-white rounded-2xl p-6 shadow-lg mb-8 animate-slide-up">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {/* Search */}
            <div className="relative">
              <svg
                className="absolute left-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
                />
              </svg>
              <input
                type="text"
                placeholder="Search events..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-12 pr-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors"
              />
            </div>

            {/* Category Filter */}
            <select
              value={filterCategory}
              onChange={(e) => setFilterCategory(e.target.value)}
              className="px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors font-medium"
            >
              {categories.map((cat) => (
                <option key={cat} value={cat}>
                  {cat}
                </option>
              ))}
            </select>

            {/* Sort */}
            <select
              value={sortBy}
              onChange={(e) => setSortBy(e.target.value)}
              className="px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors font-medium"
            >
              <option value="date">Sort by Date</option>
              <option value="seats">Sort by Availability</option>
              <option value="popularity">Sort by Popularity</option>
            </select>
          </div>
        </div>

        {/* Events Grid */}
        {filteredEvents.length === 0 ? (
          <div className="bg-white rounded-3xl p-16 text-center shadow-lg animate-fade-in">
            <svg
              className="w-24 h-24 text-gray-300 mx-auto mb-4"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"
              />
            </svg>
            <h3 className="text-2xl font-bold text-gray-900 mb-2">
              No Events Found
            </h3>
            <p className="text-gray-600 mb-6">
              Create your first event to get started!
            </p>
            <button
              onClick={() => setShowModal(true)}
              className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-8 py-3 rounded-xl font-bold shadow-lg hover:shadow-xl transition-all duration-300"
            >
              Create Event
            </button>
          </div>
        ) : (
          <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredEvents.map((event, idx) => (
              <div key={event.id} style={{ animationDelay: `${idx * 0.1}s` }}>
                <EventCard
                  event={event}
                  onRSVP={() => rsvp(event.id)}
                  onViewDetails={() => setSelectedEvent(event)}
                />
              </div>
            ))}
          </div>
        )}

        {/* My Bookings Section */}
        {myBookings.length > 0 && (
          <div className="mt-12 animate-slide-up">
            <h2 className="text-3xl font-bold text-gray-900 mb-6 flex items-center">
              <svg
                className="w-8 h-8 mr-3 text-indigo-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M15 5v2m0 4v2m0 4v2M5 5a2 2 0 00-2 2v3a2 2 0 110 4v3a2 2 0 002 2h14a2 2 0 002-2v-3a2 2 0 110-4V7a2 2 0 00-2-2H5z"
                />
              </svg>
              My Tickets
            </h2>
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
              {myBookings.map((booking) => (
                <div
                  key={booking.ticketId}
                  className="bg-white rounded-2xl shadow-lg p-6 border-l-4 border-green-500 hover:shadow-xl transition-all duration-300"
                >
                  <div className="flex justify-between items-start mb-4">
                    <div>
                      <span className="inline-block bg-green-500 text-white text-xs font-bold px-3 py-1 rounded-full mb-2">
                        CONFIRMED
                      </span>
                      <h4 className="font-bold text-lg text-gray-900">
                        {booking.eventTitle}
                      </h4>
                    </div>
                  </div>
                  <div className="space-y-2 text-sm">
                    <p className="text-gray-600">
                      <span className="font-semibold">Date:</span>{" "}
                      {new Date(booking.eventDate).toLocaleDateString()}
                    </p>
                    <p className="text-gray-600">
                      <span className="font-semibold">Location:</span>{" "}
                      {booking.eventLocation}
                    </p>
                    <p className="text-gray-600">
                      <span className="font-semibold">Ticket ID:</span>{" "}
                      {booking.ticketId}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}
      </main>

      {/* Create Event Modal */}
      {showModal && (
        <div className="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4 z-50 animate-fade-in">
          <div className="bg-white rounded-3xl max-w-2xl w-full p-8 max-h-[90vh] overflow-y-auto shadow-2xl animate-slide-up">
            <div className="flex justify-between items-center mb-6">
              <h2 className="text-3xl font-bold text-gray-900">
                Create New Event
              </h2>
              <button
                onClick={() => setShowModal(false)}
                className="p-2 hover:bg-gray-100 rounded-xl transition-colors"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M6 18L18 6M6 6l12 12"
                  />
                </svg>
              </button>
            </div>

            <form
              className="space-y-5"
              onSubmit={(e) => {
                e.preventDefault();
                createEvent();
              }}
            >
              <div>
                <label className="block text-sm font-bold text-gray-700 mb-2">
                  Event Title *
                </label>
                <input
                  type="text"
                  required
                  placeholder="e.g., Tech Conference 2025"
                  value={form.title}
                  onChange={(e) => setForm({ ...form, title: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors"
                />
              </div>

              <div>
                <label className="block text-sm font-bold text-gray-700 mb-2">
                  Description
                </label>
                <textarea
                  rows="3"
                  placeholder="Event description..."
                  value={form.description}
                  onChange={(e) =>
                    setForm({ ...form, description: e.target.value })
                  }
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors resize-none"
                />
              </div>

              <div className="grid md:grid-cols-2 gap-5">
                <div>
                  <label className="block text-sm font-bold text-gray-700 mb-2">
                    Date *
                  </label>
                  <input
                    type="date"
                    required
                    value={form.date}
                    onChange={(e) => setForm({ ...form, date: e.target.value })}
                    className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors"
                  />
                </div>

                <div>
                  <label className="block text-sm font-bold text-gray-700 mb-2">
                    Category *
                  </label>
                  <select
                    value={form.category}
                    onChange={(e) =>
                      setForm({ ...form, category: e.target.value })
                    }
                    className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors"
                  >
                    {categories
                      .filter((c) => c !== "All")
                      .map((cat) => (
                        <option key={cat} value={cat}>
                          {cat}
                        </option>
                      ))}
                  </select>
                </div>
              </div>

              <div>
                <label className="block text-sm font-bold text-gray-700 mb-2">
                  Location *
                </label>
                <input
                  type="text"
                  required
                  placeholder="e.g., New York Convention Center"
                  value={form.location}
                  onChange={(e) =>
                    setForm({ ...form, location: e.target.value })
                  }
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors"
                />
              </div>

              <div>
                <label className="block text-sm font-bold text-gray-700 mb-2">
                  Total Seats *
                </label>
                <input
                  type="number"
                  required
                  min="1"
                  placeholder="e.g., 100"
                  value={form.seats}
                  onChange={(e) => setForm({ ...form, seats: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors"
                />
              </div>

              <div className="flex gap-3 pt-4">
                <button
                  type="button"
                  onClick={() => setShowModal(false)}
                  className="flex-1 px-6 py-3 border-2 border-gray-300 text-gray-700 rounded-xl font-bold hover:bg-gray-50 transition-colors"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="flex-1 bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-6 py-3 rounded-xl font-bold shadow-lg hover:shadow-xl transition-all duration-300"
                >
                  Create Event
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* User Name Modal */}
      {showUserModal && (
        <div className="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4 z-50 animate-fade-in">
          <div className="bg-white rounded-3xl max-w-md w-full p-8 shadow-2xl animate-slide-up">
            <h3 className="text-2xl font-bold text-gray-900 mb-4">
              Enter Your Name
            </h3>
            <p className="text-gray-600 mb-6">
              We need your name to complete the booking.
            </p>
            <input
              type="text"
              placeholder="Your full name"
              value={userName}
              onChange={(e) => setUserName(e.target.value)}
              className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-indigo-500 transition-colors mb-4"
            />
            <div className="flex gap-3">
              <button
                onClick={() => setShowUserModal(false)}
                className="flex-1 px-6 py-3 border-2 border-gray-300 text-gray-700 rounded-xl font-bold hover:bg-gray-50 transition-colors"
              >
                Cancel
              </button>
              <button
                onClick={saveUserName}
                className="flex-1 bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-6 py-3 rounded-xl font-bold shadow-lg hover:shadow-xl transition-all"
              >
                Continue
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Event Details Modal */}
      {selectedEvent && (
        <div className="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4 z-50 animate-fade-in">
          <div className="bg-white rounded-3xl max-w-2xl w-full p-8 max-h-[90vh] overflow-y-auto shadow-2xl animate-slide-up">
            <div className="flex justify-between items-start mb-6">
              <div>
                <span className="inline-block bg-indigo-600 text-white text-xs font-bold px-3 py-1 rounded-full mb-2">
                  {selectedEvent.category}
                </span>
                <h2 className="text-3xl font-bold text-gray-900">
                  {selectedEvent.title}
                </h2>
              </div>
              <button
                onClick={() => setSelectedEvent(null)}
                className="p-2 hover:bg-gray-100 rounded-xl transition-colors"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M6 18L18 6M6 6l12 12"
                  />
                </svg>
              </button>
            </div>

            <div className="space-y-6">
              {selectedEvent.description && (
                <div>
                  <h4 className="font-bold text-gray-900 mb-2">Description</h4>
                  <p className="text-gray-600">{selectedEvent.description}</p>
                </div>
              )}

              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-bold text-gray-900 mb-2">Date</h4>
                  <p className="text-gray-600">
                    {new Date(selectedEvent.date).toLocaleDateString("en-US", {
                      weekday: "long",
                      year: "numeric",
                      month: "long",
                      day: "numeric",
                    })}
                  </p>
                </div>
                <div>
                  <h4 className="font-bold text-gray-900 mb-2">Location</h4>
                  <p className="text-gray-600">{selectedEvent.location}</p>
                </div>
              </div>

              <div className="bg-gradient-to-r from-indigo-50 to-purple-50 rounded-2xl p-6">
                <div className="grid grid-cols-3 gap-4 text-center">
                  <div>
                    <div className="text-3xl font-bold text-indigo-600">
                      {selectedEvent.seats}
                    </div>
                    <div className="text-sm text-gray-600">Total Seats</div>
                  </div>
                  <div>
                    <div className="text-3xl font-bold text-green-600">
                      {selectedEvent.booked}
                    </div>
                    <div className="text-sm text-gray-600">Booked</div>
                  </div>
                  <div>
                    <div className="text-3xl font-bold text-purple-600">
                      {selectedEvent.seats - selectedEvent.booked}
                    </div>
                    <div className="text-sm text-gray-600">Available</div>
                  </div>
                </div>
              </div>

              {selectedEvent.attendees &&
                selectedEvent.attendees.length > 0 && (
                  <div>
                    <h4 className="font-bold text-gray-900 mb-3">
                      Attendees ({selectedEvent.attendees.length})
                    </h4>
                    <div className="flex flex-wrap gap-2">
                      {selectedEvent.attendees.map((attendee, idx) => (
                        <span
                          key={idx}
                          className="bg-indigo-100 text-indigo-700 px-3 py-1 rounded-full text-sm font-medium"
                        >
                          {attendee}
                        </span>
                      ))}
                    </div>
                  </div>
                )}

              <div className="flex gap-3 pt-4">
                <button
                  onClick={() => deleteEvent(selectedEvent.id)}
                  className="flex-1 border-2 border-red-500 text-red-500 px-6 py-3 rounded-xl font-bold hover:bg-red-50 transition-colors"
                >
                  Delete Event
                </button>
                <button
                  onClick={() => {
                    rsvp(selectedEvent.id);
                    setSelectedEvent(null);
                  }}
                  disabled={selectedEvent.booked >= selectedEvent.seats}
                  className={`flex-1 px-6 py-3 rounded-xl font-bold transition-all ${
                    selectedEvent.booked >= selectedEvent.seats
                      ? "bg-gray-400 text-white cursor-not-allowed"
                      : "bg-gradient-to-r from-indigo-600 to-purple-600 text-white shadow-lg hover:shadow-xl"
                  }`}
                >
                  {selectedEvent.booked >= selectedEvent.seats
                    ? "Fully Booked"
                    : "Book Now"}
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
