import React, { useState, useEffect, useRef } from "react";
import {
  Zap,
  Trophy,
  Target,
  Heart,
  PlayCircle,
  RotateCcw,
  Crown,
  History,
  Award,
  Star,
  TrendingUp,
  Medal,
  Sparkles,
  Clock,
  Flame,
  Shield,
} from "lucide-react";

const ColorClashGame = () => {
  const colors = [
    {
      name: "RED",
      bg: "bg-red-500",
      text: "text-red-500",
      hover: "hover:bg-red-600",
    },
    {
      name: "BLUE",
      bg: "bg-blue-500",
      text: "text-blue-500",
      hover: "hover:bg-blue-600",
    },
    {
      name: "GREEN",
      bg: "bg-green-500",
      text: "text-green-500",
      hover: "hover:bg-green-600",
    },
    {
      name: "YELLOW",
      bg: "bg-yellow-400",
      text: "text-yellow-400",
      hover: "hover:bg-yellow-500",
    },
    {
      name: "PURPLE",
      bg: "bg-purple-500",
      text: "text-purple-500",
      hover: "hover:bg-purple-600",
    },
    {
      name: "PINK",
      bg: "bg-pink-500",
      text: "text-pink-500",
      hover: "hover:bg-pink-600",
    },
    {
      name: "ORANGE",
      bg: "bg-orange-500",
      text: "text-orange-500",
      hover: "hover:bg-orange-600",
    },
    {
      name: "CYAN",
      bg: "bg-cyan-400",
      text: "text-cyan-400",
      hover: "hover:bg-cyan-500",
    },
  ];

  const [gameState, setGameState] = useState("menu"); // menu, playing, gameOver, history, stats, achievements
  const [score, setScore] = useState(0);
  const [lives, setLives] = useState(3);
  const [timeLeft, setTimeLeft] = useState(30);
  const [currentWord, setCurrentWord] = useState("");
  const [currentColor, setCurrentColor] = useState("");
  const [displayColor, setDisplayColor] = useState("");
  const [streak, setStreak] = useState(0);
  const [maxStreak, setMaxStreak] = useState(0);
  const [highScore, setHighScore] = useState(0);
  const [showFeedback, setShowFeedback] = useState(null);
  const [difficulty, setDifficulty] = useState("medium");
  const [gameHistory, setGameHistory] = useState([]);
  const [totalGames, setTotalGames] = useState(0);
  const [totalCorrect, setTotalCorrect] = useState(0);
  const [totalWrong, setTotalWrong] = useState(0);
  const [achievements, setAchievements] = useState([]);
  const [unlockedAchievement, setUnlockedAchievement] = useState(null);
  const [correctAnswers, setCorrectAnswers] = useState(0);

  // Power-ups
  const [powerUps, setPowerUps] = useState({
    timeFreeze: 0,
    shield: 0,
    doublePoints: 0,
  });
  const [activePowerUp, setActivePowerUp] = useState(null);
  const [powerUpTimeLeft, setPowerUpTimeLeft] = useState(0);
  const [hasShield, setHasShield] = useState(false);

  const timerRef = useRef(null);
  const powerUpTimerRef = useRef(null);

  // Achievement definitions
  const achievementsList = [
    {
      id: "first_win",
      name: "First Victory",
      description: "Complete your first game",
      icon: "🎉",
      unlocked: false,
    },
    {
      id: "streak_5",
      name: "Hot Streak",
      description: "Get a 5x streak",
      icon: "🔥",
      unlocked: false,
    },
    {
      id: "streak_10",
      name: "On Fire",
      description: "Get a 10x streak",
      icon: "💥",
      unlocked: false,
    },
    {
      id: "score_100",
      name: "Century",
      description: "Score 100 points in one game",
      icon: "💯",
      unlocked: false,
    },
    {
      id: "score_200",
      name: "Double Century",
      description: "Score 200 points in one game",
      icon: "🌟",
      unlocked: false,
    },
    {
      id: "games_10",
      name: "Dedicated",
      description: "Play 10 games",
      icon: "🎮",
      unlocked: false,
    },
    {
      id: "games_50",
      name: "Veteran",
      description: "Play 50 games",
      icon: "🏆",
      unlocked: false,
    },
    {
      id: "perfect_10",
      name: "Perfect 10",
      description: "Get 10 correct answers in a row",
      icon: "⭐",
      unlocked: false,
    },
    {
      id: "speed_demon",
      name: "Speed Demon",
      description: "Complete a hard game",
      icon: "⚡",
      unlocked: false,
    },
    {
      id: "accuracy_master",
      name: "Accuracy Master",
      description: "Maintain 90% accuracy over 20 answers",
      icon: "🎯",
      unlocked: false,
    },
  ];

  useEffect(() => {
    loadGameData();
  }, []);

  useEffect(() => {
    if (
      gameState === "playing" &&
      timeLeft > 0 &&
      lives > 0 &&
      activePowerUp !== "timeFreeze"
    ) {
      timerRef.current = setTimeout(() => {
        setTimeLeft(timeLeft - 1);
      }, 1000);
    } else if ((timeLeft === 0 || lives === 0) && gameState === "playing") {
      endGame();
    }
    return () => clearTimeout(timerRef.current);
  }, [timeLeft, gameState, lives, activePowerUp]);

  useEffect(() => {
    if (activePowerUp && powerUpTimeLeft > 0) {
      powerUpTimerRef.current = setTimeout(() => {
        setPowerUpTimeLeft(powerUpTimeLeft - 1);
      }, 1000);
    } else if (powerUpTimeLeft === 0 && activePowerUp) {
      deactivatePowerUp();
    }
    return () => clearTimeout(powerUpTimerRef.current);
  }, [powerUpTimeLeft, activePowerUp]);

  const loadGameData = () => {
    const savedHighScore = localStorage.getItem("colorClashHighScore");
    const savedHistory = localStorage.getItem("colorClashHistory");
    const savedStats = localStorage.getItem("colorClashStats");
    const savedAchievements = localStorage.getItem("colorClashAchievements");
    const savedPowerUps = localStorage.getItem("colorClashPowerUps");

    if (savedHighScore) setHighScore(parseInt(savedHighScore));
    if (savedHistory) setGameHistory(JSON.parse(savedHistory));
    if (savedStats) {
      const stats = JSON.parse(savedStats);
      setTotalGames(stats.totalGames || 0);
      setTotalCorrect(stats.totalCorrect || 0);
      setTotalWrong(stats.totalWrong || 0);
    }
    if (savedAchievements) {
      setAchievements(JSON.parse(savedAchievements));
    } else {
      setAchievements(achievementsList);
    }
    if (savedPowerUps) {
      setPowerUps(JSON.parse(savedPowerUps));
    }
  };

  const saveGameData = () => {
    localStorage.setItem("colorClashHighScore", highScore.toString());
    localStorage.setItem("colorClashHistory", JSON.stringify(gameHistory));
    localStorage.setItem(
      "colorClashStats",
      JSON.stringify({
        totalGames,
        totalCorrect,
        totalWrong,
      })
    );
    localStorage.setItem(
      "colorClashAchievements",
      JSON.stringify(achievements)
    );
    localStorage.setItem("colorClashPowerUps", JSON.stringify(powerUps));
  };

  const checkAchievements = (gameScore, gameStreak, gameDifficulty) => {
    const newAchievements = [...achievements];
    let achievementUnlocked = false;

    // Check each achievement
    if (
      !newAchievements.find((a) => a.id === "first_win")?.unlocked &&
      totalGames >= 0
    ) {
      unlockAchievement(newAchievements, "first_win");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "streak_5")?.unlocked &&
      gameStreak >= 5
    ) {
      unlockAchievement(newAchievements, "streak_5");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "streak_10")?.unlocked &&
      gameStreak >= 10
    ) {
      unlockAchievement(newAchievements, "streak_10");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "score_100")?.unlocked &&
      gameScore >= 100
    ) {
      unlockAchievement(newAchievements, "score_100");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "score_200")?.unlocked &&
      gameScore >= 200
    ) {
      unlockAchievement(newAchievements, "score_200");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "games_10")?.unlocked &&
      totalGames + 1 >= 10
    ) {
      unlockAchievement(newAchievements, "games_10");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "games_50")?.unlocked &&
      totalGames + 1 >= 50
    ) {
      unlockAchievement(newAchievements, "games_50");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "perfect_10")?.unlocked &&
      correctAnswers >= 10
    ) {
      unlockAchievement(newAchievements, "perfect_10");
      achievementUnlocked = true;
    }
    if (
      !newAchievements.find((a) => a.id === "speed_demon")?.unlocked &&
      gameDifficulty === "hard"
    ) {
      unlockAchievement(newAchievements, "speed_demon");
      achievementUnlocked = true;
    }
    if (!newAchievements.find((a) => a.id === "accuracy_master")?.unlocked) {
      const totalAnswers = totalCorrect + totalWrong + correctAnswers;
      const accuracy =
        totalAnswers > 0
          ? ((totalCorrect + correctAnswers) / totalAnswers) * 100
          : 0;
      if (accuracy >= 90 && totalAnswers >= 20) {
        unlockAchievement(newAchievements, "accuracy_master");
        achievementUnlocked = true;
      }
    }

    if (achievementUnlocked) {
      setAchievements(newAchievements);
      // Award power-up for achievement
      const updatedPowerUps = { ...powerUps };
      const randomPowerUp = ["timeFreeze", "shield", "doublePoints"][
        Math.floor(Math.random() * 3)
      ];
      updatedPowerUps[randomPowerUp]++;
      setPowerUps(updatedPowerUps);
    }
  };

  const unlockAchievement = (achievementArray, achievementId) => {
    const index = achievementArray.findIndex((a) => a.id === achievementId);
    if (index !== -1 && !achievementArray[index].unlocked) {
      achievementArray[index].unlocked = true;
      const achievement = achievementArray[index];
      setUnlockedAchievement(achievement);
      setTimeout(() => setUnlockedAchievement(null), 3000);
    }
  };

  const startGame = (selectedDifficulty) => {
    setDifficulty(selectedDifficulty);
    setGameState("playing");
    setScore(0);
    setLives(3);
    setTimeLeft(
      selectedDifficulty === "easy"
        ? 45
        : selectedDifficulty === "medium"
        ? 30
        : 20
    );
    setStreak(0);
    setMaxStreak(0);
    setCorrectAnswers(0);
    setActivePowerUp(null);
    setHasShield(false);
    generateNewChallenge();
  };

  const generateNewChallenge = () => {
    const wordColor = colors[Math.floor(Math.random() * colors.length)];
    const displayColorObj = colors[Math.floor(Math.random() * colors.length)];

    setCurrentWord(wordColor.name);
    setCurrentColor(wordColor);
    setDisplayColor(displayColorObj);
  };

  const usePowerUp = (powerUpType) => {
    if (powerUps[powerUpType] > 0 && !activePowerUp) {
      const updatedPowerUps = { ...powerUps };
      updatedPowerUps[powerUpType]--;
      setPowerUps(updatedPowerUps);
      setActivePowerUp(powerUpType);

      if (powerUpType === "timeFreeze") {
        setPowerUpTimeLeft(5);
      } else if (powerUpType === "shield") {
        setHasShield(true);
        setPowerUpTimeLeft(10);
      } else if (powerUpType === "doublePoints") {
        setPowerUpTimeLeft(10);
      }
    }
  };

  const deactivatePowerUp = () => {
    if (activePowerUp === "shield") {
      setHasShield(false);
    }
    setActivePowerUp(null);
    setPowerUpTimeLeft(0);
  };

  const handleAnswer = (answer) => {
    const isCorrect =
      (answer === "match" && currentWord === displayColor.name) ||
      (answer === "different" && currentWord !== displayColor.name);

    if (isCorrect) {
      const basePoints =
        difficulty === "easy" ? 10 : difficulty === "medium" ? 15 : 25;
      const streakBonus = Math.floor(streak / 3) * 5;
      const doubleMultiplier = activePowerUp === "doublePoints" ? 2 : 1;
      const totalPoints = (basePoints + streakBonus) * doubleMultiplier;

      setScore(score + totalPoints);
      setStreak(streak + 1);
      setCorrectAnswers(correctAnswers + 1);
      setTotalCorrect(totalCorrect + 1);

      if (streak + 1 > maxStreak) {
        setMaxStreak(streak + 1);
      }

      setShowFeedback("correct");

      // Add time bonus on streak
      if (streak > 0 && streak % 5 === 0) {
        setTimeLeft((prev) => Math.min(prev + 3, 60));
      }
    } else {
      if (hasShield) {
        setHasShield(false);
        setActivePowerUp(null);
        setShowFeedback("blocked");
      } else {
        setLives(lives - 1);
        setShowFeedback("wrong");
      }
      setStreak(0);
      setCorrectAnswers(0);
      setTotalWrong(totalWrong + 1);
    }

    setTimeout(() => {
      setShowFeedback(null);
      if (lives > 0 || hasShield) generateNewChallenge();
    }, 500);
  };

  const endGame = () => {
    setGameState("gameOver");

    const newTotalGames = totalGames + 1;
    setTotalGames(newTotalGames);

    // Save to history
    const gameRecord = {
      score,
      difficulty,
      streak: maxStreak,
      date: new Date().toISOString(),
      correct: correctAnswers,
    };

    const newHistory = [gameRecord, ...gameHistory].slice(0, 20); // Keep last 20 games
    setGameHistory(newHistory);

    // Update high score
    if (score > highScore) {
      setHighScore(score);
    }

    // Check achievements
    checkAchievements(score, maxStreak, difficulty);

    // Save all data
    setTimeout(() => {
      saveGameData();
    }, 100);
  };

  const returnToMenu = () => {
    setGameState("menu");
  };

  const clearHistory = () => {
    if (
      window.confirm(
        "Are you sure you want to clear all game history and stats?"
      )
    ) {
      setGameHistory([]);
      setTotalGames(0);
      setTotalCorrect(0);
      setTotalWrong(0);
      setHighScore(0);
      localStorage.removeItem("colorClashHistory");
      localStorage.removeItem("colorClashStats");
      localStorage.removeItem("colorClashHighScore");
    }
  };

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return (
      date.toLocaleDateString() +
      " " +
      date.toLocaleTimeString([], { hour: "2-digit", minute: "2-digit" })
    );
  };

  // Menu Screen
  if (gameState === "menu") {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-600 via-purple-600 to-pink-600 flex items-center justify-center p-4">
        <div className="max-w-2xl w-full">
          <div className="text-center mb-8 animate-pulse">
            <Zap className="w-16 h-16 sm:w-20 sm:h-20 text-yellow-300 mx-auto mb-4" />
            <h1 className="text-4xl sm:text-6xl lg:text-7xl font-black text-white mb-3 drop-shadow-2xl">
              COLOR CLASH
            </h1>
            <p className="text-white/90 text-lg sm:text-xl font-medium">
              Match the word to the color - Think Fast!
            </p>
          </div>

          <div className="bg-white/95 backdrop-blur-lg rounded-3xl shadow-2xl p-6 sm:p-8 mb-6">
            <h2 className="text-2xl sm:text-3xl font-bold text-gray-800 mb-4 text-center">
              How to Play
            </h2>
            <div className="space-y-3 text-gray-700">
              <p className="text-base sm:text-lg">
                🎯 A color word appears in a specific color
              </p>
              <p className="text-base sm:text-lg">
                ⚡ Tap <span className="font-bold">MATCH</span> if the word
                matches its color
              </p>
              <p className="text-base sm:text-lg">
                🔥 Tap <span className="font-bold">DIFFERENT</span> if they
                don't match
              </p>
              <p className="text-base sm:text-lg">
                💪 Build streaks for bonus points!
              </p>
              <p className="text-base sm:text-lg">
                🎁 Unlock achievements to earn power-ups!
              </p>
            </div>
          </div>

          <div className="space-y-4">
            <h3 className="text-center text-white text-xl font-bold mb-4">
              Select Difficulty
            </h3>

            <button
              onClick={() => startGame("easy")}
              className="w-full bg-gradient-to-r from-green-400 to-green-600 hover:from-green-500 hover:to-green-700 text-white font-bold py-4 sm:py-5 px-6 rounded-2xl text-lg sm:text-xl transition-all transform hover:scale-105 active:scale-95 shadow-xl"
            >
              <div className="flex items-center justify-center gap-3">
                <PlayCircle className="w-6 h-6" />
                <span>EASY (45 seconds)</span>
              </div>
            </button>

            <button
              onClick={() => startGame("medium")}
              className="w-full bg-gradient-to-r from-yellow-400 to-orange-500 hover:from-yellow-500 hover:to-orange-600 text-white font-bold py-4 sm:py-5 px-6 rounded-2xl text-lg sm:text-xl transition-all transform hover:scale-105 active:scale-95 shadow-xl"
            >
              <div className="flex items-center justify-center gap-3">
                <Target className="w-6 h-6" />
                <span>MEDIUM (30 seconds)</span>
              </div>
            </button>

            <button
              onClick={() => startGame("hard")}
              className="w-full bg-gradient-to-r from-red-500 to-pink-600 hover:from-red-600 hover:to-pink-700 text-white font-bold py-4 sm:py-5 px-6 rounded-2xl text-lg sm:text-xl transition-all transform hover:scale-105 active:scale-95 shadow-xl"
            >
              <div className="flex items-center justify-center gap-3">
                <Zap className="w-6 h-6" />
                <span>HARD (20 seconds)</span>
              </div>
            </button>
          </div>

          {/* Quick Stats */}
          <div className="mt-6 grid grid-cols-2 gap-4">
            {highScore > 0 && (
              <div className="bg-yellow-400/90 backdrop-blur-lg rounded-2xl p-4 text-center">
                <div className="flex items-center justify-center gap-2">
                  <Crown className="w-5 h-5 text-yellow-800" />
                  <div>
                    <p className="text-xs text-yellow-900 font-semibold">
                      High Score
                    </p>
                    <p className="text-xl font-bold text-yellow-900">
                      {highScore}
                    </p>
                  </div>
                </div>
              </div>
            )}

            {totalGames > 0 && (
              <div className="bg-blue-400/90 backdrop-blur-lg rounded-2xl p-4 text-center">
                <div className="flex items-center justify-center gap-2">
                  <Trophy className="w-5 h-5 text-blue-900" />
                  <div>
                    <p className="text-xs text-blue-900 font-semibold">
                      Games Played
                    </p>
                    <p className="text-xl font-bold text-blue-900">
                      {totalGames}
                    </p>
                  </div>
                </div>
              </div>
            )}
          </div>

          {/* Menu Buttons */}
          <div className="mt-6 grid grid-cols-2 gap-4">
            <button
              onClick={() => setGameState("history")}
              className="bg-white/20 hover:bg-white/30 backdrop-blur-lg text-white font-bold py-3 px-4 rounded-xl transition-all border-2 border-white/40"
            >
              <div className="flex items-center justify-center gap-2">
                <History className="w-5 h-5" />
                <span>History</span>
              </div>
            </button>

            <button
              onClick={() => setGameState("achievements")}
              className="bg-white/20 hover:bg-white/30 backdrop-blur-lg text-white font-bold py-3 px-4 rounded-xl transition-all border-2 border-white/40"
            >
              <div className="flex items-center justify-center gap-2">
                <Award className="w-5 h-5" />
                <span>Achievements</span>
              </div>
            </button>

            <button
              onClick={() => setGameState("stats")}
              className="bg-white/20 hover:bg-white/30 backdrop-blur-lg text-white font-bold py-3 px-4 rounded-xl transition-all border-2 border-white/40"
            >
              <div className="flex items-center justify-center gap-2">
                <TrendingUp className="w-5 h-5" />
                <span>Stats</span>
              </div>
            </button>

            <button
              onClick={() => setGameState("powerups")}
              className="bg-white/20 hover:bg-white/30 backdrop-blur-lg text-white font-bold py-3 px-4 rounded-xl transition-all border-2 border-white/40 relative"
            >
              <div className="flex items-center justify-center gap-2">
                <Sparkles className="w-5 h-5" />
                <span>Power-ups</span>
              </div>
              {powerUps.timeFreeze + powerUps.shield + powerUps.doublePoints >
                0 && (
                <span className="absolute -top-2 -right-2 bg-red-500 text-white text-xs font-bold rounded-full w-6 h-6 flex items-center justify-center">
                  {powerUps.timeFreeze +
                    powerUps.shield +
                    powerUps.doublePoints}
                </span>
              )}
            </button>
          </div>
        </div>
      </div>
    );
  }

  // History Screen
  if (gameState === "history") {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-600 via-indigo-600 to-purple-700 p-4">
        <div className="max-w-4xl mx-auto py-4">
          <div className="bg-white/95 backdrop-blur-lg rounded-3xl shadow-2xl p-6 sm:p-8">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-3xl sm:text-4xl font-black text-gray-800 flex items-center gap-3">
                <History className="w-8 h-8" />
                Game History
              </h2>
              <button
                onClick={returnToMenu}
                className="bg-gray-600 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded-xl transition-all"
              >
                Back
              </button>
            </div>

            {gameHistory.length === 0 ? (
              <div className="text-center py-12">
                <Trophy className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                <p className="text-gray-600 text-lg">
                  No games played yet. Start playing to build your history!
                </p>
              </div>
            ) : (
              <>
                <div className="space-y-3 max-h-[600px] overflow-y-auto">
                  {gameHistory.map((game, index) => (
                    <div
                      key={index}
                      className="bg-gradient-to-r from-purple-100 to-pink-100 rounded-xl p-4 border-2 border-purple-200"
                    >
                      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-2">
                            <Medal
                              className={`w-5 h-5 ${
                                game.difficulty === "easy"
                                  ? "text-green-600"
                                  : game.difficulty === "medium"
                                  ? "text-yellow-600"
                                  : "text-red-600"
                              }`}
                            />
                            <span
                              className={`font-bold uppercase text-sm ${
                                game.difficulty === "easy"
                                  ? "text-green-700"
                                  : game.difficulty === "medium"
                                  ? "text-yellow-700"
                                  : "text-red-700"
                              }`}
                            >
                              {game.difficulty}
                            </span>
                          </div>
                          <p className="text-xs text-gray-600">
                            {formatDate(game.date)}
                          </p>
                        </div>

                        <div className="flex gap-4">
                          <div className="text-center">
                            <p className="text-2xl font-black text-purple-700">
                              {game.score}
                            </p>
                            <p className="text-xs text-gray-600">Score</p>
                          </div>
                          <div className="text-center">
                            <p className="text-2xl font-black text-orange-600">
                              🔥{game.streak}
                            </p>
                            <p className="text-xs text-gray-600">Streak</p>
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>

                <button
                  onClick={clearHistory}
                  className="mt-6 w-full bg-red-500 hover:bg-red-600 text-white font-bold py-3 px-4 rounded-xl transition-all"
                >
                  Clear History
                </button>
              </>
            )}
          </div>
        </div>
      </div>
    );
  }

  // Stats Screen
  if (gameState === "stats") {
    const accuracy =
      totalCorrect + totalWrong > 0
        ? ((totalCorrect / (totalCorrect + totalWrong)) * 100).toFixed(1)
        : 0;

    return (
      <div className="min-h-screen bg-gradient-to-br from-green-600 via-teal-600 to-blue-700 p-4">
        <div className="max-w-4xl mx-auto py-4">
          <div className="bg-white/95 backdrop-blur-lg rounded-3xl shadow-2xl p-6 sm:p-8">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-3xl sm:text-4xl font-black text-gray-800 flex items-center gap-3">
                <TrendingUp className="w-8 h-8" />
                Your Statistics
              </h2>
              <button
                onClick={returnToMenu}
                className="bg-gray-600 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded-xl transition-all"
              >
                Back
              </button>
            </div>

            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 mb-6">
              <div className="bg-gradient-to-br from-blue-500 to-blue-600 rounded-2xl p-6 text-white text-center">
                <Trophy className="w-12 h-12 mx-auto mb-3" />
                <p className="text-4xl font-black mb-2">{highScore}</p>
                <p className="text-sm font-semibold opacity-90">High Score</p>
              </div>

              <div className="bg-gradient-to-br from-purple-500 to-purple-600 rounded-2xl p-6 text-white text-center">
                <Target className="w-12 h-12 mx-auto mb-3" />
                <p className="text-4xl font-black mb-2">{totalGames}</p>
                <p className="text-sm font-semibold opacity-90">Games Played</p>
              </div>

              <div className="bg-gradient-to-br from-green-500 to-green-600 rounded-2xl p-6 text-white text-center">
                <Star className="w-12 h-12 mx-auto mb-3" />
                <p className="text-4xl font-black mb-2">{accuracy}%</p>
                <p className="text-sm font-semibold opacity-90">Accuracy</p>
              </div>

              <div className="bg-gradient-to-br from-yellow-400 to-orange-500 rounded-2xl p-6 text-white text-center">
                <Zap className="w-12 h-12 mx-auto mb-3" />
                <p className="text-4xl font-black mb-2">{totalCorrect}</p>
                <p className="text-sm font-semibold opacity-90">
                  Correct Answers
                </p>
              </div>

              <div className="bg-gradient-to-br from-red-500 to-pink-600 rounded-2xl p-6 text-white text-center">
                <Heart className="w-12 h-12 mx-auto mb-3" />
                <p className="text-4xl font-black mb-2">{totalWrong}</p>
                <p className="text-sm font-semibold opacity-90">
                  Wrong Answers
                </p>
              </div>

              <div className="bg-gradient-to-br from-indigo-500 to-purple-600 rounded-2xl p-6 text-white text-center">
                <Award className="w-12 h-12 mx-auto mb-3" />
                <p className="text-4xl font-black mb-2">
                  {achievements.filter((a) => a.unlocked).length}
                </p>
                <p className="text-sm font-semibold opacity-90">Achievements</p>
              </div>
            </div>

            {gameHistory.length > 0 && (
              <div className="bg-gray-100 rounded-2xl p-6">
                <h3 className="text-xl font-bold text-gray-800 mb-4">
                  Recent Performance
                </h3>
                <div className="space-y-2">
                  {gameHistory.slice(0, 5).map((game, index) => (
                    <div
                      key={index}
                      className="flex justify-between items-center bg-white rounded-lg p-3"
                    >
                      <span className="text-sm text-gray-600">
                        {formatDate(game.date)}
                      </span>
                      <span
                        className={`font-bold px-3 py-1 rounded-full text-xs ${
                          game.difficulty === "easy"
                            ? "bg-green-200 text-green-800"
                            : game.difficulty === "medium"
                            ? "bg-yellow-200 text-yellow-800"
                            : "bg-red-200 text-red-800"
                        }`}
                      >
                        {game.difficulty.toUpperCase()}
                      </span>
                      <span className="text-lg font-black text-purple-600">
                        {game.score}
                      </span>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    );
  }

  // Achievements Screen
  if (gameState === "achievements") {
    const unlockedCount = achievements.filter((a) => a.unlocked).length;

    return (
      <div className="min-h-screen bg-gradient-to-br from-yellow-500 via-orange-500 to-red-600 p-4">
        <div className="max-w-4xl mx-auto py-4">
          <div className="bg-white/95 backdrop-blur-lg rounded-3xl shadow-2xl p-6 sm:p-8">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-3xl sm:text-4xl font-black text-gray-800 flex items-center gap-3">
                <Award className="w-8 h-8" />
                Achievements
              </h2>
              <button
                onClick={returnToMenu}
                className="bg-gray-600 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded-xl transition-all"
              >
                Back
              </button>
            </div>

            <div className="mb-6 bg-gradient-to-r from-purple-500 to-pink-500 rounded-2xl p-4 text-white text-center">
              <p className="text-lg font-semibold">
                Progress: {unlockedCount} / {achievements.length}
              </p>
              <div className="w-full bg-white/30 rounded-full h-4 mt-2">
                <div
                  className="bg-white h-4 rounded-full transition-all duration-500"
                  style={{
                    width: `${(unlockedCount / achievements.length) * 100}%`,
                  }}
                />
              </div>
            </div>

            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
              {achievements.map((achievement) => (
                <div
                  key={achievement.id}
                  className={`rounded-2xl p-5 border-2 transition-all ${
                    achievement.unlocked
                      ? "bg-gradient-to-br from-yellow-100 to-orange-100 border-yellow-400"
                      : "bg-gray-100 border-gray-300 opacity-60"
                  }`}
                >
                  <div className="flex items-start gap-4">
                    <div
                      className={`text-4xl ${
                        achievement.unlocked ? "animate-bounce" : "grayscale"
                      }`}
                    >
                      {achievement.icon}
                    </div>
                    <div className="flex-1">
                      <h3
                        className={`font-bold text-lg mb-1 ${
                          achievement.unlocked
                            ? "text-gray-800"
                            : "text-gray-600"
                        }`}
                      >
                        {achievement.name}
                      </h3>
                      <p className="text-sm text-gray-600">
                        {achievement.description}
                      </p>
                      {achievement.unlocked && (
                        <p className="text-xs text-green-600 font-semibold mt-2">
                          ✓ Unlocked!
                        </p>
                      )}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Power-ups Screen
  if (gameState === "powerups") {
    return (
      <div className="min-h-screen bg-gradient-to-br from-purple-600 via-pink-600 to-red-600 p-4">
        <div className="max-w-4xl mx-auto py-4">
          <div className="bg-white/95 backdrop-blur-lg rounded-3xl shadow-2xl p-6 sm:p-8">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-3xl sm:text-4xl font-black text-gray-800 flex items-center gap-3">
                <Sparkles className="w-8 h-8" />
                Power-ups
              </h2>
              <button
                onClick={returnToMenu}
                className="bg-gray-600 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded-xl transition-all"
              >
                Back
              </button>
            </div>

            <div className="bg-gradient-to-r from-blue-100 to-purple-100 rounded-2xl p-6 mb-6">
              <p className="text-gray-700 text-center">
                <strong>💡 Tip:</strong> Unlock achievements to earn power-ups!
                Use them strategically during gameplay.
              </p>
            </div>

            <div className="grid grid-cols-1 sm:grid-cols-3 gap-6">
              <div className="bg-gradient-to-br from-cyan-400 to-blue-500 rounded-2xl p-6 text-white">
                <div className="text-center mb-4">
                  <Clock className="w-16 h-16 mx-auto mb-3" />
                  <h3 className="text-2xl font-black mb-2">Time Freeze</h3>
                  <p className="text-sm opacity-90 mb-4">
                    Freeze the timer for 5 seconds
                  </p>
                  <div className="bg-white/30 rounded-full py-2 px-4 inline-block">
                    <span className="text-3xl font-black">
                      {powerUps.timeFreeze}
                    </span>
                  </div>
                </div>
              </div>

              <div className="bg-gradient-to-br from-green-400 to-emerald-500 rounded-2xl p-6 text-white">
                <div className="text-center mb-4">
                  <Shield className="w-16 h-16 mx-auto mb-3" />
                  <h3 className="text-2xl font-black mb-2">Shield</h3>
                  <p className="text-sm opacity-90 mb-4">
                    Block one wrong answer for 10 seconds
                  </p>
                  <div className="bg-white/30 rounded-full py-2 px-4 inline-block">
                    <span className="text-3xl font-black">
                      {powerUps.shield}
                    </span>
                  </div>
                </div>
              </div>

              <div className="bg-gradient-to-br from-yellow-400 to-orange-500 rounded-2xl p-6 text-white">
                <div className="text-center mb-4">
                  <Star className="w-16 h-16 mx-auto mb-3" />
                  <h3 className="text-2xl font-black mb-2">Double Points</h3>
                  <p className="text-sm opacity-90 mb-4">
                    2x points for 10 seconds
                  </p>
                  <div className="bg-white/30 rounded-full py-2 px-4 inline-block">
                    <span className="text-3xl font-black">
                      {powerUps.doublePoints}
                    </span>
                  </div>
                </div>
              </div>
            </div>

            <div className="mt-8 bg-yellow-50 border-2 border-yellow-300 rounded-2xl p-6">
              <h3 className="text-xl font-bold text-gray-800 mb-3">
                How to Get Power-ups:
              </h3>
              <ul className="space-y-2 text-gray-700">
                <li className="flex items-center gap-2">
                  <span className="text-green-600 font-bold">✓</span>
                  Unlock achievements to earn power-ups
                </li>
                <li className="flex items-center gap-2">
                  <span className="text-green-600 font-bold">✓</span>
                  Each achievement grants a random power-up
                </li>
                <li className="flex items-center gap-2">
                  <span className="text-green-600 font-bold">✓</span>
                  Use power-ups during gameplay by clicking their icons
                </li>
              </ul>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Game Over Screen
  if (gameState === "gameOver") {
    return (
      <div className="min-h-screen bg-gradient-to-br from-gray-800 via-gray-900 to-black flex items-center justify-center p-4">
        <div className="max-w-md w-full text-center">
          <Trophy className="w-20 h-20 sm:w-24 sm:h-24 text-yellow-400 mx-auto mb-6 animate-pulse" />

          <h1 className="text-4xl sm:text-5xl font-black text-white mb-4">
            GAME OVER!
          </h1>

          <div className="bg-white/10 backdrop-blur-lg rounded-3xl p-6 sm:p-8 mb-6 border-2 border-white/20">
            <div className="space-y-4">
              <div>
                <p className="text-white/70 text-sm uppercase tracking-wider">
                  Final Score
                </p>
                <p className="text-5xl sm:text-6xl font-black text-yellow-400">
                  {score}
                </p>
              </div>

              {score === highScore && score > 0 && (
                <div className="bg-yellow-400/20 rounded-xl p-3 animate-pulse">
                  <p className="text-yellow-300 font-bold text-lg">
                    🎉 NEW HIGH SCORE! 🎉
                  </p>
                </div>
              )}

              <div className="grid grid-cols-2 gap-4 pt-4">
                <div className="bg-white/5 rounded-xl p-3">
                  <p className="text-white/70 text-xs uppercase">Max Streak</p>
                  <p className="text-2xl font-bold text-purple-400">
                    🔥{maxStreak}
                  </p>
                </div>
                <div className="bg-white/5 rounded-xl p-3">
                  <p className="text-white/70 text-xs uppercase">High Score</p>
                  <p className="text-2xl font-bold text-green-400">
                    {highScore}
                  </p>
                </div>
              </div>
            </div>
          </div>

          {/* Achievement Notification */}
          {unlockedAchievement && (
            <div className="bg-gradient-to-r from-yellow-400 to-orange-500 rounded-2xl p-4 mb-4 animate-bounce">
              <p className="text-white font-bold text-lg mb-2">
                🎉 Achievement Unlocked! 🎉
              </p>
              <p className="text-white text-2xl">
                {unlockedAchievement.icon} {unlockedAchievement.name}
              </p>
              <p className="text-white/90 text-sm mt-1">
                {unlockedAchievement.description}
              </p>
              <p className="text-white font-semibold mt-2">
                + Power-up earned!
              </p>
            </div>
          )}

          <div className="space-y-3">
            <button
              onClick={() => startGame(difficulty)}
              className="w-full bg-gradient-to-r from-green-500 to-emerald-600 hover:from-green-600 hover:to-emerald-700 text-white font-bold py-4 px-6 rounded-2xl text-lg transition-all transform hover:scale-105 active:scale-95 shadow-xl"
            >
              <div className="flex items-center justify-center gap-2">
                <RotateCcw className="w-5 h-5" />
                <span>PLAY AGAIN</span>
              </div>
            </button>

            <button
              onClick={returnToMenu}
              className="w-full bg-white/10 hover:bg-white/20 text-white font-bold py-4 px-6 rounded-2xl text-lg transition-all border-2 border-white/30"
            >
              MAIN MENU
            </button>
          </div>
        </div>
      </div>
    );
  }

  // Playing Screen
  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-600 via-indigo-600 to-purple-700 p-4">
      <div className="max-w-4xl mx-auto py-4">
        {/* Achievement Unlock Notification */}
        {unlockedAchievement && (
          <div className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 animate-bounce">
            <div className="bg-gradient-to-r from-yellow-400 to-orange-500 rounded-2xl p-4 shadow-2xl">
              <p className="text-white font-bold text-center">
                🎉 {unlockedAchievement.icon} {unlockedAchievement.name}{" "}
                Unlocked!
              </p>
            </div>
          </div>
        )}

        {/* Stats Bar */}
        <div className="bg-white/95 backdrop-blur-sm rounded-2xl shadow-2xl p-4 mb-6">
          <div className="grid grid-cols-2 sm:grid-cols-4 gap-3 sm:gap-4">
            <div className="bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl p-3 text-center">
              <p className="text-white/80 text-xs font-semibold uppercase mb-1">
                Score
              </p>
              <p className="text-2xl sm:text-3xl font-black text-white">
                {score}
              </p>
            </div>

            <div className="bg-gradient-to-br from-red-500 to-pink-600 rounded-xl p-3 text-center">
              <div className="flex items-center justify-center gap-1 mb-1">
                {[...Array(3)].map((_, i) => (
                  <Heart
                    key={i}
                    className={`w-4 h-4 ${
                      i < lives ? "fill-white text-white" : "text-white/30"
                    }`}
                  />
                ))}
              </div>
              <p className="text-white/80 text-xs font-semibold uppercase">
                Lives
              </p>
            </div>

            <div className="bg-gradient-to-br from-purple-500 to-purple-600 rounded-xl p-3 text-center">
              <p className="text-white/80 text-xs font-semibold uppercase mb-1">
                Time
              </p>
              <p className="text-2xl sm:text-3xl font-black text-white">
                {timeLeft}s
              </p>
            </div>

            <div className="bg-gradient-to-br from-yellow-400 to-orange-500 rounded-xl p-3 text-center">
              <p className="text-white/80 text-xs font-semibold uppercase mb-1">
                Streak
              </p>
              <p className="text-2xl sm:text-3xl font-black text-white">
                🔥{streak}
              </p>
            </div>
          </div>
        </div>

        {/* Power-ups Bar */}
        <div className="bg-white/95 backdrop-blur-sm rounded-2xl shadow-2xl p-4 mb-6">
          <div className="flex items-center justify-between mb-3">
            <h3 className="text-lg font-bold text-gray-800 flex items-center gap-2">
              <Sparkles className="w-5 h-5" />
              Power-ups
            </h3>
            {activePowerUp && (
              <span className="text-sm font-semibold text-purple-600">
                Active: {powerUpTimeLeft}s
              </span>
            )}
          </div>
          <div className="grid grid-cols-3 gap-3">
            <button
              onClick={() => usePowerUp("timeFreeze")}
              disabled={activePowerUp !== null || powerUps.timeFreeze === 0}
              className={`rounded-xl p-3 text-white font-bold transition-all transform hover:scale-105 active:scale-95 disabled:opacity-50 disabled:cursor-not-allowed ${
                activePowerUp === "timeFreeze"
                  ? "bg-gradient-to-br from-cyan-600 to-blue-700 ring-4 ring-cyan-300"
                  : "bg-gradient-to-br from-cyan-400 to-blue-500"
              }`}
            >
              <Clock className="w-6 h-6 mx-auto mb-1" />
              <p className="text-xs">Freeze ({powerUps.timeFreeze})</p>
            </button>

            <button
              onClick={() => usePowerUp("shield")}
              disabled={activePowerUp !== null || powerUps.shield === 0}
              className={`rounded-xl p-3 text-white font-bold transition-all transform hover:scale-105 active:scale-95 disabled:opacity-50 disabled:cursor-not-allowed ${
                activePowerUp === "shield"
                  ? "bg-gradient-to-br from-green-600 to-emerald-700 ring-4 ring-green-300"
                  : "bg-gradient-to-br from-green-400 to-emerald-500"
              }`}
            >
              <Shield className="w-6 h-6 mx-auto mb-1" />
              <p className="text-xs">Shield ({powerUps.shield})</p>
            </button>

            <button
              onClick={() => usePowerUp("doublePoints")}
              disabled={activePowerUp !== null || powerUps.doublePoints === 0}
              className={`rounded-xl p-3 text-white font-bold transition-all transform hover:scale-105 active:scale-95 disabled:opacity-50 disabled:cursor-not-allowed ${
                activePowerUp === "doublePoints"
                  ? "bg-gradient-to-br from-yellow-600 to-orange-700 ring-4 ring-yellow-300"
                  : "bg-gradient-to-br from-yellow-400 to-orange-500"
              }`}
            >
              <Star className="w-6 h-6 mx-auto mb-1" />
              <p className="text-xs">2x ({powerUps.doublePoints})</p>
            </button>
          </div>
        </div>

        {/* Game Area */}
        <div className="bg-white/95 backdrop-blur-sm rounded-3xl shadow-2xl p-8 sm:p-12 mb-6 relative overflow-hidden">
          {/* Feedback Animation */}
          {showFeedback && (
            <div
              className={`absolute inset-0 flex items-center justify-center z-10 ${
                showFeedback === "correct"
                  ? "bg-green-500/90"
                  : showFeedback === "blocked"
                  ? "bg-blue-500/90"
                  : "bg-red-500/90"
              } animate-pulse`}
            >
              <div className="text-6xl sm:text-8xl">
                {showFeedback === "correct"
                  ? "✓"
                  : showFeedback === "blocked"
                  ? "🛡️"
                  : "✗"}
              </div>
            </div>
          )}

          {/* Active Power-up Overlay */}
          {activePowerUp && (
            <div className="absolute top-4 right-4 bg-purple-600 text-white px-4 py-2 rounded-full font-bold text-sm animate-pulse">
              {activePowerUp === "timeFreeze" && "❄️ Time Frozen"}
              {activePowerUp === "shield" && "🛡️ Shield Active"}
              {activePowerUp === "doublePoints" && "⭐ 2x Points"}
            </div>
          )}

          <div className="text-center">
            <p className="text-gray-600 text-lg sm:text-xl font-semibold mb-6 sm:mb-8">
              Does the word match its color?
            </p>

            <div className="mb-8 sm:mb-12">
              <div
                className={`text-6xl sm:text-8xl lg:text-9xl font-black ${displayColor.text} drop-shadow-lg animate-pulse`}
              >
                {currentWord}
              </div>
            </div>

            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 max-w-2xl mx-auto">
              <button
                onClick={() => handleAnswer("match")}
                disabled={showFeedback !== null}
                className="bg-gradient-to-r from-green-500 to-emerald-600 hover:from-green-600 hover:to-emerald-700 disabled:from-gray-400 disabled:to-gray-500 text-white font-black py-6 sm:py-8 px-6 rounded-2xl text-xl sm:text-2xl transition-all transform hover:scale-105 active:scale-95 shadow-xl disabled:cursor-not-allowed"
              >
                ✓ MATCH
              </button>

              <button
                onClick={() => handleAnswer("different")}
                disabled={showFeedback !== null}
                className="bg-gradient-to-r from-red-500 to-pink-600 hover:from-red-600 hover:to-pink-700 disabled:from-gray-400 disabled:to-gray-500 text-white font-black py-6 sm:py-8 px-6 rounded-2xl text-xl sm:text-2xl transition-all transform hover:scale-105 active:scale-95 shadow-xl disabled:cursor-not-allowed"
              >
                ✗ DIFFERENT
              </button>
            </div>
          </div>
        </div>

        {/* Streak Bonus Info */}
        {streak >= 3 && (
          <div className="bg-gradient-to-r from-yellow-400 to-orange-500 rounded-2xl p-4 text-center animate-bounce">
            <p className="text-white font-bold text-lg">
              🔥 Streak Bonus Active! +{Math.floor(streak / 3) * 5} points per
              correct answer!
            </p>
          </div>
        )}
      </div>
    </div>
  );
};

export default ColorClashGame;
