import { useEffect, useState } from "react";
import axios from "axios";
import { Link } from "react-router-dom";

export default function StudentList() {
  const [students, setStudents] = useState([]);

  const fetchStudents = async () => {
    const res = await axios.get("http://localhost:5000/api/students");
    setStudents(res.data);
  };

  const deleteStudent = async (id) => {
    if (confirm("Are you sure you want to delete this student?")) {
      await axios.delete(`http://localhost:5000/api/students/${id}`);
      fetchStudents();
    }
  };

  useEffect(() => {
    fetchStudents();
  }, []);

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-pink-50 to-blue-50 p-6">
      {/* Header Section */}
      <div className="flex flex-col sm:flex-row justify-between items-center mb-8">
        <div>
          <h1 className="text-4xl font-bold bg-gradient-to-r from-purple-600 via-pink-600 to-blue-600 bg-clip-text text-transparent mb-2">
            Student Management
          </h1>
          <p className="text-gray-600 text-sm">Manage your students efficiently</p>
        </div>
        <Link
          to="/add"
          className="mt-4 sm:mt-0 bg-gradient-to-r from-purple-500 to-pink-500 text-white px-6 py-3 rounded-xl font-medium shadow-lg hover:shadow-xl hover:scale-105 transition-all duration-300"
        >
          + Add New Student
        </Link>
      </div>

      {/* Table Section */}
      <div className="overflow-x-auto bg-white/80 backdrop-blur-sm shadow-2xl rounded-3xl border border-white/50">
        <table className="min-w-full border-collapse">
          <thead className="bg-gradient-to-r from-purple-500 via-pink-500 to-blue-500 text-white">
            <tr>
              <th className="px-6 py-5 text-left font-semibold text-sm uppercase tracking-wider">
                Image
              </th>
              <th className="px-6 py-5 text-left font-semibold text-sm uppercase tracking-wider">
                Name
              </th>
              <th className="px-6 py-5 text-left font-semibold text-sm uppercase tracking-wider">
                Email
              </th>
              <th className="px-6 py-5 text-left font-semibold text-sm uppercase tracking-wider">
                Course
              </th>
              <th className="px-6 py-5 text-center font-semibold text-sm uppercase tracking-wider">
                Actions
              </th>
            </tr>
          </thead>
          <tbody>
            {students.length > 0 ? (
              students.map((student, idx) => (
                <tr
                  key={student._id}
                  className="border-b border-gray-100 hover:bg-gradient-to-r hover:from-purple-50 hover:to-pink-50 transition-all duration-200"
                >
                  <td className="px-6 py-5">
                    {student.image ? (
                      <div className="flex items-center">
                        <div className="relative">
                          <div className="absolute inset-0 bg-gradient-to-r from-purple-400 to-pink-400 rounded-full blur-sm"></div>
                          <img
                            src={`http://localhost:5000${student.image}`}
                            alt={student.name}
                            className="relative w-16 h-16 object-cover rounded-full border-2 border-white shadow-lg"
                          />
                        </div>
                      </div>
                    ) : (
                      <div className="w-16 h-16 rounded-full bg-gradient-to-br from-gray-200 to-gray-300 flex items-center justify-center">
                        <span className="text-gray-500 text-xs">No Image</span>
                      </div>
                    )}
                  </td>
                  <td className="px-6 py-5">
                    <span className="text-gray-800 font-semibold text-base">
                      {student.name}
                    </span>
                  </td>
                  <td className="px-6 py-5 text-gray-600">{student.email}</td>
                  <td className="px-6 py-5">
                    <span className="inline-block px-3 py-1 bg-gradient-to-r from-blue-100 to-purple-100 text-purple-700 rounded-full text-sm font-medium">
                      {student.course}
                    </span>
                  </td>
                  <td className="px-6 py-5">
                    <div className="flex justify-center gap-2">
                      <Link
                        to={`/view/${student._id}`}
                        className="bg-gradient-to-r from-blue-500 to-cyan-500 text-white px-5 py-2 rounded-lg text-sm font-medium hover:shadow-lg hover:scale-105 transition-all duration-200"
                      >
                        View
                      </Link>
                      <Link
                        to={`/edit/${student._id}`}
                        className="bg-gradient-to-r from-green-500 to-emerald-500 text-white px-5 py-2 rounded-lg text-sm font-medium hover:shadow-lg hover:scale-105 transition-all duration-200"
                      >
                        Edit
                      </Link>
                      <button
                        onClick={() => deleteStudent(student._id)}
                        className="bg-gradient-to-r from-red-500 to-pink-500 text-white px-5 py-2 rounded-lg text-sm font-medium hover:shadow-lg hover:scale-105 transition-all duration-200"
                      >
                        Delete
                      </button>
                    </div>
                  </td>
                </tr>
              ))
            ) : (
              <tr>
                <td
                  colSpan="5"
                  className="py-12 text-center"
                >
                  <div className="flex flex-col items-center">
                    <div className="w-20 h-20 bg-gradient-to-br from-gray-200 to-gray-300 rounded-full flex items-center justify-center mb-4">
                      <span className="text-3xl">😕</span>
                    </div>
                    <p className="text-gray-500 font-medium">No students found</p>
                  </div>
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Footer */}
      <div className="text-center mt-8">
        <p className="text-gray-500 text-sm font-medium bg-gradient-to-r from-purple-600 to-pink-600 bg-clip-text text-transparent">
          © {new Date().getFullYear()} Student Management System — MERN Stack
        </p>
      </div>
    </div>
  );
}