import { useEffect, useState } from "react";
import axios from "axios";
import { useNavigate, useParams, Link } from "react-router-dom";

export default function EditStudent() {
  const { id } = useParams();
  const navigate = useNavigate();
  const [form, setForm] = useState({
    name: "",
    email: "",
    course: "",
    image: null,
    oldImage: "",
  });
  const [preview, setPreview] = useState(null);

  useEffect(() => {
    axios.get("http://localhost:5000/api/students").then((res) => {
      const student = res.data.find((s) => s._id === id);
      if (student) {
        setForm({
          name: student.name,
          email: student.email,
          course: student.course,
          image: null,
          oldImage: student.image,
        });
      }
    });
  }, [id]);

  const handleChange = (e) => {
    const { name, value, files } = e.target;
    if (files) {
      setForm({ ...form, [name]: files[0] });
      setPreview(URL.createObjectURL(files[0]));
    } else {
      setForm({ ...form, [name]: value });
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const formData = new FormData();
    formData.append("name", form.name);
    formData.append("email", form.email);
    formData.append("course", form.course);
    if (form.image) formData.append("image", form.image);
    else formData.append("image", form.oldImage);

    await axios.put(`http://localhost:5000/api/students/${id}`, formData);
    navigate("/");
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-pink-50 to-blue-50 flex items-center justify-center px-4 py-10">
      <div className="bg-white/80 backdrop-blur-sm shadow-2xl rounded-3xl p-8 w-full max-w-md border border-white/50">
        {/* Header */}
        <div className="text-center mb-6">
          <h2 className="text-3xl font-bold bg-gradient-to-r from-green-600 via-emerald-600 to-teal-600 bg-clip-text text-transparent mb-2">
            Edit Student
          </h2>
          <p className="text-gray-500 text-sm">Update student information</p>
        </div>

        {/* Current/New Image Preview */}
        <div className="flex justify-center mb-6">
          <div className="relative">
            <div className="absolute inset-0 bg-gradient-to-r from-green-400 via-emerald-400 to-teal-400 rounded-full blur-lg opacity-75"></div>
            <img
              src={preview || `http://localhost:5000${form.oldImage}`}
              alt="Student"
              className="relative w-32 h-32 rounded-full object-cover border-4 border-white shadow-2xl"
            />
          </div>
        </div>

        <form onSubmit={handleSubmit} className="flex flex-col gap-5">
          {/* Name Input */}
          <div>
            <label className="block text-gray-700 mb-2 font-medium text-sm">
              Student Name
            </label>
            <input
              name="name"
              value={form.name}
              onChange={handleChange}
              placeholder="Student name"
              className="w-full border-2 border-gray-200 p-3 rounded-xl focus:ring-2 focus:ring-green-400 focus:border-transparent outline-none transition-all"
              required
            />
          </div>

          {/* Email Input */}
          <div>
            <label className="block text-gray-700 mb-2 font-medium text-sm">
              Email Address
            </label>
            <input
              name="email"
              type="email"
              value={form.email}
              onChange={handleChange}
              placeholder="Email address"
              className="w-full border-2 border-gray-200 p-3 rounded-xl focus:ring-2 focus:ring-green-400 focus:border-transparent outline-none transition-all"
              required
            />
          </div>

          {/* Course Input */}
          <div>
            <label className="block text-gray-700 mb-2 font-medium text-sm">
              Course
            </label>
            <input
              name="course"
              value={form.course}
              onChange={handleChange}
              placeholder="Course"
              className="w-full border-2 border-gray-200 p-3 rounded-xl focus:ring-2 focus:ring-green-400 focus:border-transparent outline-none transition-all"
              required
            />
          </div>

          {/* File Upload */}
          <div>
            <label className="block text-gray-700 mb-2 font-medium text-sm">
              Change Image {preview ? "(New image selected)" : "(Optional)"}
            </label>
            <div className="relative">
              <input
                type="file"
                name="image"
                accept="image/*"
                onChange={handleChange}
                className="block w-full text-sm text-gray-600 border-2 border-gray-200 rounded-xl cursor-pointer bg-gray-50 file:mr-4 file:py-3 file:px-4 file:rounded-l-xl file:border-0 file:text-sm file:font-semibold file:bg-gradient-to-r file:from-green-500 file:to-emerald-500 file:text-white hover:file:from-green-600 hover:file:to-emerald-600 file:cursor-pointer transition-all"
              />
            </div>
          </div>

          {/* Submit Button */}
          <button
            type="submit"
            className="bg-gradient-to-r from-green-500 to-emerald-500 text-white py-3 rounded-xl font-semibold shadow-lg hover:shadow-xl hover:scale-105 transition-all duration-300 mt-2"
          >
            Update Student
          </button>

          {/* Back Button */}
          <Link
            to="/"
            className="bg-gradient-to-r from-gray-500 to-gray-600 text-white py-3 rounded-xl font-semibold shadow-lg hover:shadow-xl hover:scale-105 transition-all duration-300 text-center"
          >
            Back to List
          </Link>
        </form>
      </div>
    </div>
  );
}