import React, { useState, useEffect } from "react";
import API from "../api";
import { Link, useNavigate } from "react-router-dom";

export default function OtpVerify() {
  const [otp, setOtp] = useState("");
  const [remaining, setRemaining] = useState(30); // ✅ 30s expiry
  const [isLoading, setIsLoading] = useState(false);
  const [isExpired, setIsExpired] = useState(false);
  const nav = useNavigate();

  // Countdown timer logic
  useEffect(() => {
    if (!sessionStorage.getItem("tempToken")) {
      alert("No temporary session found. Please login again.");
      nav("/login");
      return;
    }

    const tick = () => {
      setRemaining((prev) => {
        const newVal = Math.max(0, prev - 1);
        if (newVal <= 0) setIsExpired(true);
        return newVal;
      });
    };

    const interval = setInterval(tick, 1000);
    return () => clearInterval(interval);
  }, [nav]);

  // Verify OTP
  async function handleVerify(e) {
    e.preventDefault();

    if (isExpired) {
      alert("OTP expired. Please request a new code.");
      return;
    }

    setIsLoading(true);
    try {
      const tempToken = sessionStorage.getItem("tempToken");
      const res = await API.post("/auth/verify", { tempToken, otp });
      const { token } = res.data;

      // Save JWT for 10-minute login
      localStorage.setItem("authToken", token);
      alert("OTP verified successfully!");
      nav("/dashboard");
    } catch (err) {
      alert(err.response?.data?.message || "Invalid or expired OTP");
    } finally {
      setIsLoading(false);
    }
  }

  // Format timer
  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, "0")}`;
  };

  const getTimerColor = () => {
    if (remaining <= 10) return "text-red-600";
    if (remaining <= 20) return "text-orange-600";
    return "text-green-600";
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-indigo-100 via-purple-50 to-pink-100 flex items-center justify-center p-4">
      <div className="w-full max-w-md">
        {/* Card */}
        <div className="bg-white rounded-2xl shadow-xl overflow-hidden">
          {/* Header */}
          <div className="bg-gradient-to-r from-indigo-600 to-purple-600 px-8 py-10 text-center">
            <div className="w-20 h-20 bg-white rounded-full flex items-center justify-center mx-auto mb-4">
              <svg
                className="w-10 h-10 text-indigo-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth="2"
                  d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"
                />
              </svg>
            </div>
            <h2 className="text-3xl font-bold text-white mb-2">
              Check Your Email
            </h2>
            <p className="text-indigo-100">We've sent a 4-digit OTP code</p>
          </div>

          {/* Form */}
          <div className="px-8 py-10">
            {/* Timer Display */}
            <div className="mb-8 text-center">
              <div className="inline-flex items-center justify-center space-x-2 bg-gray-50 rounded-lg px-6 py-4 border border-gray-200">
                <svg
                  className={`w-5 h-5 ${getTimerColor()}`}
                  fill="currentColor"
                  viewBox="0 0 20 20"
                >
                  <path
                    fillRule="evenodd"
                    d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z"
                    clipRule="evenodd"
                  />
                </svg>
                <div>
                  <span className="text-sm text-gray-600 block">
                    Time remaining
                  </span>
                  <span className={`text-2xl font-bold ${getTimerColor()}`}>
                    {formatTime(remaining)}
                  </span>
                </div>
              </div>

              {isExpired && (
                <div className="mt-4 bg-red-50 border border-red-200 rounded-lg p-3">
                  <p className="text-sm text-red-800 font-medium">
                    OTP expired. Please request a new one.
                  </p>
                </div>
              )}
            </div>

            {/* OTP Form */}
            <form onSubmit={handleVerify} className="space-y-6">
              <div>
                <label
                  htmlFor="otp"
                  className="block text-sm font-medium text-gray-700 mb-2"
                >
                  Verification Code
                </label>
                <input
                  id="otp"
                  type="text"
                  value={otp}
                  onChange={(e) =>
                    setOtp(e.target.value.replace(/\D/g, "").slice(0, 4))
                  }
                  placeholder="Enter 4-digit code"
                  maxLength="4"
                  required
                  disabled={isExpired}
                  className="w-full px-4 py-4 text-center text-2xl font-bold tracking-widest rounded-lg border border-gray-300 focus:ring-2 focus:ring-indigo-500 focus:border-transparent transition duration-200 outline-none disabled:bg-gray-100 disabled:cursor-not-allowed"
                />
                <p className="mt-2 text-xs text-gray-500 text-center">
                  Enter the 4-digit OTP sent to your email
                </p>
              </div>

              <button
                type="submit"
                disabled={isLoading || isExpired || otp.length !== 4}
                className="w-full bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-semibold py-3 px-4 rounded-lg hover:from-indigo-700 hover:to-purple-700 focus:ring-4 focus:ring-indigo-300 transition duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isLoading ? (
                  <span className="flex items-center justify-center">
                    <svg
                      className="animate-spin h-5 w-5 mr-3"
                      viewBox="0 0 24 24"
                    >
                      <circle
                        className="opacity-25"
                        cx="12"
                        cy="12"
                        r="10"
                        stroke="currentColor"
                        strokeWidth="4"
                        fill="none"
                      />
                      <path
                        className="opacity-75"
                        fill="currentColor"
                        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                      />
                    </svg>
                    Verifying...
                  </span>
                ) : (
                  "Verify Code"
                )}
              </button>
            </form>

            {/* Divider */}
            <div className="mt-8 relative">
              <div className="absolute inset-0 flex items-center">
                <div className="w-full border-t border-gray-300"></div>
              </div>
              <div className="relative flex justify-center text-sm">
                <span className="px-4 bg-white text-gray-500">Or</span>
              </div>
            </div>

            {/* Back to Login */}
            <div className="mt-6 text-center">
              <Link
                to="/login"
                className="text-indigo-600 font-semibold hover:text-indigo-800 transition duration-200"
              >
                ← Back to Login To Regenerate OTP
              </Link>
            </div>
          </div>
        </div>

        {/* Help Note */}
        <div className="mt-6 bg-blue-50 border border-blue-200 rounded-lg p-4">
          <div className="flex items-start">
            <svg
              className="w-5 h-5 text-blue-600 mt-0.5 mr-3 flex-shrink-0"
              fill="currentColor"
              viewBox="0 0 20 20"
            >
              <path
                fillRule="evenodd"
                d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"
                clipRule="evenodd"
              />
            </svg>
            <div>
              <p className="text-sm text-blue-800 font-medium">
                Having trouble?
              </p>
              <p className="text-xs text-blue-700 mt-1">
                Check your spam folder or request a new code.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
