const express = require("express");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const User = require("../models/User");
const { sendOtpEmail } = require("../utils/mailer");

const router = express.Router();

const OTP_EXPIRE_SECONDS = Number(process.env.OTP_EXPIRE_SECONDS || 30);
const JWT_EXPIRE_MINUTES = Number(process.env.JWT_EXPIRE_MINUTES || 10);
const TEMP_TOKEN_EXPIRE_SECONDS = Number(
  process.env.TEMP_TOKEN_EXPIRE_SECONDS || 300
);

// Generate 4 digit OTP
function getOtp() {
  return Math.floor(1000 + Math.random() * 9000).toString();
}

router.get("/", (req, res) => {
  res.json("Welcome To The OTP Authentication");
});

// register
router.post("/register", async (req, res) => {
  try {
    const { email, password } = req.body;

    if (!email || !password)
      return res.status(400).json({ message: "email and password required" });

    const existing = await User.findOne({ email });
    if (existing)
      return res.status(400).json({ message: "Email already registered" });

    const salt = await bcrypt.genSalt(10);
    const passwordHash = await bcrypt.hash(password, salt);

    const user = await User.create({ email, passwordHash });
    return res.json({ message: "User registered", userId: user._id });
  } catch (err) {
    console.error(err);
    return res.status(500).json({ message: "Server error" });
  }
});

// login - verify email/password, generate OTP, send it, return a tempToken for OTP verification
router.post("/login", async (req, res) => {
  try {
    const { email, password } = req.body;
    if (!email || !password)
      return res.status(400).json({ message: "email and password required" });

    const user = await User.findOne({ email });
    if (!user) return res.status(400).json({ message: "Invalid credentials" });

    const match = await bcrypt.compare(password, user.passwordHash);
    if (!match) return res.status(400).json({ message: "Invalid credentials" });

    // create OTP and expiry
    const otp = getOtp();
    const otpExpiresAt = new Date(Date.now() + OTP_EXPIRE_SECONDS * 1000);
    user.otp = otp;
    user.otpExpiresAt = otpExpiresAt;
    await user.save();

    // send mail
    await sendOtpEmail(user.email, otp);

    // create a temp token to identify user in verify step
    const tempToken = jwt.sign(
      { userId: user._id },
      process.env.OTP_JWT_SECRET || process.env.JWT_SECRET,
      { expiresIn: `${TEMP_TOKEN_EXPIRE_SECONDS}s` }
    );

    return res.json({
      message: "OTP sent to email",
      tempToken,
      expiresIn: TEMP_TOKEN_EXPIRE_SECONDS,
    });
  } catch (err) {
    console.error(err);
    return res.status(500).json({ message: "Server error" });
  }
});

// Verify OTP -> if ok, JWT Time (10 mins)
router.post("/verify", async (req, res) => {
  try {
    const { tempToken, otp } = req.body;
    if (!tempToken || !otp)
      return res.status(400).json({ message: "tempToken and otp required" });

    // Verify temp token
    let decode;
    try {
      decode = jwt.verify(
        tempToken,
        process.env.OTP_JWT_SECRET || process.env.JWT_SECRET
      );
    } catch (e) {
      return res.status(401).json({ message: "Invalid or expired temp token" });
    }

    const user = await User.findById(decode.userId);
    if (!user) return res.status(400).json({ message: "User not found" });

    // check otp and expiry
    if (!user.otp || !user.otpExpiresAt)
      return res
        .status(400)
        .json({ message: "No OTP found. Please login again." });

    if (new Date() > user.otpExpiresAt) {
      // clear stored otp
      user.otp = undefined;
      user.otpExpiresAt = undefined;
      await user.save();
      return res
        .status(400)
        .json({ message: "OTP expired. Please login again." });
    }

    if (user.otp !== otp)
      return res.status(400).json({ message: "Incorrect OTP" });

    // success: clear otp and issue jwt
    user.otp = undefined;
    user.otpExpiresAt = undefined;
    await user.save();

    const token = jwt.sign(
      { userId: user._id, email: user.email },
      process.env.JWT_SECRET,
      { expiresIn: `${JWT_EXPIRE_MINUTES}m` }
    );

    return res.json({
      message: "Authenticated",
      token,
      expiresInMinutes: JWT_EXPIRE_MINUTES,
    });
  } catch (err) {
    console.error(err);
    return res.status(500).json({ message: "Server error" });
  }
});

// protected route

const authMiddleware = (req, res, next) => {
  const authHeader = req.headers.authorization;
  if (!authHeader) return res.status(401).json({ message: "No token" });

  const token = authHeader.split(" ")[1];
  if (!token) return res.status(401).json({ message: "No token" });

  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (e) {
    return res.status(401).json({ message: "Invalid or expired token" });
  }
};

router.get("/me", async (req, res) => {
  try {
    const user = await User.findById(req.params.userId).select(
      "-passwordHash -otp -otpExpiresAt"
    );
    return res.json({ user });
  } catch (err) {
    return res.status(500).json({ message: "Server error" });
  }
});

module.exports = router;
