import React, { useState, useRef } from "react";

// compressImage: compresses an image file to approximately targetKB (kilobytes)
async function compressImage(file, targetKB = 100, options = {}) {
  const maxIterations = 12;
  const targetBytes = targetKB * 1024;
  const img = await createImageBitmap(file);

  let mime = "image/jpeg";
  if (file.type === "image/png") mime = "image/png";

  // start with original dimensions
  let [w, h] = [img.width, img.height];

  // helper to get blob at given quality and dimensions
  const toBlob = (width, height, quality) =>
    new Promise((res) => {
      const canvas = document.createElement("canvas");
      canvas.width = Math.max(1, Math.floor(width));
      canvas.height = Math.max(1, Math.floor(height));
      const ctx = canvas.getContext("2d");
      // maintain aspect ratio and draw
      ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
      canvas.toBlob(res, mime, quality);
    });

  // If PNG and has alpha, we'll convert to jpeg for better compression unless user insisted PNG
  if (mime === "image/png") {
    // convert to jpeg for compressing (most of the time smaller)
    mime = "image/jpeg";
  }

  // Strategy:
  // 1. Try reducing quality from 0.92 down to 0.3
  // 2. If not enough, scale down dimensions progressively and repeat
  let quality = 0.92;
  for (let iter = 0; iter < maxIterations; iter++) {
    // inner loop reduce quality
    for (let qIter = 0; qIter < 7; qIter++) {
      const q = Math.max(0.1, quality - qIter * 0.12);
      // draw and get blob
      const blob = await toBlob(w, h, q);
      if (!blob) continue;
      if (blob.size <= targetBytes || q <= 0.11) {
        // success or reached min quality
        return blob;
      }
    }
    // if still too big, reduce dimensions and try again
    w = Math.floor(w * 0.8);
    h = Math.floor(h * 0.8);
  }
  // fallback: return last available blob at very low quality
  return await toBlob(w, h, 0.1);
}

export default function ImageCompressor() {
  const [file, setFile] = useState(null);
  const [originalUrl, setOriginalUrl] = useState(null);
  const [resultUrl, setResultUrl] = useState(null);
  const [status, setStatus] = useState("");
  const [targetKB, setTargetKB] = useState(100);
  const [processing, setProcessing] = useState(false);
  const downloadRef = useRef();

  const onFile = (f) => {
    if (!f) return;
    setFile(f);
    setOriginalUrl(URL.createObjectURL(f));
    setResultUrl(null);
    setStatus("");
  };

  const handleCompress = async () => {
    if (!file) return setStatus("Please choose an image file first.");
    setProcessing(true);
    setStatus("Compressing...");
    try {
      const blob = await compressImage(file, Number(targetKB));
      const url = URL.createObjectURL(blob);
      setResultUrl(url);
      setStatus(`Done — ${Math.round(blob.size / 1024)} KB`);
    } catch (e) {
      console.error(e);
      setStatus("Compression failed: " + e.message);
    } finally {
      setProcessing(false);
    }
  };

  const handleDownload = () => {
    if (!resultUrl) return;
    const a = document.createElement("a");
    a.href = resultUrl;
    // suggest filename
    const name =
      file && file.name ? file.name.replace(/\.[^.]+$/, "") : "image";
    a.download = `${name}-compressed.jpg`;
    a.click();
  };

  return (
    <div className="space-y-4">
      <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
        <div className="sm:col-span-2">
          <label className="block text-sm font-medium text-slate-300 mb-2">
            Select image
          </label>
          <input
            accept="image/*"
            type="file"
            onChange={(e) => onFile(e.target.files?.[0])}
            className="w-full text-slate-200 file:bg-slate-700 file:text-slate-100 file:px-3 file:py-2 file:rounded-md"
          />
          <p className="text-xs text-slate-400 mt-2">
            Client-side only — images are not uploaded to any server.
          </p>
        </div>

        <div className="flex flex-col justify-between">
          <div>
            <label className="block text-sm font-medium text-slate-300 mb-2">
              Target size (KB)
            </label>
            <input
              type="number"
              value={targetKB}
              onChange={(e) => setTargetKB(e.target.value)}
              className="w-full rounded-md bg-slate-800 border border-slate-700 px-3 py-2 text-slate-100"
            />
          </div>
          <div className="mt-3">
            <button
              onClick={handleCompress}
              disabled={processing}
              className="w-full inline-flex items-center justify-center px-4 py-2 bg-emerald-500 hover:bg-emerald-600 text-slate-900 font-semibold rounded-md shadow-md disabled:opacity-60"
            >
              {processing ? "Processing..." : "Convert"}
            </button>
            <button
              onClick={() => {
                setFile(null);
                setOriginalUrl(null);
                setResultUrl(null);
                setStatus("");
              }}
              className="w-full mt-2 px-4 py-2 border border-slate-700 rounded-md text-slate-300"
            >
              Reset
            </button>
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div className="bg-slate-800/50 rounded-md p-3 flex flex-col items-center">
          <h3 className="text-sm font-semibold text-slate-200 mb-2">
            Original
          </h3>
          {originalUrl ? (
            <>
              <img
                src={originalUrl}
                alt="original"
                className="max-h-60 object-contain rounded-md border border-slate-700"
              />
              <p className="text-xs text-slate-400 mt-2">
                {file.name} • {Math.round(file.size / 1024)} KB
              </p>
            </>
          ) : (
            <div className="text-slate-500 text-sm">No image selected</div>
          )}
        </div>

        <div className="bg-slate-800/50 rounded-md p-3 flex flex-col items-center">
          <h3 className="text-sm font-semibold text-slate-200 mb-2">Result</h3>
          {resultUrl ? (
            <>
              <img
                src={resultUrl}
                alt="result"
                className="max-h-60 object-contain rounded-md border border-slate-700"
              />
              <p className="text-xs text-slate-400 mt-2">{status}</p>
              <div className="mt-3 flex gap-2">
                <button
                  onClick={handleDownload}
                  className="px-3 py-1 bg-indigo-500 rounded text-sm font-medium"
                >
                  Download
                </button>
                <a
                  href={resultUrl}
                  target="_blank"
                  rel="noreferrer"
                  className="px-3 py-1 border rounded text-sm"
                >
                  Open
                </a>
              </div>
            </>
          ) : (
            <div className="text-slate-500 text-sm">No result yet</div>
          )}
        </div>
      </div>

      <div className="text-sm text-slate-300">{status}</div>
    </div>
  );
}
