const express = require('express');
const router = express.Router();
const Inquiry = require('../models/Inquiry');
const nodemailer = require('nodemailer');
const { auth, requireAdmin } = require('../middleware/auth');

// Submit a new inquiry (public)
router.post('/', async (req, res) => {
  try {
    const { name, email, subject, message } = req.body;
    const ip = req.ip || req.headers['x-forwarded-for'] || '';
    const inquiry = new Inquiry({ name, email, subject, message, ip });
    await inquiry.save();

    // Send confirmation email to the user
    const transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST,
      port: Number(process.env.SMTP_PORT),
      secure: process.env.SMTP_SECURE === 'true', // true for 465, false for other ports
      auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASS
      }
    });

    const mailOptions = {
      from: process.env.FROM_EMAIL,
      to: email,
      subject: `Thanks for contacting us${subject ? ' — ' + subject : ''}`,
      text: `Hi ${name || ''},\n\nThanks for reaching out. We received your message:\n\n${message}\n\nWe'll get back to you soon.\n\nRegards,\nTeam`
    };

    transporter.sendMail(mailOptions, (err, info) => {
      if (err) console.error('Mail send error', err);
    });

    res.status(201).json({ message: 'Inquiry submitted', inquiry });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: err.message });
  }
});

// Admin: list all inquiries
router.get('/', auth, requireAdmin, async (req, res) => {
  try {
    const list = await Inquiry.find().sort({ createdAt: -1 });
    res.json(list);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
});

// Admin: delete inquiry by id
router.delete('/:id', auth, requireAdmin, async (req, res) => {
  try {
    const item = await Inquiry.findByIdAndDelete(req.params.id);
    if (!item) return res.status(404).json({ message: 'Not found' });
    res.json({ message: 'Deleted' });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
});

module.exports = router;
