import React, { useEffect, useState } from 'react';
import API from '../api';
import BlogCard from '../components/BlogCard';
import { Link } from 'react-router-dom';

export default function Home() {
  const [blogs, setBlogs] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    let mounted = true;
    API.get('/blogs')
      .then(res => { if (mounted) setBlogs(res.data); })
      .catch(err => console.error(err))
      .finally(() => { if (mounted) setLoading(false); });
    return () => (mounted = false);
  }, []);

  return (
    <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
      {/* Hero */}
      <section className="bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 rounded-2xl p-8 mb-8 text-white shadow-lg">
        <div className="md:flex md:items-center md:justify-between">
          <div>
            <h1 className="text-3xl sm:text-4xl font-extrabold leading-tight">Welcome to BlogSphere</h1>
            <p className="mt-2 text-gray-300 max-w-2xl">
              Discover hand-picked stories, ideas, and guides. Write your first post and share it with the world —
              reading is simpler and delightful here.
            </p>
            <div className="mt-4 flex gap-3">
              <Link to="/create" className="bg-amber-400 text-black px-4 py-2 rounded-md font-medium shadow hover:shadow-md">Create a Post</Link>
              <a href="#latest" className="text-gray-200 px-4 py-2 rounded-md hover:text-amber-300">See latest</a>
            </div>
          </div>
        </div>
      </section>

      {/* Latest blogs */}
      <section id="latest" className="mb-8">
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-2xl font-semibold">Latest posts</h2>
          <div className="text-sm text-gray-400">Total: {blogs.length}</div>
        </div>

        {loading ? (
          <div className="text-center py-12 text-gray-400">Loading posts…</div>
        ) : blogs.length === 0 ? (
          <div className="rounded-lg border border-dashed border-gray-700 p-8 text-center text-gray-400">
            No blogs yet. Be the first to <Link to="/create" className="text-amber-400">publish</Link>!
          </div>
        ) : (
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
            {blogs.map(b => <BlogCard key={b._id} blog={b} />)}
          </div>
        )}
      </section>

      {/* Footer CTA */}
      <section className="mt-8 rounded-lg p-6 border border-gray-800 bg-gray-900/40 text-gray-200">
        <div className="sm:flex sm:items-center sm:justify-between">
          <div>
            <h3 className="text-lg font-semibold">Enjoy what you read?</h3>
            <p className="text-sm text-gray-400">Follow authors you like and come back for more curated content.</p>
          </div>
          <div className="mt-4 sm:mt-0">
            <Link to="/register" className="px-4 py-2 bg-amber-400 text-black rounded-md font-medium">Join now</Link>
          </div>
        </div>
      </section>
    </main>
  );
}
