import React, { useState, useRef } from "react";
import API from "../api";
import { useNavigate } from "react-router-dom";
import { toast } from "react-toastify";

export default function CreateBlog() {
  const [form, setForm] = useState({ title: "", summary: "", content: "" });
  const [imageFile, setImageFile] = useState(null);
  const [preview, setPreview] = useState(null);
  const [loading, setLoading] = useState(false);
  const fileRef = useRef();
  const nav = useNavigate();

  const onFile = (file) => {
    if (!file) return setImageFile(null);
    setImageFile(file);
    setPreview(URL.createObjectURL(file));
  };

  const handleDrop = (e) => {
    e.preventDefault();
    const file = e.dataTransfer.files[0];
    if (file && file.type.startsWith("image/")) onFile(file);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    try {
      const data = new FormData();
      data.append("title", form.title);
      data.append("summary", form.summary);
      data.append("content", form.content);
      if (imageFile) data.append("image", imageFile);

      const res = await API.post("/blogs", data, {
        headers: { "Content-Type": "multipart/form-data" },
      });
      nav(`/blog/${res.data._id}`);
      toast.success("Blog Publisj successfully!");
    } catch (err) {
      toast.error(
        err.response?.data?.message || err.message || "Publish failed"
      );
    } finally {
      setLoading(false);
    }
  };

  return (
    <main className="max-w-3xl mx-auto px-4 py-10">
      <div className="bg-gray-900 rounded-2xl p-6 shadow">
        <h2 className="text-2xl font-semibold text-white mb-2">
          Create a new post
        </h2>
        <p className="text-sm text-gray-400 mb-4">
          Write something awesome — keep it clear and helpful.
        </p>

        <form onSubmit={handleSubmit} className="space-y-4">
          <input
            required
            placeholder="Title"
            value={form.title}
            onChange={(e) => setForm({ ...form, title: e.target.value })}
            className="w-full bg-gray-800 border text-white focus:outline-0 focus:border-amber-300 border-gray-700 rounded-md px-4 py-3"
          />

          <input
            placeholder="Short summary (optional)"
            value={form.summary}
            onChange={(e) => setForm({ ...form, summary: e.target.value })}
            className="w-full bg-gray-800 border text-white focus:outline-0 focus:border-amber-300 border-gray-700 rounded-md px-4 py-3"
          />

          <textarea
            required
            placeholder="Write your content here..."
            rows={10}
            value={form.content}
            onChange={(e) => setForm({ ...form, content: e.target.value })}
            className="w-full bg-gray-800 border text-white focus:outline-0 focus:border-amber-300 border-gray-700 rounded-md px-4 py-3"
          />

          {/* Image upload */}
          <div
            onDrop={handleDrop}
            onDragOver={(e) => e.preventDefault()}
            className="border-2 border-dashed border-gray-700 rounded-md p-4 flex items-center justify-between gap-4"
          >
            <div className="flex items-center gap-4">
              <div className="text-gray-400 text-sm">
                Drag & drop an image here, or
              </div>
              <label className="bg-amber-400 text-black px-3 py-2 rounded-md cursor-pointer">
                <span>Select image</span>
                <input
                  ref={fileRef}
                  type="file"
                  accept="image/*"
                  className="hidden"
                  onChange={(e) => onFile(e.target.files[0])}
                />
              </label>
            </div>

            <div className="flex items-center gap-3">
              {preview ? (
                <div className="w-24 h-16 rounded-md overflow-hidden border border-gray-700">
                  <img
                    src={preview}
                    alt="preview"
                    className="w-full h-full object-cover"
                  />
                </div>
              ) : (
                <div className="text-gray-500 text-sm">No image</div>
              )}

              {imageFile && (
                <button
                  type="button"
                  onClick={() => {
                    setImageFile(null);
                    setPreview(null);
                    fileRef.current.value = null;
                  }}
                  className="text-sm text-red-400 hover:underline"
                >
                  Remove
                </button>
              )}
            </div>
          </div>

          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-400">
              You can upload JPG/PNG up to 5MB
            </div>
            <button
              type="submit"
              className="bg-amber-400 text-black px-4 py-2 rounded-md font-medium shadow hover:bg-amber-500"
              disabled={loading}
            >
              {loading ? "Publishing…" : "Publish"}
            </button>
          </div>
        </form>
      </div>
    </main>
  );
}
