import React, { useEffect, useState } from "react";
import API from "../api";

export default function AdminPanel() {
  const [users, setUsers] = useState([]);
  const [blogs, setBlogs] = useState([]);
  const [loading, setLoading] = useState(true);

  const fetchAll = async () => {
    try {
      const [u, b] = await Promise.all([
        API.get("/admin/users"),
        API.get("/admin/blogs"),
      ]);
      setUsers(u.data);
      setBlogs(b.data);
    } catch (err) {
      alert(err.response?.data?.message || "Failed to fetch admin data");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchAll();
  }, []);

  const deleteUser = async (id) => {
    if (!confirm("Delete this user and their blogs?")) return;
    await API.delete(`/admin/users/${id}`);
    fetchAll();
  };

  const deleteBlog = async (id) => {
    if (!confirm("Delete this blog?")) return;
    await API.delete(`/admin/blogs/${id}`);
    fetchAll();
  };

  if (loading)
    return <div className="p-8 text-gray-400">Loading dashboard…</div>;

  return (
    <main className="max-w-6xl mx-auto px-4 py-10">
      <header className="mb-6 flex items-center justify-between">
        <h1 className="text-2xl font-semibold">Admin Dashboard</h1>
        <div className="text-sm text-gray-400">Manage users and content</div>
      </header>

      <section className="grid grid-cols-1 sm:grid-cols-3 gap-6 mb-8">
        <div className="bg-gray-200 rounded-lg p-4">
          <div className="text-sm text-gray-400">Users</div>
          <div className="text-2xl font-bold mt-2">{users.length}</div>
        </div>
        <div className="bg-gray-200 rounded-lg p-4">
          <div className="text-sm text-gray-400">Blogs</div>
          <div className="text-2xl font-bold mt-2">{blogs.length}</div>
        </div>
        <div className="bg-gray-200 rounded-lg p-4">
          <div className="text-sm text-gray-400">Active admins</div>
          <div className="text-2xl font-bold mt-2">
            {users.filter((u) => u.role === "admin").length}
          </div>
        </div>
      </section>

      <section className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div className="bg-gray-200 rounded-lg p-4">
          <h2 className="text-lg font-semibold mb-3">Users</h2>
          <div className="space-y-2">
            {users.map((u) => (
              <div
                key={u._id}
                className="flex items-center justify-between p-3 rounded-md border border-gray-800"
              >
                <div>
                  <div className="font-medium">{u.name}</div>
                  <div className="text-xs text-gray-400">
                    {u.email} • {u.role}
                  </div>
                </div>
                <div className="flex items-center gap-2">
                  <button
                    className="text-sm text-red-400"
                    onClick={() => deleteUser(u._id)}
                  >
                    Delete
                  </button>
                </div>
              </div>
            ))}
          </div>
        </div>

        <div className="bg-gray-200 rounded-lg p-4">
          <h2 className="text-lg font-semibold mb-3">Blogs</h2>
          <div className="space-y-2">
            {blogs.map((b) => (
              <div
                key={b._id}
                className="flex items-center justify-between p-3 rounded-md border border-gray-800"
              >
                <div>
                  <div className="font-medium">{b.title}</div>
                  <div className="text-xs text-gray-400">
                    {b.author?.name} •{" "}
                    {new Date(b.createdAt).toLocaleDateString()}
                  </div>
                </div>
                <div className="flex items-center gap-2">
                  <a href={`/blog/${b._id}`} className="text-sm text-amber-300">
                    View
                  </a>
                  <button
                    className="text-sm text-red-400"
                    onClick={() => deleteBlog(b._id)}
                  >
                    Delete
                  </button>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>
    </main>
  );
}
