import React, { useState, useEffect } from "react";
import { Link, useNavigate, useLocation } from "react-router-dom";

export default function Navbar() {
  const navigate = useNavigate();
  const location = useLocation();
  const [user, setUser] = useState(() =>
    JSON.parse(localStorage.getItem("user") || "null")
  );
  const [open, setOpen] = useState(false);

  // Sync user state when localStorage changes or custom event triggers
  useEffect(() => {
    const syncUser = () =>
      setUser(JSON.parse(localStorage.getItem("user") || "null"));
    window.addEventListener("storage", syncUser);
    window.addEventListener("userUpdated", syncUser);
    return () => {
      window.removeEventListener("storage", syncUser);
      window.removeEventListener("userUpdated", syncUser);
    };
  }, []);

  // Logout
  const handleLogout = () => {
    localStorage.removeItem("token");
    localStorage.removeItem("user");
    setUser(null);
    // trigger global event so Navbar updates immediately
    window.dispatchEvent(new Event("userUpdated"));
    navigate("/");
  };

  // Styled link helper
  const NavLink = ({ to, children }) => (
    <Link
      to={to}
      className={`px-3 py-2 rounded-md text-sm font-medium transition ${
        location.pathname === to
          ? "bg-amber-400/20 text-amber-400"
          : "text-gray-100 hover:text-amber-300"
      }`}
      onClick={() => setOpen(false)}
    >
      {children}
    </Link>
  );

  return (
    <header className="sticky top-0 z-40 backdrop-blur bg-gray-900/80 border-b border-gray-800">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center h-16">
          {/* Logo */}
          <Link to="/" className="flex items-center gap-2">
            <div className="bg-amber-400/20 text-amber-300 w-10 h-10 rounded-lg flex items-center justify-center text-lg font-bold">
              BS
            </div>
            <span className="hidden sm:block text-white font-semibold">
              BlogSphere
            </span>
          </Link>

          {/* Desktop links */}
          <nav className="hidden md:flex items-center space-x-2">
            <NavLink to="/">Home</NavLink>

            {!user && (
              <>
                <NavLink to="/login">Login</NavLink>
                <NavLink to="/register">Register</NavLink>
              </>
            )}

            {user && (
              <>
                <NavLink to="/create">Create Blog</NavLink>
                {user.role === "admin" && <NavLink to="/admin">Admin</NavLink>}
                <button
                  onClick={handleLogout}
                  className="ml-3 px-3 py-2 rounded-md text-sm font-medium bg-amber-400 text-black hover:bg-amber-500 transition"
                >
                  Logout
                </button>
              </>
            )}
          </nav>

          {/* Mobile toggle */}
          <div className="md:hidden flex items-center">
            <button
              onClick={() => setOpen(!open)}
              className="p-2 rounded-md hover:bg-gray-800/60 focus:outline-none"
            >
              <svg
                className="w-6 h-6 text-white"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                {open ? (
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth="2"
                    d="M6 18L18 6M6 6l12 12"
                  />
                ) : (
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth="2"
                    d="M4 6h16M4 12h16M4 18h16"
                  />
                )}
              </svg>
            </button>
          </div>
        </div>
      </div>

      {/* Mobile dropdown */}
      {open && (
        <div className="md:hidden px-4 pb-4 space-y-2">
          <NavLink to="/">Home</NavLink>

          {!user && (
            <>
              <NavLink to="/login">Login</NavLink>
              <NavLink to="/register">Register</NavLink>
            </>
          )}

          {user && (
            <>
              <NavLink to="/create">Create Blog</NavLink>
              {user.role === "admin" && <NavLink to="/admin">Admin</NavLink>}
              <button
                onClick={handleLogout}
                className="w-full text-left px-3 py-2 rounded-md bg-amber-400 text-black"
              >
                Logout
              </button>
            </>
          )}
        </div>
      )}
    </header>
  );
}
