const express = require('express');
const Blog = require('../models/Blog');
const { auth } = require('../middlewares/authMiddleware');
const upload = require('../middlewares/multerConfig');

const router = express.Router();

// list all blogs
router.get('/', async (req, res) => {
  try {
    const blogs = await Blog.find().populate('author', 'name email').sort({ createdAt: -1 });
    res.json(blogs);
  } catch (err) { res.status(500).json({ error: err.message }); }
});

// get a single blog by id
router.get('/:id', async (req, res) => {
  try {
    const blog = await Blog.findById(req.params.id).populate('author', 'name email');
    if (!blog) return res.status(404).json({ message: 'Not found' });
    res.json(blog);
  } catch (err) { res.status(500).json({ error: err.message }); }
});

// create blog (auth)
router.post('/', auth, upload.single('image'), async (req, res) => {
  try {
    const { title, summary, content } = req.body;
    const image = req.file ? `/uploads/${req.file.filename}` : undefined;
    const blog = new Blog({ author: req.user._id, title, summary, content, image });
    await blog.save();
    res.json(blog);
  } catch (err) { res.status(500).json({ error: err.message }); }
});

// update blog (auth owner)
router.put('/:id', auth, upload.single('image'), async (req, res) => {
  try {
    const blog = await Blog.findById(req.params.id);
    if (!blog) return res.status(404).json({ message: 'Not found' });
    if (String(blog.author) !== String(req.user._id) && req.user.role !== 'admin') {
      return res.status(403).json({ message: 'Not allowed' });
    }
    const { title, summary, content } = req.body;
    if (title) blog.title = title;
    if (summary) blog.summary = summary;
    if (content) blog.content = content;
    if (req.file) blog.image = `/uploads/${req.file.filename}`;
    await blog.save();
    res.json(blog);
  } catch (err) { res.status(500).json({ error: err.message }); }
});

// delete blog (auth owner or admin)
router.delete('/:id', auth, async (req, res) => {
  try {
    const blog = await Blog.findById(req.params.id);
    if (!blog) return res.status(404).json({ message: 'Not found' });
    if (String(blog.author) !== String(req.user._id) && req.user.role !== 'admin') {
      return res.status(403).json({ message: 'Not allowed' });
    }
    await blog.remove();
    res.json({ message: 'Deleted' });
  } catch (err) { res.status(500).json({ error: err.message }); }
});

module.exports = router;
