const express = require("express");
const User = require("../models/User");
const Blog = require("../models/Blog");
const jwt = require("jsonwebtoken");
const bcrypt = require("bcryptjs");
const { auth, adminOnly } = require("../middlewares/authMiddleware");

const router = express.Router();

/* ---------------------- PUBLIC ROUTES ---------------------- */

// Admin Register (for Postman or seed use only)
router.post("/register", async (req, res) => {
  try {
    const { name, email, password } = req.body;
    const existing = await User.findOne({ email });
    if (existing)
      return res.status(400).json({ message: "Email already exists" });

    const hashed = await bcrypt.hash(password, 10);
    const admin = await User.create({
      name,
      email,
      passwordHash: hashed,
      role: "admin",
    });

    res.json({ message: "Admin registered successfully", admin });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
});

// Admin Login
router.post("/login", async (req, res) => {
  try {
    const { email, password } = req.body;
    const admin = await User.findOne({ email, role: "admin" });
    if (!admin)
      return res.status(401).json({ message: "Invalid admin credentials" });

    const match = await bcrypt.compare(password, admin.passwordHash);
    if (!match)
      return res.status(401).json({ message: "Invalid admin credentials" });

    const token = jwt.sign(
      { id: admin._id, role: admin.role },
      process.env.JWT_SECRET,
      { expiresIn: "7d" }
    );

    res.json({
      message: "Admin login successful",
      token,
      admin: {
        id: admin._id,
        name: admin.name,
        email: admin.email,
        role: admin.role,
      },
    });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
});

/* ---------------------- PROTECTED ROUTES ---------------------- */

router.use(auth, adminOnly);

// list all users
router.get("/users", async (req, res) => {
  const users = await User.find().select("-passwordHash");
  res.json(users);
});

// delete user
router.delete("/users/:id", async (req, res) => {
  await User.findByIdAndDelete(req.params.id);
  await Blog.deleteMany({ author: req.params.id });
  res.json({ message: "User and user blogs deleted" });
});

// list all blogs (admin)
router.get("/blogs", async (req, res) => {
  const blogs = await Blog.find().populate("author", "name email");
  res.json(blogs);
});

// delete any blog (admin)
router.delete("/blogs/:id", async (req, res) => {
  await Blog.findByIdAndDelete(req.params.id);
  res.json({ message: "Blog deleted" });
});

module.exports = router;
