// --- Initial Data and Element Selection ---
const quotes = {
  easy: [
    "Dog ran fast, ate big red apple.",
    "A quiet life can be a good life.",
    "The sun is warm and the sky is blue.",
    "We can see a great deal on Earth.",
    "They use a pen to sign the check.",
  ],
  medium: [
    "Typing fast requires focus and rhythm.",
    "Practice makes a typist perfect, so keep going.",
    "Speed is nothing without accuracy and purpose.",
    "Consistency beats speed every single time you race.",
    "Stay calm and keep typing, remember your posture.",
  ],
  hard: [
    "Jaded zombies acted quaintly but kept driving their oxen forward.",
    "Coding is a magical superpower in the 21st century; master the syntax!",
    "The quick brown fox jumps over the lazy dog, simultaneously breaking the sound barrier.",
    "Exquisite paragraphs require careful thought and punctuation, often including semi-colons; use them correctly.",
    "Few black taxis zipped past the gigantic marble monument, their headlights reflecting sharply off the slick pavement.",
  ],
};

let timer,
  timeLeft,
  currentQuote,
  wpmData = [],
  chartInstance;
let correctChars = 0;
let totalCharsTyped = 0;
let totalErrors = 0;
let isRunning = false;

// DOM Elements
const quoteEl = document.getElementById("quote");
const inputEl = document.getElementById("input");
const timeEl = document.getElementById("time");
const wpmEl = document.getElementById("wpm");
const accEl = document.getElementById("accuracy");
const resultPopup = document.getElementById("result-popup");
const finalWpm = document.getElementById("final-wpm");
const finalGrossWpm = document.getElementById("final-gross-wpm");
const finalAcc = document.getElementById("final-accuracy");
const finalErrors = document.getElementById("final-errors");
const boardEl = document.getElementById("board");
const historyListEl = document.getElementById("history-list");
const themeBtn = document.getElementById("theme-toggle");
const startBtn = document.getElementById("start-btn");
const restartBtn = document.getElementById("restart-btn");
const resetTestBtn = document.getElementById("reset-test-btn");
const timerSelect = document.getElementById("timer");
const difficultySelect = document.getElementById("difficulty");
const setupEl = document.querySelector(".setup");
const testAreaEl = document.getElementById("test-area");

const clearHistoryBtn = document.getElementById("clear-history-btn");

// --- Event Listeners and Initial Setup ---

// Theme toggle
themeBtn.addEventListener("click", () => {
  document.body.classList.toggle("dark");
  themeBtn.textContent = document.body.classList.contains("dark") ? "☀️" : "🌙";
  localStorage.setItem(
    "theme",
    document.body.classList.contains("dark") ? "dark" : "light"
  );
});
if (localStorage.getItem("theme") === "dark") {
  document.body.classList.add("dark");
  themeBtn.textContent = "☀️";
}

startBtn.addEventListener("click", startTest);
restartBtn.addEventListener("click", resetUI);
resetTestBtn.addEventListener("click", resetTest);
inputEl.addEventListener("input", checkInput);
clearHistoryBtn.addEventListener("click", clearHistory);

timerSelect.addEventListener("change", () => {
  timeEl.textContent = timerSelect.value;
});

// Initial displays
showLeaderboard();
showHistory();
timeEl.textContent = timerSelect.value;

// --- Core Logic Functions ---

function startTest() {
  if (isRunning) return;

  isRunning = true;

  // Determine quote and time based on setup controls
  const difficulty = difficultySelect.value;
  const quoteList = quotes[difficulty];
  currentQuote = quoteList[Math.floor(Math.random() * quoteList.length)];
  timeLeft = parseInt(timerSelect.value);

  // UI Setup: Hide setup, show test area
  setupEl.classList.add("hidden");
  testAreaEl.classList.remove("hidden");

  // Reset state variables
  correctChars = 0;
  totalCharsTyped = 0;
  totalErrors = 0;
  wpmData = [];
  inputEl.value = "";
  accEl.textContent = 100;
  wpmEl.textContent = 0;
  timeEl.textContent = timeLeft;

  quoteEl.textContent = currentQuote;
  inputEl.disabled = false;
  inputEl.focus();

  timer = setInterval(updateTimer, 1000);
}

function resetTest() {
  // This is the in-test reset button logic
  if (!isRunning) return;

  clearInterval(timer);
  isRunning = false;

  // Reset UI back to setup
  testAreaEl.classList.add("hidden");
  setupEl.classList.remove("hidden");

  // Reset stats display
  timeEl.textContent = timerSelect.value;
  wpmEl.textContent = 0;
  accEl.textContent = 100;
  inputEl.disabled = true;
  inputEl.value = "";
  quoteEl.innerHTML = "";

  if (chartInstance) {
    chartInstance.destroy();
  }

  alert("Test has been reset.");
}

function resetUI() {
  // Resets the UI to the initial setup screen after the test ends
  resultPopup.classList.add("hidden"); // Hide the result popup
  testAreaEl.classList.add("hidden"); // Hide the test area
  setupEl.classList.remove("hidden"); // Show the setup screen

  // Clear test data for the next run
  wpmEl.textContent = 0;
  accEl.textContent = 100;
  timeEl.textContent = timerSelect.value;
  inputEl.value = "";

  if (chartInstance) {
    chartInstance.destroy(); // Destroy old chart
  }
}

function updateTimer() {
  timeLeft--;
  timeEl.textContent = timeLeft;

  let elapsed = parseInt(timerSelect.value) - timeLeft;

  let netWpm = Math.round(correctChars / 5 / (elapsed / 60)) || 0;

  wpmData.push(netWpm);
  wpmEl.textContent = netWpm;

  if (timeLeft <= 0) {
    endTest();
  }
}

function checkInput() {
  const typed = inputEl.value;
  totalCharsTyped = typed.length;

  if (totalCharsTyped > 0) {
    const lastTypedChar = typed[totalCharsTyped - 1];
    const lastQuoteChar = currentQuote[totalCharsTyped - 1];

    if (lastTypedChar !== lastQuoteChar) {
      totalErrors++;
    }
  }

  correctChars = 0;
  let quoteHTML = "";

  for (let i = 0; i < currentQuote.length; i++) {
    let char = currentQuote[i];
    let typedChar = typed[i];

    if (i < typed.length) {
      if (typedChar === char) {
        quoteHTML += `<span style="color: green;">${char}</span>`;
        correctChars++;
      } else {
        quoteHTML += `<span style="color: red; text-decoration: underline;">${char}</span>`;
      }
    } else {
      quoteHTML += `<span>${char}</span>`;
    }
  }

  quoteEl.innerHTML = quoteHTML;

  const acc =
    totalCharsTyped > 0
      ? Math.round((correctChars / totalCharsTyped) * 100)
      : 100;
  accEl.textContent = acc;

  if (typed.length > currentQuote.length) {
    inputEl.value = typed.substring(0, currentQuote.length);
  }

  if (typed.length === currentQuote.length) {
    endTest();
  }
}

function endTest() {
  if (!isRunning) return;
  isRunning = false;

  clearInterval(timer);
  inputEl.disabled = true;

  const timeLimit = parseInt(timerSelect.value);
  const timeTaken = timeLimit - timeLeft;

  // WPM and Metrics Calculation
  const grossWpm = Math.round(totalCharsTyped / 5 / (timeTaken / 60));
  const netWpm = Math.round(correctChars / 5 / (timeTaken / 60));
  const finalAccCalc =
    totalCharsTyped > 0
      ? Math.round((correctChars / totalCharsTyped) * 100)
      : 0;

  // Display Final Results
  finalWpm.textContent = netWpm;
  finalGrossWpm.textContent = grossWpm;
  finalAcc.textContent = finalAccCalc;
  finalErrors.textContent = totalErrors;
  resultPopup.classList.remove("hidden");

  // Save Data
  saveToHistory(netWpm, grossWpm, finalAccCalc, totalErrors, timeTaken);
  saveToLeaderboard(netWpm, finalAccCalc);

  showLeaderboard();
  showHistory();
  drawChart();
}

// --- Leaderboard and History Functions ---

function saveToHistory(netWpm, grossWpm, acc, errors, duration) {
  const now = new Date();
  const formattedDate = `${now.toLocaleDateString()} ${now.toLocaleTimeString()}`;
  const source = difficultySelect.value.toUpperCase();

  const entry = {
    netWpm,
    grossWpm,
    acc: parseInt(acc),
    errors,
    duration,
    date: formattedDate,
    source,
  };

  let history = JSON.parse(localStorage.getItem("typingHistory") || "[]");
  history.unshift(entry);
  localStorage.setItem("typingHistory", JSON.stringify(history));
}

function showHistory() {
  const history = JSON.parse(localStorage.getItem("typingHistory") || "[]");

  if (history.length === 0) {
    historyListEl.innerHTML = "";
    document.getElementById("no-history-msg").classList.remove("hidden");
    // Hide the clear button if there is no history
    clearHistoryBtn.classList.add("hidden");
    return;
  }

  document.getElementById("no-history-msg").classList.add("hidden");
  clearHistoryBtn.classList.remove("hidden"); // Show the clear button

  historyListEl.innerHTML = history
    .map(
      (h) =>
        `<li>
            <span>⚡ <b>${h.netWpm}</b> Net WPM (${h.grossWpm} Gross)</span>
            <span>🎯 Acc: ${h.acc}% | Errors: ${h.errors}</span>
            <span class="time-stat">Source: ${h.source} | ${h.duration}s | ${h.date}</span>
        </li>`
    )
    .join("");
}

function clearHistory() {
  // 🗑 UPDATED FUNCTION: Clears history and prompts for leaderboard clear
  const confirmationHistory = confirm(
    "Are you sure you want to permanently delete your entire **Test History**? This action cannot be undone."
  );

  if (confirmationHistory) {
    localStorage.removeItem("typingHistory");

    // --- NEW LEADERBOARD CLEAR LOGIC ---
    const confirmationLeaderboard = confirm(
      "Would you also like to clear the **Leaderboard** scores? (These are separate from your personal history.)"
    );

    if (confirmationLeaderboard) {
      localStorage.removeItem("leaderboard");
      showLeaderboard(); // Refresh the leaderboard display
      alert("Test history and Leaderboard have been cleared.");
    } else {
      alert("Test history has been cleared.");
    }

    showHistory(); // Refresh the history display
  }
}

function saveToLeaderboard(wpm, acc) {
  const name = prompt("Enter your name for the Leaderboard (Max 15 chars):");
  if (!name) return;

  const entry = { name: name.substring(0, 15), wpm, acc: parseInt(acc) };
  let board = JSON.parse(localStorage.getItem("leaderboard") || "[]");

  board.push(entry);
  board.sort((a, b) => b.wpm - a.wpm);
  board = board.slice(0, 5);

  localStorage.setItem("leaderboard", JSON.stringify(board));
}

function showLeaderboard() {
  const board = JSON.parse(localStorage.getItem("leaderboard") || "[]");
  boardEl.innerHTML = board
    .map(
      (b, i) =>
        `<li>${["🥇", "🥈", "🥉"][i] || "⭐"} <span><b>${
          b.name
        }</b></span> <span>${b.wpm} WPM (${b.acc}%)</span></li>`
    )
    .join("");
}

function drawChart() {
  const ctx = document.getElementById("wpmChart");

  if (chartInstance) {
    chartInstance.destroy();
  }

  chartInstance = new Chart(ctx, {
    type: "line",
    data: {
      labels: wpmData.map((_, i) => i + 1),
      datasets: [
        {
          label: "Net WPM over time (per second)",
          data: wpmData,
          borderColor: document.body.classList.contains("dark")
            ? "#ff80c0"
            : "#ff66b2",
          backgroundColor: "transparent",
          tension: 0.3,
          pointBackgroundColor: document.body.classList.contains("dark")
            ? "#ff80c0"
            : "#ff66b2",
        },
      ],
    },
    options: {
      responsive: true,
      scales: {
        y: {
          beginAtZero: true,
          title: { display: true, text: "Net WPM" },
        },
        x: {
          title: { display: true, text: "Time (s)" },
        },
      },
      plugins: {
        legend: {
          display: false,
        },
      },
    },
  });
}
