// Trendy Product Store - fixed script.js (offcanvas wishlist + local images)
// 12 local SVG product placeholders used from /assets/

const PRODUCTS = [
  {
    id: "p1",
    title: "Everyday T-Shirt",
    category: "Clothing",
    price: 799,
    rating: 4.3,
    img: "assets/p1.svg",
    desc: "Soft cotton tee. Perfect for daily wear.",
  },
  {
    id: "p2",
    title: "Classic Leather Belt",
    category: "Accessories",
    price: 1299,
    rating: 4.6,
    img: "assets/p2.svg",
    desc: "Genuine leather belt with metal buckle.",
  },
  {
    id: "p3",
    title: "Minimal Desk Lamp",
    category: "Home",
    price: 2599,
    rating: 4.5,
    img: "assets/p3.svg",
    desc: "Adjustable LED lamp with warm light.",
  },
  {
    id: "p4",
    title: "Wireless Earbuds",
    category: "Gadgets",
    price: 3499,
    rating: 4.2,
    img: "assets/p4.svg",
    desc: "Noise-isolating earbuds with long battery life.",
  },
  {
    id: "p5",
    title: "Casual Hoodie",
    category: "Clothing",
    price: 1999,
    rating: 4.4,
    img: "assets/p5.svg",
    desc: "Cozy hoodie for chilly nights.",
  },
  {
    id: "p6",
    title: "Sleek Water Bottle",
    category: "Accessories",
    price: 699,
    rating: 4.1,
    img: "assets/p6.svg",
    desc: "Insulated stainless steel bottle.",
  },
  {
    id: "p7",
    title: "Decorative Cushion",
    category: "Home",
    price: 899,
    rating: 4.0,
    img: "assets/p7.svg",
    desc: "Soft cushion for your sofa.",
  },
  {
    id: "p8",
    title: "Smart Watch",
    category: "Gadgets",
    price: 7999,
    rating: 4.7,
    img: "assets/p8.svg",
    desc: "Track fitness & notifications in style.",
  },
  {
    id: "p9",
    title: "Summer Shorts",
    category: "Clothing",
    price: 649,
    rating: 4.0,
    img: "assets/p9.svg",
    desc: "Lightweight breathable shorts.",
  },
  {
    id: "p10",
    title: "Fashion Sunglasses",
    category: "Accessories",
    price: 1199,
    rating: 4.3,
    img: "assets/p10.svg",
    desc: "UV protection with style.",
  },
  {
    id: "p11",
    title: "Aroma Diffuser",
    category: "Home",
    price: 1499,
    rating: 4.2,
    img: "assets/p11.svg",
    desc: "Create a relaxing atmosphere.",
  },
  {
    id: "p12",
    title: "Portable Charger",
    category: "Gadgets",
    price: 2199,
    rating: 4.4,
    img: "assets/p12.svg",
    desc: "Fast charging power bank.",
  },
];

let state = {
  products: PRODUCTS.slice(),
  cart: JSON.parse(localStorage.getItem("cart")) || {},
  wishlist: JSON.parse(localStorage.getItem("wishlist")) || {},
};

// DOM refs
const productsGrid = document.getElementById("productsGrid");
const categorySelect = document.getElementById("categorySelect");
const priceRange = document.getElementById("priceRange");
const priceValue = document.getElementById("priceValue");
const ratingSelect = document.getElementById("ratingSelect");
const sortSelect = document.getElementById("sortSelect");
const clearFilters = document.getElementById("clearFilters");
const searchInput = document.getElementById("searchInput");
const resultsCount = document.getElementById("resultsCount");
const cartCount = document.getElementById("cartCount");
const wishCount = document.getElementById("wishCount");
const cartItems = document.getElementById("cartItems");
const cartTotal = document.getElementById("cartTotal");
const wishlistOffcanvasEl = document.getElementById("wishlistOffcanvas");

// init
priceValue.textContent = "₹" + priceRange.value;
renderProducts();
updateCartUI();
updateWishUI();

// events
categorySelect.addEventListener("change", renderProducts);
priceRange.addEventListener("input", () => {
  priceValue.textContent = "₹" + priceRange.value;
  renderProducts();
});
ratingSelect.addEventListener("change", renderProducts);
sortSelect.addEventListener("change", renderProducts);
clearFilters.addEventListener("click", () => {
  categorySelect.value = "";
  priceRange.value = 5000;
  priceValue.textContent = "₹5000";
  ratingSelect.value = "0";
  sortSelect.value = "featured";
  searchInput.value = "";
  renderProducts();
});
searchInput.addEventListener("input", renderProducts);

// when wishlist offcanvas opens, render items (ensures up-to-date)
if (wishlistOffcanvasEl) {
  wishlistOffcanvasEl.addEventListener("show.bs.offcanvas", function () {
    renderWishlistItems();
  });
}

// functions
function renderProducts() {
  const q = searchInput.value.trim().toLowerCase();
  const cat = categorySelect.value;
  const maxPrice = Number(priceRange.value);
  const minRating = Number(ratingSelect.value);
  const sortBy = sortSelect.value;

  let list = state.products.filter(
    (p) => p.price <= maxPrice && p.rating >= minRating
  );
  if (cat) list = list.filter((p) => p.category === cat);
  if (q)
    list = list.filter((p) => (p.title + p.desc).toLowerCase().includes(q));

  if (sortBy === "price-asc") list.sort((a, b) => a.price - b.price);
  else if (sortBy === "price-desc") list.sort((a, b) => b.price - a.price);
  else if (sortBy === "rating-desc") list.sort((a, b) => b.rating - a.rating);

  resultsCount.textContent = list.length;
  productsGrid.innerHTML =
    list.map((p) => productCardHtml(p)).join("") ||
    '<div class="col-12 text-center text-muted">No products found.</div>';
}

function productCardHtml(p) {
  const liked = !!state.wishlist[p.id];
  return `
    <div class="col-sm-6 col-md-4 col-lg-3 mb-4">
      <div class="card h-100 position-relative">
        <button class="favorite-btn ${
          liked ? "active" : ""
        }" onclick="toggleWishlist('${p.id}', true)">
          ${liked ? "❤️" : "🤍"}
        </button>
        <img src="${p.img}" alt="${escapeHtml(
    p.title
  )}" class="product-img rounded-top">
        <div class="card-body d-flex flex-column">
          <h6 class="card-title mb-1">${escapeHtml(p.title)}</h6>
          <div class="d-flex justify-content-between align-items-center mb-2">
            <span class="badge bg-warning text-dark">${p.rating} ★</span>
            <span class="text-success fw-bold">₹${p.price}</span>
          </div>
          <p class="small text-muted mb-3">${escapeHtml(p.desc)}</p>
          <div class="mt-auto d-flex gap-2">
            <button class="btn btn-outline-secondary btn-sm w-50" onclick="quickView('${
              p.id
            }')">Quick View</button>
            <button class="btn btn-primary btn-sm w-50" onclick="addToCart('${
              p.id
            }')">Add to Cart</button>
          </div>
        </div>
      </div>
    </div>
  `;
}

function quickView(id) {
  const p = state.products.find((x) => x.id === id);
  document.getElementById("quickViewBody").innerHTML = `
    <div class="d-flex gap-3 flex-column flex-lg-row">
      <img src="${
        p.img
      }" class="img-fluid rounded" style="max-width:320px;object-fit:cover;">
      <div>
        <h4 class="fw-bold">${escapeHtml(p.title)}</h4>
        <div class="mb-2"><strong class="text-success">₹${
          p.price
        }</strong> · <span class="badge bg-warning text-dark">${
    p.rating
  } ★</span></div>
        <p class="text-muted">${escapeHtml(p.desc)}</p>
        <div class="d-flex gap-2">
          <button class="btn btn-primary" onclick="addToCart('${
            p.id
          }');closeQuickView();">Add to Cart</button>
          <button class="btn btn-outline-danger" onclick="toggleWishlist('${
            p.id
          }', true);closeQuickView();">Wishlist</button>
        </div>
      </div>
    </div>
  `;
  new bootstrap.Modal(document.getElementById("quickViewModal")).show();
}
function closeQuickView() {
  const modalClose = document.querySelector("#quickViewModal .btn-close-modal");
  if (modalClose) modalClose.click();
}

// cart functions
function addToCart(id) {
  if (!state.cart[id])
    state.cart[id] = { ...state.products.find((p) => p.id === id), qty: 1 };
  else state.cart[id].qty += 1;
  localStorage.setItem("cart", JSON.stringify(state.cart));
  updateCartUI();
  showToast("Added to cart");
}

function removeFromCart(id) {
  if (state.cart[id]) delete state.cart[id];
  localStorage.setItem("cart", JSON.stringify(state.cart));
  updateCartUI();
}

function changeQty(id, delta) {
  if (!state.cart[id]) return;
  state.cart[id].qty = Math.max(1, state.cart[id].qty + delta);
  localStorage.setItem("cart", JSON.stringify(state.cart));
  updateCartUI();
}

function updateCartUI() {
  const totalItems = Object.values(state.cart).reduce((s, i) => s + i.qty, 0);
  const totalPrice = Object.values(state.cart).reduce(
    (s, i) => s + i.qty * i.price,
    0
  );
  cartCount.textContent = totalItems;
  cartTotal.textContent = "₹" + totalPrice;
  renderCartItems();
}

function renderCartItems() {
  const items = Object.values(state.cart);
  if (items.length === 0) {
    cartItems.innerHTML = '<p class="text-muted">Your cart is empty.</p>';
    return;
  }
  cartItems.innerHTML = items
    .map(
      (it) => `
    <div class="d-flex align-items-center justify-content-between mb-3">
      <div class="d-flex gap-3 align-items-center">
        <img src="${
          it.img
        }" style="width:64px;height:64px;object-fit:cover;border-radius:8px;">
        <div>
          <div class="fw-bold">${escapeHtml(it.title)}</div>
          <div class="text-muted small">₹${it.price} · Qty: ${it.qty}</div>
        </div>
      </div>
      <div class="d-flex gap-2 align-items-center">
        <button class="btn btn-sm btn-outline-secondary" onclick="changeQty('${
          it.id
        }',-1)">-</button>
        <button class="btn btn-sm btn-outline-secondary" onclick="changeQty('${
          it.id
        }',1)">+</button>
        <button class="btn btn-sm btn-danger" onclick="removeFromCart('${
          it.id
        }')">Remove</button>
      </div>
    </div>
  `
    )
    .join("");
  cartTotal.textContent =
    "₹" + Object.values(state.cart).reduce((s, i) => s + i.qty * i.price, 0);
}

// wishlist functions
function toggleWishlist(id, skipRenderOffcanvas) {
  if (state.wishlist[id]) {
    delete state.wishlist[id];
    showToast("Removed from wishlist");
  } else {
    state.wishlist[id] = { ...state.products.find((p) => p.id === id) };
    showToast("Added to wishlist");
  }
  localStorage.setItem("wishlist", JSON.stringify(state.wishlist));
  updateWishUI();
  renderProducts();
  // if wishlist offcanvas is open, re-render items
  try {
    const off = bootstrap.Offcanvas.getInstance(
      document.getElementById("wishlistOffcanvas")
    );
    if (off) renderWishlistItems();
  } catch (e) {
    /* ignore */
  }
}

function updateWishUI() {
  const count = Object.keys(state.wishlist).length;
  wishCount.textContent = count;
}

// render wishlist offcanvas content
function renderWishlistItems() {
  const items = Object.values(state.wishlist);
  const container = document.getElementById("wishlistItems");
  if (!container) return;
  if (items.length === 0) {
    container.innerHTML =
      '<p class="text-muted text-center">Your wishlist is empty.</p>';
    return;
  }
  container.innerHTML = items
    .map(
      (it) => `
    <div class="d-flex align-items-center justify-content-between mb-3 border-bottom pb-2">
      <div class="d-flex gap-3 align-items-center">
        <img src="${
          it.img
        }" style="width:70px;height:70px;object-fit:cover;border-radius:8px;">
        <div>
          <div class="fw-bold">${escapeHtml(it.title)}</div>
          <div class="text-muted small">₹${it.price}</div>
        </div>
      </div>
      <div class="d-flex gap-2 align-items-center">
        <button class="btn btn-sm btn-outline-primary" onclick="addToCartFromWishlist('${
          it.id
        }')">Add to Cart</button>
        <button class="btn btn-sm btn-danger" onclick="removeFromWishlist('${
          it.id
        }')">Remove</button>
      </div>
    </div>
  `
    )
    .join("");
}

// add to cart from wishlist
function addToCartFromWishlist(id) {
  addToCart(id);
  renderWishlistItems();
}

// remove from wishlist
function removeFromWishlist(id) {
  if (state.wishlist[id]) delete state.wishlist[id];
  localStorage.setItem("wishlist", JSON.stringify(state.wishlist));
  updateWishUI();
  renderWishlistItems();
  renderProducts();
}

// utilities
function escapeHtml(str) {
  if (!str) return "";
  return String(str).replace(
    /[&<>"']/g,
    (s) =>
      ({ "&": "&amp;", "<": "&lt;", ">": "&gt;", '"': "&quot;", "'": "&#039;" }[
        s
      ])
  );
}

function showToast(msg) {
  const el = document.createElement("div");
  el.className =
    "toast align-items-center text-bg-primary border-0 position-fixed";
  el.style.top = "20px";
  el.style.right = "20px";
  el.style.zIndex = 1200;
  el.innerHTML = `<div class="d-flex"><div class="toast-body">${msg}</div><button class="btn-close btn-close-white me-2 m-auto" onclick="this.parentNode.parentNode.remove()"></button></div>`;
  document.body.appendChild(el);
  setTimeout(() => el.remove(), 2000);
}

// checkout demo
document.getElementById("checkoutBtn").addEventListener("click", () => {
  if (Object.keys(state.cart).length === 0) {
    alert("Your cart is empty");
    return;
  }
  alert("Demo checkout — thank you for shopping!");
  state.cart = {};
  localStorage.removeItem("cart");
  updateCartUI();
});

// init
updateCartUI();
updateWishUI();
renderProducts();
