let tasks = [];
let currentFilter = "all";

function loadTasks() {
  const stored = localStorage.getItem("taskflowTasks");
  if (stored) {
    tasks = JSON.parse(stored);
    renderTasks();
    updateStats();
  }
}

function saveTasks() {
  localStorage.setItem("taskflowTasks", JSON.stringify(tasks));
}

function addTask() {
  const title = document.getElementById("taskTitle").value.trim();
  const description = document.getElementById("taskDescription").value.trim();
  const priority = document.getElementById("taskPriority").value;
  const date = document.getElementById("taskDate").value;

  if (!title) {
    alert("Please enter a task title!");
    return;
  }

  const task = {
    id: Date.now(),
    title,
    description,
    priority,
    date: date || new Date().toISOString().split("T")[0],
    completed: false,
    createdAt: new Date().toLocaleString(),
  };

  tasks.unshift(task);
  saveTasks();
  renderTasks();
  updateStats();

  document.getElementById("taskTitle").value = "";
  document.getElementById("taskDescription").value = "";
  document.getElementById("taskDate").value = "";
}

function toggleComplete(id) {
  const task = tasks.find((t) => t.id === id);
  if (task) {
    task.completed = !task.completed;
    saveTasks();
    renderTasks();
    updateStats();
  }
}

function deleteTask(id) {
  if (confirm("Are you sure you want to delete this task?")) {
    tasks = tasks.filter((t) => t.id !== id);
    saveTasks();
    renderTasks();
    updateStats();
  }
}

function filterTasks(filter) {
  currentFilter = filter;
  document.querySelectorAll(".filter-btn").forEach((btn) => {
    btn.classList.remove("active");
  });
  event.target.closest(".filter-btn").classList.add("active");
  renderTasks();
}

function renderTasks() {
  const container = document.getElementById("tasksContainer");
  let filteredTasks = tasks;

  if (currentFilter === "pending") {
    filteredTasks = tasks.filter((t) => !t.completed);
  } else if (currentFilter === "completed") {
    filteredTasks = tasks.filter((t) => t.completed);
  } else if (currentFilter === "high") {
    filteredTasks = tasks.filter((t) => t.priority === "high");
  }

  if (filteredTasks.length === 0) {
    container.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-inbox"></i>
                        <h4>No tasks found</h4>
                        <p>${
                          currentFilter === "all"
                            ? "Create your first task to get started!"
                            : "No tasks match this filter."
                        }</p>
                    </div>
                `;
    return;
  }

  container.innerHTML = filteredTasks
    .map(
      (task) => `
                <div class="task-card ${task.completed ? "completed" : ""}">
                    <div class="task-header">
                        <h5 class="task-title">${task.title}</h5>
                        <span class="task-priority priority-${task.priority}">
                            ${task.priority.toUpperCase()}
                        </span>
                    </div>
                    ${
                      task.description
                        ? `<p class="task-description">${task.description}</p>`
                        : ""
                    }
                    <div class="task-meta">
                        <span><i class="fas fa-calendar"></i> ${
                          task.date
                        }</span>
                        <span><i class="fas fa-clock"></i> ${
                          task.createdAt
                        }</span>
                    </div>
                    <div class="task-actions">
                        <button class="action-btn btn-complete" onclick="toggleComplete(${
                          task.id
                        })">
                            <i class="fas ${
                              task.completed ? "fa-undo" : "fa-check"
                            }"></i>
                            ${task.completed ? "Undo" : "Complete"}
                        </button>
                        <button class="action-btn btn-delete" onclick="deleteTask(${
                          task.id
                        })">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </div>
                </div>
            `
    )
    .join("");
}

function updateStats() {
  const total = tasks.length;
  const completed = tasks.filter((t) => t.completed).length;
  const pending = total - completed;
  const progress = total > 0 ? Math.round((completed / total) * 100) : 0;

  document.getElementById("totalTasks").textContent = total;
  document.getElementById("completedTasks").textContent = completed;
  document.getElementById("pendingTasks").textContent = pending;
  document.getElementById("progressPercent").textContent = progress + "%";
  document.getElementById("progressBar").style.width = progress + "%";
}

document.getElementById("taskTitle").addEventListener("keypress", function (e) {
  if (e.key === "Enter") {
    addTask();
  }
});

// Set minimum date to today
const today = new Date().toISOString().split("T")[0];
document.getElementById("taskDate").setAttribute("min", today);

loadTasks();
