import React, { useState, useEffect, useRef } from "react";

const ROWS = 20;
const COLS = 20;
const INITIAL_SNAKE = [
  { r: 10, c: 9 },
  { r: 10, c: 10 },
  { r: 10, c: 11 },
];
const INITIAL_DIRECTION = { r: 0, c: -1 }; // moving left
const SPEED = 150; // ms per tick

function randomFood(snake) {
  while (true) {
    const r = Math.floor(Math.random() * ROWS);
    const c = Math.floor(Math.random() * COLS);
    const collision = snake.some((s) => s.r === r && s.c === c);
    if (!collision) return { r, c };
  }
}

export default function SnakeGame() {
  const [snake, setSnake] = useState(INITIAL_SNAKE);
  const [dir, setDir] = useState(INITIAL_DIRECTION);
  const [food, setFood] = useState(() => randomFood(INITIAL_SNAKE));
  const [running, setRunning] = useState(false);
  const [score, setScore] = useState(0);
  const [high, setHigh] = useState(() =>
    Number(localStorage.getItem("snake-high") || 0)
  );
  const tickRef = useRef(null);
  const dirRef = useRef(dir);
  dirRef.current = dir;
  const snakeRef = useRef(snake);
  snakeRef.current = snake;

  useEffect(() => {
    function onKey(e) {
      const k = e.key;
      const d = dirRef.current;
      if (k === "ArrowUp" || k === "w") {
        if (d.r === 1) return;
        setDir({ r: -1, c: 0 });
      } else if (k === "ArrowDown" || k === "s") {
        if (d.r === -1) return;
        setDir({ r: 1, c: 0 });
      } else if (k === "ArrowLeft" || k === "a") {
        if (d.c === 1) return;
        setDir({ r: 0, c: -1 });
      } else if (k === "ArrowRight" || k === "d") {
        if (d.c === -1) return;
        setDir({ r: 0, c: 1 });
      } else if (k === " ") {
        setRunning((r) => !r);
      }
    }
    window.addEventListener("keydown", onKey);
    return () => window.removeEventListener("keydown", onKey);
  }, []);

  useEffect(() => {
    if (running) {
      tickRef.current = setInterval(() => {
        setSnake((prev) => {
          const head = prev[0];
          const newHead = {
            r: (head.r + dirRef.current.r + ROWS) % ROWS,
            c: (head.c + dirRef.current.c + COLS) % COLS,
          };
          // check self collision
          const hitSelf = prev.some(
            (seg) => seg.r === newHead.r && seg.c === newHead.c
          );
          if (hitSelf) {
            // game over
            clearInterval(tickRef.current);
            setRunning(false);
            setHigh((h) => {
              const newHigh = Math.max(h, score);
              localStorage.setItem("snake-high", newHigh);
              return newHigh;
            });
            return prev;
          }
          let grew = false;
          if (newHead.r === food.r && newHead.c === food.c) {
            grew = true;
            setScore((s) => s + 1);
            setFood((snakeNow) => randomFood([newHead, ...prev]));
          }
          const newSnake = [newHead, ...prev];
          if (!grew) newSnake.pop();
          return newSnake;
        });
      }, SPEED);
    } else {
      if (tickRef.current) clearInterval(tickRef.current);
    }
    return () => {
      if (tickRef.current) clearInterval(tickRef.current);
    };
  }, [running, food, score]);

  function start() {
    setSnake(INITIAL_SNAKE);
    setDir(INITIAL_DIRECTION);
    setFood(randomFood(INITIAL_SNAKE));
    setScore(0);
    setRunning(true);
  }
  function stop() {
    setRunning(false);
  }
  function reset() {
    setSnake(INITIAL_SNAKE);
    setDir(INITIAL_DIRECTION);
    setFood(randomFood(INITIAL_SNAKE));
    setScore(0);
    setRunning(false);
  }

  return (
    <div className="mx-auto bg-gray-800 p-4 rounded-lg shadow-lg">
      <div className="flex items-center justify-between mb-3">
        <div className="space-x-3">
          <button
            onClick={start}
            className="px-3 py-1 bg-green-600 rounded hover:bg-green-500"
          >
            Start
          </button>
          <button
            onClick={() => setRunning((r) => !r)}
            className="px-3 py-1 bg-yellow-600 rounded hover:bg-yellow-500"
          >
            {running ? "Pause" : "Resume"}
          </button>
          <button
            onClick={reset}
            className="px-3 py-1 bg-red-600 rounded hover:bg-red-500"
          >
            Reset
          </button>
        </div>
        <div className="text-sm">
          <div>
            Score: <span className="font-bold">{score}</span>
          </div>
          <div>
            High: <span className="font-bold">{high}</span>
          </div>
        </div>
      </div>

      <div className="flex gap-4">
        <div className="bg-black/80 p-2 rounded">
          <div
            className="grid"
            style={{
              gridTemplateColumns: `repeat(${COLS}, 20px)`,
              gridTemplateRows: `repeat(${ROWS}, 20px)`,
            }}
          >
            {Array.from({ length: ROWS }).map((_, r) =>
              Array.from({ length: COLS }).map((_, c) => {
                const isHead = snake[0].r === r && snake[0].c === c;
                const isBody = snake.some(
                  (seg, idx) => idx > 0 && seg.r === r && seg.c === c
                );
                const isFood = food.r === r && food.c === c;
                const cellKey = `${r}-${c}`;
                const base = "w-[20px] h-[20px] border border-gray-700";
                const cls = isHead
                  ? base + " bg-green-400"
                  : isBody
                  ? base + " bg-green-600"
                  : isFood
                  ? base + " bg-red-500"
                  : base + " bg-transparent";
                return <div key={cellKey} className={cls} />;
              })
            )}
          </div>
        </div>

        <div className="text-sm w-64">
          <div className="mb-3">
            <h2 className="font-semibold">Controls</h2>
            <ul className="list-disc ml-5">
              <li>Arrow keys or WASD to move</li>
              <li>Space to pause/resume</li>
              <li>Start / Pause / Reset buttons</li>
            </ul>
          </div>

          <div>
            <h3 className="font-semibold mb-1">Tips</h3>
            <p className="text-xs">
              Snake wraps around the edges. Try to trap yourself carefully —
              colliding with your own body ends the game.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
