import React, { useState, useEffect } from "react";
import {
  Trash2,
  Plus,
  FileText,
  RotateCcw,
  Receipt,
  ShoppingCart,
  Calculator,
  Edit2,
  Check,
  X,
} from "lucide-react";

export default function RestaurantBillGenerator() {
  const [items, setItems] = useState([]);
  const [name, setName] = useState("");
  const [price, setPrice] = useState("");
  const [qty, setQty] = useState(1);
  const [category, setCategory] = useState("Food");
  const [gst, setGst] = useState(5);
  const [discount, setDiscount] = useState(0);
  const [customerName, setCustomerName] = useState("");
  const [tableNumber, setTableNumber] = useState("");
  const [paymentMethod, setPaymentMethod] = useState("Cash");
  const [restaurantName, setRestaurantName] = useState("The Grand Restaurant");
  const [editingId, setEditingId] = useState(null);
  const [editQty, setEditQty] = useState(1);

  const categories = ["Food", "Beverages", "Desserts", "Starters"];

  const addItem = () => {
    if (!name || !price) {
      alert("Please enter item name and price!");
      return;
    }
    const newItem = {
      id: Date.now(),
      name,
      price: Number(price),
      qty: Number(qty),
      category,
      total: Number(price) * Number(qty),
    };
    setItems([...items, newItem]);
    setName("");
    setPrice("");
    setQty(1);
  };

  const removeItem = (id) => {
    setItems(items.filter((item) => item.id !== id));
  };

  const startEdit = (item) => {
    setEditingId(item.id);
    setEditQty(item.qty);
  };

  const saveEdit = (id) => {
    setItems(
      items.map((item) => {
        if (item.id === id) {
          return {
            ...item,
            qty: Number(editQty),
            total: item.price * Number(editQty),
          };
        }
        return item;
      })
    );
    setEditingId(null);
  };

  const cancelEdit = () => {
    setEditingId(null);
    setEditQty(1);
  };

  const subtotal = items.reduce((acc, it) => acc + it.total, 0);
  const gstAmount = (subtotal * gst) / 100;
  const finalTotal = subtotal + gstAmount - discount;

  const generateBill = () => {
    if (items.length === 0) {
      alert("Please add items to generate bill!");
      return;
    }

    const date = new Date();
    const billNumber = `BILL-${Date.now()}`;

    const rows = items
      .map(
        (it, idx) => `
        <tr>
          <td style="padding:8px;text-align:left;border-bottom:1px dashed #ddd">${
            idx + 1
          }</td>
          <td style="padding:8px;text-align:left;border-bottom:1px dashed #ddd">${
            it.name
          }<br/><small style="color:#666">${it.category}</small></td>
          <td style="padding:8px;text-align:center;border-bottom:1px dashed #ddd">₹${it.price.toFixed(
            2
          )}</td>
          <td style="padding:8px;text-align:center;border-bottom:1px dashed #ddd">${
            it.qty
          }</td>
          <td style="padding:8px;text-align:right;border-bottom:1px dashed #ddd;font-weight:600">₹${it.total.toFixed(
            2
          )}</td>
        </tr>
      `
      )
      .join("");

    const html = `
      <html>
        <head>
          <title>Bill - ${restaurantName}</title>
          <style>
            @media print {
              body { margin: 0; }
              .no-print { display: none; }
            }
            body { 
              font-family: 'Courier New', monospace; 
              padding: 20px;
              max-width: 800px;
              margin: 0 auto;
              background: #f5f5f5;
            }
            .bill-container {
              background: white;
              padding: 30px;
              border-radius: 10px;
              box-shadow: 0 0 20px rgba(0,0,0,0.1);
            }
            .header { 
              text-align: center; 
              margin-bottom: 20px;
              border-bottom: 2px solid #333;
              padding-bottom: 15px;
            }
            .header h1 { 
              margin: 0; 
              font-size: 28px;
              color: #333;
              text-transform: uppercase;
              letter-spacing: 2px;
            }
            .header-info {
              margin-top: 10px;
              font-size: 12px;
              color: #666;
            }
            .bill-info {
              display: flex;
              justify-content: space-between;
              margin: 15px 0;
              font-size: 13px;
            }
            table { 
              width: 100%; 
              border-collapse: collapse; 
              margin: 20px 0;
            }
            th {
              background: #333;
              color: white;
              padding: 10px;
              text-align: left;
              font-weight: 600;
            }
            td { padding: 8px; }
            .totals { 
              margin-top: 20px;
              border-top: 2px solid #333;
              padding-top: 15px;
            }
            .totals-row {
              display: flex;
              justify-content: space-between;
              padding: 5px 0;
              font-size: 14px;
            }
            .final-total {
              display: flex;
              justify-content: space-between;
              padding: 12px;
              background: #f0f0f0;
              margin-top: 10px;
              border-radius: 5px;
              font-size: 18px;
              font-weight: 700;
            }
            .footer {
              margin-top: 30px;
              text-align: center;
              font-size: 12px;
              border-top: 1px dashed #ccc;
              padding-top: 15px;
            }
            .print-btn {
              background: #4CAF50;
              color: white;
              border: none;
              padding: 12px 30px;
              font-size: 16px;
              border-radius: 5px;
              cursor: pointer;
              margin: 20px auto;
              display: block;
            }
            .print-btn:hover {
              background: #45a049;
            }
          </style>
        </head>
        <body>
          <div class="bill-container">
            <div class="header">
              <h1>🍽 ${restaurantName}</h1>
              <div class="header-info">
                <div>123 Food Street, Culinary City, FC 12345</div>
                <div>Phone: +91 98765 43210 | Email: info@restaurant.com</div>
                <div>GSTIN: 29XXXXX1234X1ZX</div>
              </div>
            </div>

            <div class="bill-info">
              <div>
                <strong>Bill No:</strong> ${billNumber}<br/>
                <strong>Date:</strong> ${date.toLocaleDateString()}<br/>
                <strong>Time:</strong> ${date.toLocaleTimeString()}
              </div>
              <div style="text-align: right">
                ${
                  customerName
                    ? `<strong>Customer:</strong> ${customerName}<br/>`
                    : ""
                }
                ${
                  tableNumber
                    ? `<strong>Table:</strong> ${tableNumber}<br/>`
                    : ""
                }
                <strong>Payment:</strong> ${paymentMethod}
              </div>
            </div>

            <table>
              <thead>
                <tr>
                  <th style="width:40px">#</th>
                  <th>Item</th>
                  <th style="text-align:center;width:100px">Price</th>
                  <th style="text-align:center;width:60px">Qty</th>
                  <th style="text-align:right;width:100px">Total</th>
                </tr>
              </thead>
              <tbody>
                ${rows}
              </tbody>
            </table>

            <div class="totals">
              <div class="totals-row">
                <span>Subtotal:</span>
                <span>₹${subtotal.toFixed(2)}</span>
              </div>
              <div class="totals-row">
                <span>GST (${gst}%):</span>
                <span>₹${gstAmount.toFixed(2)}</span>
              </div>
              ${
                discount > 0
                  ? `
                <div class="totals-row" style="color: #d32f2f">
                  <span>Discount:</span>
                  <span>- ₹${Number(discount).toFixed(2)}</span>
                </div>
              `
                  : ""
              }
              <div class="final-total">
                <span>GRAND TOTAL:</span>
                <span>₹${finalTotal.toFixed(2)}</span>
              </div>
            </div>

            <div class="footer">
              <p style="font-weight:600;font-size:14px">Thank you for dining with us! 🙏</p>
              <p>Please visit us again!</p>
              <p style="margin-top:10px;color:#666">This is a computer generated bill</p>
            </div>

            <button class="print-btn no-print" onclick="window.print()">🖨 Print Bill</button>
          </div>
        </body>
      </html>
    `;

    const w = window.open("", "_blank", "width=900,height=700");
    if (w) {
      w.document.write(html);
      w.document.close();
    } else {
      alert("Pop-up blocked! Please allow pop-ups to generate bill.");
    }
  };

  const resetAll = () => {
    if (confirm("Are you sure you want to reset everything?")) {
      setItems([]);
      setDiscount(0);
      setGst(5);
      setCustomerName("");
      setTableNumber("");
      setPaymentMethod("Cash");
    }
  };

  const groupedItems = items.reduce((acc, item) => {
    if (!acc[item.category]) acc[item.category] = [];
    acc[item.category].push(item);
    return acc;
  }, {});

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 to-blue-50 py-8 px-4">
      <div className="max-w-6xl mx-auto">
        {/* Header */}
        <div className="bg-white rounded-2xl shadow-xl p-6 mb-6">
          <div className="flex items-center justify-center gap-3 mb-4">
            <Receipt className="w-10 h-10 text-purple-600" />
            <h1 className="text-4xl font-bold text-gray-800">
              Restaurant Bill Generator
            </h1>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <input
              className="px-4 py-2 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
              placeholder="Restaurant Name"
              value={restaurantName}
              onChange={(e) => setRestaurantName(e.target.value)}
            />
            <input
              className="px-4 py-2 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
              placeholder="Customer Name (Optional)"
              value={customerName}
              onChange={(e) => setCustomerName(e.target.value)}
            />
            <input
              className="px-4 py-2 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
              placeholder="Table Number (Optional)"
              value={tableNumber}
              onChange={(e) => setTableNumber(e.target.value)}
            />
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Left Panel - Add Items */}
          <div className="lg:col-span-2">
            <div className="bg-white rounded-2xl shadow-xl p-6">
              <div className="flex items-center gap-2 mb-4">
                <ShoppingCart className="w-6 h-6 text-purple-600" />
                <h2 className="text-2xl font-bold text-gray-800">Add Items</h2>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                <input
                  className="px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
                  placeholder="Item Name"
                  value={name}
                  onChange={(e) => setName(e.target.value)}
                  onKeyPress={(e) => e.key === "Enter" && addItem()}
                />
                <select
                  className="px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
                  value={category}
                  onChange={(e) => setCategory(e.target.value)}
                >
                  {categories.map((cat) => (
                    <option key={cat} value={cat}>
                      {cat}
                    </option>
                  ))}
                </select>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                <input
                  className="px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
                  placeholder="Price (₹)"
                  type="number"
                  value={price}
                  onChange={(e) => setPrice(e.target.value)}
                  onKeyPress={(e) => e.key === "Enter" && addItem()}
                />
                <input
                  className="px-4 py-3 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
                  placeholder="Quantity"
                  type="number"
                  min="1"
                  value={qty}
                  onChange={(e) => setQty(e.target.value)}
                  onKeyPress={(e) => e.key === "Enter" && addItem()}
                />
                <button
                  onClick={addItem}
                  className="bg-gradient-to-r from-purple-600 to-blue-600 text-white px-6 py-3 rounded-lg font-semibold hover:from-purple-700 hover:to-blue-700 transition-all flex items-center justify-center gap-2"
                >
                  <Plus className="w-5 h-5" /> Add Item
                </button>
              </div>

              {/* Items List */}
              <div className="mt-6">
                <h3 className="text-xl font-bold text-gray-800 mb-4">
                  Order Items ({items.length})
                </h3>

                {items.length === 0 ? (
                  <div className="text-center py-12 text-gray-400">
                    <ShoppingCart className="w-16 h-16 mx-auto mb-4 opacity-50" />
                    <p className="text-lg">No items added yet</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {Object.entries(groupedItems).map(([cat, catItems]) => (
                      <div key={cat} className="bg-gray-50 rounded-lg p-4">
                        <h4 className="font-bold text-purple-600 mb-3">
                          {cat}
                        </h4>
                        {catItems.map((item) => (
                          <div
                            key={item.id}
                            className="bg-white rounded-lg p-4 mb-2 flex items-center justify-between shadow-sm"
                          >
                            <div className="flex-1">
                              <h5 className="font-semibold text-gray-800">
                                {item.name}
                              </h5>
                              <p className="text-sm text-gray-600">
                                ₹{item.price.toFixed(2)} × {item.qty}
                              </p>
                            </div>

                            {editingId === item.id ? (
                              <div className="flex items-center gap-2">
                                <input
                                  type="number"
                                  min="1"
                                  value={editQty}
                                  onChange={(e) => setEditQty(e.target.value)}
                                  className="w-20 px-2 py-1 border-2 border-purple-500 rounded"
                                />
                                <button
                                  onClick={() => saveEdit(item.id)}
                                  className="text-green-600 hover:text-green-700"
                                >
                                  <Check className="w-5 h-5" />
                                </button>
                                <button
                                  onClick={cancelEdit}
                                  className="text-red-600 hover:text-red-700"
                                >
                                  <X className="w-5 h-5" />
                                </button>
                              </div>
                            ) : (
                              <div className="flex items-center gap-4">
                                <span className="font-bold text-lg text-purple-600">
                                  ₹{item.total.toFixed(2)}
                                </span>
                                <button
                                  onClick={() => startEdit(item)}
                                  className="text-blue-600 hover:text-blue-700"
                                >
                                  <Edit2 className="w-5 h-5" />
                                </button>
                                <button
                                  onClick={() => removeItem(item.id)}
                                  className="text-red-600 hover:text-red-700"
                                >
                                  <Trash2 className="w-5 h-5" />
                                </button>
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    ))}
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* Right Panel - Summary */}
          <div className="lg:col-span-1">
            <div className="bg-white rounded-2xl shadow-xl p-6 sticky top-6">
              <div className="flex items-center gap-2 mb-4">
                <Calculator className="w-6 h-6 text-purple-600" />
                <h2 className="text-2xl font-bold text-gray-800">
                  Bill Summary
                </h2>
              </div>

              {/* GST Selection */}
              <div className="mb-4">
                <label className="block font-semibold text-gray-700 mb-2">
                  GST %
                </label>
                <div className="flex gap-2">
                  {[5, 12, 18].map((g) => (
                    <button
                      key={g}
                      onClick={() => setGst(g)}
                      className={`flex-1 py-2 rounded-lg font-semibold transition-all ${
                        gst === g
                          ? "bg-purple-600 text-white"
                          : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                      }`}
                    >
                      {g}%
                    </button>
                  ))}
                </div>
              </div>

              {/* Discount */}
              <div className="mb-4">
                <label className="block font-semibold text-gray-700 mb-2">
                  Discount (₹)
                </label>
                <input
                  type="number"
                  placeholder="Enter discount amount"
                  className="w-full px-4 py-2 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
                  value={discount}
                  onChange={(e) => setDiscount(Number(e.target.value))}
                />
              </div>

              {/* Payment Method */}
              <div className="mb-6">
                <label className="block font-semibold text-gray-700 mb-2">
                  Payment Method
                </label>
                <select
                  className="w-full px-4 py-2 border-2 border-gray-300 rounded-lg focus:border-purple-500 focus:outline-none"
                  value={paymentMethod}
                  onChange={(e) => setPaymentMethod(e.target.value)}
                >
                  <option value="Cash">Cash</option>
                  <option value="Card">Card</option>
                  <option value="UPI">UPI</option>
                  <option value="Online">Online</option>
                </select>
              </div>

              {/* Calculations */}
              <div className="bg-gradient-to-br from-purple-50 to-blue-50 rounded-lg p-4 mb-6">
                <div className="flex justify-between py-2 border-b border-gray-300">
                  <span className="text-gray-700">Subtotal:</span>
                  <span className="font-semibold">₹{subtotal.toFixed(2)}</span>
                </div>
                <div className="flex justify-between py-2 border-b border-gray-300">
                  <span className="text-gray-700">GST ({gst}%):</span>
                  <span className="font-semibold">₹{gstAmount.toFixed(2)}</span>
                </div>
                {discount > 0 && (
                  <div className="flex justify-between py-2 border-b border-gray-300 text-red-600">
                    <span>Discount:</span>
                    <span className="font-semibold">
                      - ₹{discount.toFixed(2)}
                    </span>
                  </div>
                )}
                <div className="flex justify-between py-3 mt-2">
                  <span className="text-xl font-bold text-gray-800">
                    TOTAL:
                  </span>
                  <span className="text-2xl font-bold text-purple-600">
                    ₹{finalTotal.toFixed(2)}
                  </span>
                </div>
              </div>

              {/* Action Buttons */}
              <div className="space-y-3">
                <button
                  onClick={generateBill}
                  className="w-full bg-gradient-to-r from-green-600 to-green-700 text-white py-3 rounded-lg font-semibold hover:from-green-700 hover:to-green-800 transition-all flex items-center justify-center gap-2 shadow-lg"
                >
                  <FileText className="w-5 h-5" /> Generate Bill
                </button>
                <button
                  onClick={resetAll}
                  className="w-full bg-gradient-to-r from-red-600 to-red-700 text-white py-3 rounded-lg font-semibold hover:from-red-700 hover:to-red-800 transition-all flex items-center justify-center gap-2"
                >
                  <RotateCcw className="w-5 h-5" /> Reset All
                </button>
              </div>
            </div>
          </div>
        </div>

        {/* Footer */}
        <div className="text-center mt-8 text-gray-600">
          <p className="text-sm">
            Advanced Restaurant Bill Generator - Professional Billing Solution
          </p>
        </div>
      </div>
    </div>
  );
}
