import React, { useState } from "react";
import {
  Search,
  Clock,
  Users,
  Flame,
  ChefHat,
  Heart,
  BookOpen,
  Plus,
  X,
  Star,
  Filter,
} from "lucide-react";

export default function RecipeFinder() {
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState("all");
  const [favorites, setFavorites] = useState([]);
  const [selectedRecipe, setSelectedRecipe] = useState(null);
  const [mealPlan, setMealPlan] = useState([]);

  const categories = [
    "All",
    "Breakfast",
    "Lunch",
    "Dinner",
    "Dessert",
    "Snacks",
    "Vegetarian",
  ];

  const allRecipes = [
    // Existing Recipes (6)
    {
      id: 1,
      name: "Avocado Toast Supreme",
      category: "Breakfast",
      time: 10,
      servings: 2,
      calories: 320,
      difficulty: "Easy",
      rating: 4.8,
      image:
        "https://images.unsplash.com/photo-1541519227354-08fa5d50c44d?w=400",
      ingredients: [
        "2 slices whole grain bread",
        "1 ripe avocado",
        "2 eggs",
        "Cherry tomatoes",
        "Salt & pepper",
        "Red pepper flakes",
      ],
      instructions: [
        "Toast the bread until golden brown",
        "Mash avocado with salt and pepper",
        "Fry or poach eggs to your liking",
        "Spread avocado on toast, top with egg and tomatoes",
        "Sprinkle with red pepper flakes",
      ],
      tags: ["Healthy", "Quick", "Protein-rich"],
    },
    {
      id: 2,
      name: "Mediterranean Quinoa Bowl",
      category: "Lunch",
      time: 25,
      servings: 4,
      calories: 450,
      difficulty: "Medium",
      rating: 4.9,
      image: "https://images.unsplash.com/photo-1546069901-ba9599a7e63c?w=400",
      ingredients: [
        "1 cup quinoa",
        "Cucumber",
        "Cherry tomatoes",
        "Feta cheese",
        "Olives",
        "Olive oil",
        "Lemon juice",
      ],
      instructions: [
        "Cook quinoa according to package instructions",
        "Dice cucumber and tomatoes",
        "Mix cooled quinoa with vegetables",
        "Add crumbled feta and olives",
        "Dress with olive oil and lemon juice",
      ],
      tags: ["Vegetarian", "Mediterranean", "Protein-rich"],
    },
    {
      id: 3,
      name: "Grilled Salmon with Asparagus",
      category: "Dinner",
      time: 35,
      servings: 2,
      calories: 520,
      difficulty: "Medium",
      rating: 4.7,
      image:
        "https://images.unsplash.com/photo-1467003909585-2f8a72700288?w=400",
      ingredients: [
        "2 salmon fillets",
        "Asparagus spears",
        "Garlic",
        "Lemon",
        "Olive oil",
        "Herbs",
      ],
      instructions: [
        "Preheat grill to medium-high",
        "Season salmon with salt, pepper, and herbs",
        "Toss asparagus with olive oil and garlic",
        "Grill salmon 4-5 minutes per side",
        "Grill asparagus until tender",
        "Serve with lemon wedges",
      ],
      tags: ["Healthy", "Protein-rich", "Low-carb"],
    },
    {
      id: 4,
      name: "Chocolate Lava Cake",
      category: "Dessert",
      time: 20,
      servings: 4,
      calories: 380,
      difficulty: "Hard",
      rating: 4.9,
      image:
        "https://images.unsplash.com/photo-1624353365286-3f8d62daad51?w=400",
      ingredients: [
        "Dark chocolate",
        "Butter",
        "Eggs",
        "Sugar",
        "Flour",
        "Vanilla extract",
      ],
      instructions: [
        "Melt chocolate and butter together",
        "Whisk eggs and sugar until fluffy",
        "Combine chocolate mixture with eggs",
        "Fold in flour gently",
        "Pour into greased ramekins",
        "Bake at 425°F for 12-14 minutes",
      ],
      tags: ["Sweet", "Indulgent", "Party"],
    },
    {
      id: 5,
      name: "Greek Yogurt Parfait",
      category: "Breakfast",
      time: 5,
      servings: 1,
      calories: 280,
      difficulty: "Easy",
      rating: 4.6,
      image:
        "https://images.unsplash.com/photo-1488477181946-6428a0291777?w=400",
      ingredients: [
        "Greek yogurt",
        "Granola",
        "Mixed berries",
        "Honey",
        "Nuts",
      ],
      instructions: [
        "Layer yogurt in a glass",
        "Add granola layer",
        "Top with fresh berries",
        "Drizzle with honey",
        "Sprinkle nuts on top",
      ],
      tags: ["Healthy", "Quick", "Protein-rich"],
    },
    {
      id: 6,
      name: "Spicy Chicken Tacos",
      category: "Dinner",
      time: 30,
      servings: 4,
      calories: 410,
      difficulty: "Easy",
      rating: 4.8,
      image:
        "https://images.unsplash.com/photo-1565299585323-38d6b0865b47?w=400",
      ingredients: [
        "Chicken breast",
        "Taco shells",
        "Lettuce",
        "Tomatoes",
        "Cheese",
        "Sour cream",
        "Spices",
      ],
      instructions: [
        "Season and cook chicken with spices",
        "Shred the cooked chicken",
        "Warm taco shells",
        "Assemble with lettuce, tomatoes, and cheese",
        "Top with sour cream and serve",
      ],
      tags: ["Spicy", "Mexican", "Family-friendly"],
    },
    // New Recipes (12)
    {
      id: 7,
      name: "Fluffy Buttermilk Pancakes",
      category: "Breakfast",
      time: 20,
      servings: 4,
      calories: 350,
      difficulty: "Medium",
      rating: 4.5,
      image: "https://images.unsplash.com/photo-1557999710-18dc60b1356e?w=400",
      ingredients: ["Flour", "Milk", "Eggs", "Baking Powder", "Butter"],
      instructions: [
        "Mix dry ingredients",
        "Mix wet ingredients",
        "Combine and whisk until just mixed",
        "Cook on a hot griddle until golden",
      ],
      tags: ["Sweet", "Classic", "Comfort"],
    },
    {
      id: 8,
      name: "Classic Caesar Salad",
      category: "Lunch",
      time: 15,
      servings: 2,
      calories: 300,
      difficulty: "Easy",
      rating: 4.4,
      image:
        "https://images.unsplash.com/photo-1512852646294-87455855f707?w=400",
      ingredients: ["Romaine", "Croutons", "Parmesan", "Caesar Dressing"],
      instructions: [
        "Wash and chop romaine lettuce",
        "Toss with dressing and parmesan",
        "Top with croutons and serve immediately",
      ],
      tags: ["Quick", "Light", "Salad"],
    },
    {
      id: 9,
      name: "Cheesy Veggie Wrap",
      category: "Lunch",
      time: 10,
      servings: 1,
      calories: 350,
      difficulty: "Easy",
      rating: 4.3,
      image:
        "https://images.unsplash.com/photo-1619895092036-247547562892?w=400",
      ingredients: ["Tortilla", "Hummus", "Spinach", "Bell Peppers", "Cheese"],
      instructions: [
        "Spread hummus on the tortilla",
        "Layer with spinach, peppers, and cheese",
        "Wrap tightly and enjoy",
      ],
      tags: ["Vegetarian", "Quick", "Portable"],
    },
    {
      id: 10,
      name: "Beef and Broccoli Stir Fry",
      category: "Dinner",
      time: 30,
      servings: 3,
      calories: 580,
      difficulty: "Medium",
      rating: 4.6,
      image:
        "https://images.unsplash.com/photo-1534937229272-b0625345a90d?w=400",
      ingredients: ["Beef", "Broccoli", "Soy Sauce", "Ginger", "Rice"],
      instructions: [
        "Cook rice",
        "Stir fry beef until browned",
        "Add broccoli and sauce, cook until tender",
        "Serve over rice",
      ],
      tags: ["Asian", "Quick", "Family-friendly"],
    },
    {
      id: 11,
      name: "Creamy Tiramisu",
      category: "Dessert",
      time: 45,
      servings: 8,
      calories: 420,
      difficulty: "Hard",
      rating: 4.9,
      image:
        "https://images.unsplash.com/photo-1602888494056-cae44f1c97f4?w=400",
      ingredients: [
        "Ladyfingers",
        "Mascarpone",
        "Coffee",
        "Sugar",
        "Cocoa Powder",
      ],
      instructions: [
        "Dip ladyfingers in coffee",
        "Layer with mascarpone mixture",
        "Repeat layers and chill overnight",
        "Dust with cocoa powder before serving",
      ],
      tags: ["Italian", "Rich", "Indulgent"],
    },
    {
      id: 12,
      name: "New York Cheesecake",
      category: "Dessert",
      time: 75,
      servings: 12,
      calories: 450,
      difficulty: "Hard",
      rating: 4.7,
      image:
        "https://images.unsplash.com/photo-1594770267757-010e9ac5b866?w=400",
      ingredients: [
        "Cream Cheese",
        "Graham Crackers",
        "Sugar",
        "Eggs",
        "Vanilla",
      ],
      instructions: [
        "Prepare crust and press into pan",
        "Mix filling ingredients",
        "Pour filling and bake slowly",
        "Chill completely before slicing",
      ],
      tags: ["Sweet", "Classic", "Baked"],
    },
    {
      id: 13,
      name: "Spicy Beef Nachos",
      category: "Snacks",
      time: 20,
      servings: 4,
      calories: 600,
      difficulty: "Easy",
      rating: 4.5,
      image:
        "https://images.unsplash.com/photo-1588668297750-77a834279090?w=400",
      ingredients: [
        "Tortilla Chips",
        "Ground Beef",
        "Cheese",
        "Jalapeños",
        "Salsa",
      ],
      instructions: [
        "Brown ground beef and season",
        "Layer chips, beef, and cheese on a pan",
        "Bake until cheese is melted",
        "Top with salsa and jalapeños",
      ],
      tags: ["Party", "Salty", "Comfort"],
    },
    {
      id: 14,
      name: "Crispy Vegetable Spring Rolls",
      category: "Snacks",
      time: 40,
      servings: 6,
      calories: 250,
      difficulty: "Medium",
      rating: 4.6,
      image:
        "https://images.unsplash.com/photo-1534007996362-e6446e12e02f?w=400",
      ingredients: ["Spring Roll Wrappers", "Cabbage", "Carrots", "Soy Sauce"],
      instructions: [
        "Sauté vegetables",
        "Fill wrappers and fold tightly",
        "Fry until golden and crispy",
        "Serve with dipping sauce",
      ],
      tags: ["Asian", "Fried", "Appetizer"],
    },
    {
      id: 15,
      name: "Loaded Baked Potato Skins",
      category: "Snacks",
      time: 60,
      servings: 4,
      calories: 300,
      difficulty: "Medium",
      rating: 4.7,
      image: "https://images.unsplash.com/photo-1559858711-2d7c50a11e51?w=400",
      ingredients: ["Potatoes", "Bacon", "Cheddar", "Sour Cream", "Chives"],
      instructions: [
        "Bake potatoes, scoop out flesh",
        "Fill skins with cheese and bacon bits",
        "Bake until cheese melts",
        "Top with sour cream and chives",
      ],
      tags: ["Salty", "Comfort", "Party"],
    },
    {
      id: 16,
      name: "Creamy Vegetable Curry",
      category: "Vegetarian",
      time: 40,
      servings: 4,
      calories: 480,
      difficulty: "Medium",
      rating: 4.5,
      image:
        "https://images.unsplash.com/photo-1626804574977-f203407c570b?w=400",
      ingredients: [
        "Mixed Vegetables",
        "Coconut Milk",
        "Curry Paste",
        "Rice",
        "Cilantro",
      ],
      instructions: [
        "Sauté curry paste",
        "Add vegetables and coconut milk",
        "Simmer until vegetables are tender",
        "Serve with rice and cilantro",
      ],
      tags: ["Vegetarian", "Spicy", "Indian"],
    },
    {
      id: 17,
      name: "Caprese Skewers",
      category: "Vegetarian",
      time: 10,
      servings: 4,
      calories: 180,
      difficulty: "Easy",
      rating: 4.8,
      image: "https://images.unsplash.com/photo-1543085526-92147361114d?w=400",
      ingredients: [
        "Cherry Tomatoes",
        "Mozzarella Balls",
        "Basil Leaves",
        "Balsamic Glaze",
      ],
      instructions: [
        "Thread tomato, basil, and mozzarella onto skewers",
        "Drizzle with balsamic glaze",
        "Serve chilled",
      ],
      tags: ["Vegetarian", "Quick", "Appetizer"],
    },
    {
      id: 18,
      name: "Mushroom Risotto",
      category: "Vegetarian",
      time: 45,
      servings: 3,
      calories: 550,
      difficulty: "Hard",
      rating: 4.7,
      image:
        "https://images.unsplash.com/photo-1510001077740-1b20757d5985?w=400",
      ingredients: ["Arborio Rice", "Mushrooms", "Vegetable Broth", "Parmesan"],
      instructions: [
        "Sauté mushrooms and onions",
        "Add rice and toast lightly",
        "Add broth slowly, stirring constantly until creamy",
        "Stir in Parmesan",
      ],
      tags: ["Vegetarian", "Italian", "Comfort"],
    },
  ];

  // Helper function to find a recipe by ID
  const getRecipeById = (id) => allRecipes.find((r) => r.id === id);

  const filteredRecipes = allRecipes.filter((recipe) => {
    const matchesSearch =
      recipe.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
      recipe.tags.some((tag) =>
        tag.toLowerCase().includes(searchQuery.toLowerCase())
      );
    const matchesCategory =
      selectedCategory === "all" ||
      recipe.category.toLowerCase() === selectedCategory; // Fixed category matching
    return matchesSearch && matchesCategory;
  });

  const favoriteRecipes = favorites.map(getRecipeById).filter(Boolean); // Get full recipe objects for favorites

  const toggleFavorite = (recipeId) => {
    setFavorites((prev) =>
      prev.includes(recipeId)
        ? prev.filter((id) => id !== recipeId)
        : [...prev, recipeId]
    );
  };

  const addToMealPlan = (recipe) => {
    if (!mealPlan.find((r) => r.id === recipe.id)) {
      setMealPlan([...mealPlan, recipe]);
    }
  };

  const removeFromMealPlan = (recipeId) => {
    setMealPlan(mealPlan.filter((r) => r.id !== recipeId));
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-orange-50 via-yellow-50 to-pink-50 p-6">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-4xl font-bold bg-gradient-to-r from-orange-600 to-pink-600 bg-clip-text text-transparent mb-2">
            Recipe Finder 👩‍🍳
          </h1>
          <p className="text-gray-600">Discover and plan delicious meals</p>
        </div>

        {/* Search & Filter */}
        <div className="bg-white rounded-2xl p-6 shadow-xl mb-8">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search recipes, ingredients, or tags..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="w-full pl-12 pr-4 py-3 border-2 border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-500 focus:border-transparent"
              />
            </div>
            <div className="flex gap-2 overflow-x-auto pb-2">
              {categories.map((cat) => (
                <button
                  key={cat}
                  onClick={() => setSelectedCategory(cat.toLowerCase())}
                  className={`px-4 py-3 rounded-xl font-medium whitespace-nowrap transition-all ${
                    selectedCategory === cat.toLowerCase()
                      ? "bg-gradient-to-r from-orange-500 to-pink-500 text-white shadow-lg"
                      : "bg-gray-100 text-gray-600 hover:bg-gray-200"
                  }`}
                >
                  {cat}
                </button>
              ))}
            </div>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Recipe Grid */}
          <div className="lg:col-span-2">
            {filteredRecipes.length === 0 ? (
              <div className="text-center py-10 bg-white rounded-2xl shadow-xl text-gray-500">
                <Filter className="w-12 h-12 mx-auto mb-3 opacity-50" />
                <p className="text-lg font-semibold">
                  No recipes found for "{searchQuery}" in{" "}
                  {selectedCategory === "all" ? "All" : selectedCategory}
                </p>
                <p className="text-sm mt-1">
                  Try a different search or category.
                </p>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {filteredRecipes.map((recipe) => (
                  <div
                    key={recipe.id}
                    className="bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all cursor-pointer group"
                    onClick={() => setSelectedRecipe(recipe)}
                  >
                    <div className="relative h-48 overflow-hidden">
                      <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent z-10" />
                      <img
                        src={recipe.image}
                        alt={recipe.name}
                        className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                      />
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          toggleFavorite(recipe.id);
                        }}
                        className="absolute top-4 right-4 z-20 p-2 bg-white/90 rounded-full hover:bg-white transition-all"
                      >
                        <Heart
                          className={`w-5 h-5 ${
                            favorites.includes(recipe.id)
                              ? "fill-red-500 text-red-500"
                              : "text-gray-600"
                          }`}
                        />
                      </button>
                      <div className="absolute bottom-4 left-4 z-20">
                        <span className="px-3 py-1 bg-white/90 rounded-full text-sm font-semibold text-gray-800">
                          {recipe.category}
                        </span>
                      </div>
                    </div>

                    <div className="p-4">
                      <h3 className="font-bold text-lg text-gray-800 mb-2">
                        {recipe.name}
                      </h3>

                      <div className="flex items-center gap-4 text-sm text-gray-600 mb-3">
                        <div className="flex items-center gap-1">
                          <Clock className="w-4 h-4" />
                          <span>{recipe.time} min</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <Users className="w-4 h-4" />
                          <span>{recipe.servings} servings</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <Flame className="w-4 h-4" />
                          <span>{recipe.calories} cal</span>
                        </div>
                      </div>

                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-1">
                          <Star className="w-4 h-4 fill-yellow-400 text-yellow-400" />
                          <span className="text-sm font-semibold">
                            {recipe.rating}
                          </span>
                        </div>
                        <button
                          onClick={(e) => {
                            e.stopPropagation();
                            addToMealPlan(recipe);
                          }}
                          className="px-4 py-2 bg-gradient-to-r from-orange-500 to-pink-500 text-white rounded-lg hover:from-orange-600 hover:to-pink-600 transition-all text-sm font-medium"
                        >
                          Add to Plan
                        </button>
                      </div>

                      <div className="flex flex-wrap gap-2 mt-3">
                        {recipe.tags.slice(0, 3).map((tag) => (
                          <span
                            key={tag}
                            className="px-2 py-1 bg-orange-100 text-orange-700 rounded-full text-xs font-medium"
                          >
                            {tag}
                          </span>
                        ))}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>

          {/* Meal Plan & Favorites Sidebar */}
          <div className="space-y-6">
            {/* Meal Plan Sidebar - Removed sticky classes */}
            <div className="bg-white rounded-2xl p-6 shadow-xl">
              <h2 className="text-xl font-bold text-gray-800 mb-4 flex items-center gap-2">
                <ChefHat className="w-5 h-5 text-orange-600" />
                My Meal Plan
              </h2>

              {mealPlan.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  <BookOpen className="w-12 h-12 mx-auto mb-2 opacity-50" />
                  <p className="text-sm">No meals planned yet</p>
                  <p className="text-xs mt-1">Add recipes to start planning</p>
                </div>
              ) : (
                <div className="space-y-3">
                  {mealPlan.map((recipe) => (
                    <div
                      key={recipe.id}
                      className="bg-gradient-to-br from-orange-50 to-pink-50 rounded-xl p-3"
                    >
                      <div className="flex items-start gap-3">
                        <img
                          src={recipe.image}
                          alt={recipe.name}
                          className="w-16 h-16 rounded-lg object-cover"
                        />
                        <div
                          className="flex-1 min-w-0 cursor-pointer"
                          onClick={() => setSelectedRecipe(recipe)} // View detail from plan
                        >
                          <h3 className="font-semibold text-sm text-gray-800 mb-1 truncate hover:text-orange-600 transition">
                            {recipe.name}
                          </h3>
                          <div className="flex items-center gap-2 text-xs text-gray-600">
                            <Clock className="w-3 h-3" />
                            <span>{recipe.time} min</span>
                            <span className="w-1 h-1 bg-gray-400 rounded-full" />
                            <span>{recipe.calories} cal</span>
                          </div>
                        </div>
                        <button
                          onClick={() => removeFromMealPlan(recipe.id)}
                          className="text-red-500 hover:bg-red-100 p-1 rounded transition-all flex-shrink-0"
                        >
                          <X className="w-4 h-4" />
                        </button>
                      </div>
                    </div>
                  ))}

                  <div className="pt-3 border-t border-orange-200">
                    <div className="flex justify-between text-sm mb-1">
                      <span className="text-gray-600">Total Calories</span>
                      <span className="font-bold text-gray-800">
                        {mealPlan.reduce((sum, r) => sum + r.calories, 0)} cal
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Total Time</span>
                      <span className="font-bold text-gray-800">
                        {mealPlan.reduce((sum, r) => sum + r.time, 0)} min
                      </span>
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Favorites List */}
            <div className="bg-white rounded-2xl p-6 shadow-xl">
              <h2 className="text-xl font-bold text-gray-800 mb-4 flex items-center gap-2">
                <Heart className="w-5 h-5 text-red-500 fill-red-500" />
                Favorites ({favoriteRecipes.length})
              </h2>
              {favoriteRecipes.length === 0 ? (
                <div className="text-center py-4 text-gray-500">
                  <p className="text-sm">
                    Click the heart on a recipe to save it!
                  </p>
                </div>
              ) : (
                <div className="space-y-3">
                  {favoriteRecipes.map((recipe) => (
                    <div
                      key={recipe.id}
                      className="bg-red-50 rounded-xl p-3 border border-red-200"
                    >
                      <div className="flex items-start gap-3">
                        <img
                          src={recipe.image}
                          alt={recipe.name}
                          className="w-16 h-16 rounded-lg object-cover flex-shrink-0"
                        />
                        <div
                          className="flex-1 min-w-0 cursor-pointer"
                          onClick={() => setSelectedRecipe(recipe)}
                        >
                          <h3 className="font-semibold text-sm text-gray-800 mb-1 truncate hover:text-red-600 transition">
                            {recipe.name}
                          </h3>
                          <div className="flex items-center gap-2 text-xs text-gray-600">
                            <Star className="w-3 h-3 fill-yellow-400 text-yellow-400" />
                            <span>{recipe.rating}</span>
                            <span className="w-1 h-1 bg-gray-400 rounded-full" />
                            <span>{recipe.category}</span>
                          </div>
                        </div>
                        <button
                          onClick={() => toggleFavorite(recipe.id)}
                          className="text-red-500 hover:bg-red-100 p-1 rounded transition-all flex-shrink-0"
                        >
                          <X className="w-4 h-4" />
                        </button>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Recipe Detail Modal (Unchanged) */}
        {selectedRecipe && (
          <div
            className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4"
            onClick={() => setSelectedRecipe(null)}
          >
            <div
              className="bg-white rounded-2xl max-w-3xl w-full max-h-[90vh] overflow-y-auto"
              onClick={(e) => e.stopPropagation()}
            >
              <div className="relative h-64">
                <img
                  src={selectedRecipe.image}
                  alt={selectedRecipe.name}
                  className="w-full h-full object-cover"
                />
                <button
                  onClick={() => setSelectedRecipe(null)}
                  className="absolute top-4 right-4 p-2 bg-white rounded-full hover:bg-gray-100 transition-all"
                >
                  <X className="w-5 h-5" />
                </button>
              </div>

              <div className="p-6">
                <h2 className="text-3xl font-bold text-gray-800 mb-4">
                  {selectedRecipe.name}
                </h2>

                <div className="flex flex-wrap items-center gap-6 mb-6 text-sm md:text-base">
                  <div className="flex items-center gap-2">
                    <Clock className="w-5 h-5 text-gray-600" />
                    <span>{selectedRecipe.time} minutes</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Users className="w-5 h-5 text-gray-600" />
                    <span>{selectedRecipe.servings} servings</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Flame className="w-5 h-5 text-gray-600" />
                    <span>{selectedRecipe.calories} cal</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Star className="w-5 h-5 fill-yellow-400 text-yellow-400" />
                    <span className="font-semibold">
                      {selectedRecipe.rating}
                    </span>
                  </div>
                </div>

                <div className="mb-6">
                  <h3 className="text-xl font-bold mb-3">Ingredients</h3>
                  <ul className="space-y-2">
                    {selectedRecipe.ingredients.map((ingredient, idx) => (
                      <li
                        key={idx}
                        className="flex items-center gap-2 text-gray-700"
                      >
                        <div className="w-2 h-2 bg-orange-500 rounded-full flex-shrink-0" />
                        {ingredient}
                      </li>
                    ))}
                  </ul>
                </div>

                <div>
                  <h3 className="text-xl font-bold mb-3">Instructions</h3>
                  <ol className="space-y-3">
                    {selectedRecipe.instructions.map((instruction, idx) => (
                      <li key={idx} className="flex gap-3 text-gray-700">
                        <span className="flex-shrink-0 w-6 h-6 bg-gradient-to-br from-orange-500 to-pink-500 text-white rounded-full flex items-center justify-center text-sm font-bold">
                          {idx + 1}
                        </span>
                        <span className="flex-1">{instruction}</span>
                      </li>
                    ))}
                  </ol>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
