import React, { useEffect, useState } from "react";
import { Mic, Volume2, Moon, Sun, Trash2 } from "lucide-react";

export default function Translator() {
  const [langs, setLangs] = useState([]);
  const [from, setFrom] = useState("auto"); // Auto detect added
  const [to, setTo] = useState("hi");
  const [text, setText] = useState("");
  const [result, setResult] = useState("");
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [dark, setDark] = useState(false);
  const [history, setHistory] = useState([]);

  // Load history on mount
  useEffect(() => {
    const saved = JSON.parse(localStorage.getItem("history")) || [];
    setHistory(saved);
  }, []);

  // Save history automatically
  useEffect(() => {
    localStorage.setItem("history", JSON.stringify(history));
  }, [history]);

  useEffect(() => {
    setLangs([
      { code: "auto", name: "Auto Detect" },
      { code: "en", name: "English" },
      { code: "hi", name: "Hindi" },
      { code: "es", name: "Spanish" },
      { code: "fr", name: "French" },
      { code: "gu", name: "Gujarati" },
      { code: "de", name: "German" },
      { code: "it", name: "Italian" },
      { code: "ja", name: "Japanese" },
    ]);
  }, []);

  // ✔ Auto-detect + Translate
  async function handleTranslate() {
    setError(null);
    setResult("");

    if (!text.trim()) {
      setError("Enter text to translate.");
      return;
    }

    setLoading(true);

    try {
      let langpair = from === "auto" ? `auto|${to}` : `${from}|${to}`;

      const response = await fetch(
        `https://api.mymemory.translated.net/get?q=${encodeURIComponent(
          text
        )}&langpair=${langpair}`
      );

      const data = await response.json();
      setResult(data.responseData.translatedText);

      // Save to history
      setHistory((prev) => [
        { text, result: data.responseData.translatedText, from, to },
        ...prev,
      ]);
    } catch (e) {
      setError("Translation failed.");
    } finally {
      setLoading(false);
    }
  }

  function handleSwap() {
    if (from === "auto") return;
    setFrom(to);
    setTo(from);
    setResult("");
  }

  // ✔ Speech-to-text
  function startSpeechToText() {
    const rec = new window.webkitSpeechRecognition();
    rec.lang = from === "auto" ? "en-US" : from;
    rec.onresult = (e) => setText(e.results[0][0].transcript);
    rec.start();
  }

  // ✔ Text-to-speech output
  function speakText() {
    if (!result) return;
    const speech = new SpeechSynthesisUtterance(result);
    speech.lang = to;
    window.speechSynthesis.speak(speech);
  }

  return (
    <div className={dark ? "bg-gray-900 text-white min-h-screen p-5" : "p-5"}>
      {/* Dark Mode Toggle */}
      <button
        onClick={() => setDark(!dark)}
        className="mb-4 p-2 border rounded flex items-center gap-2"
      >
        {dark ? <Sun size={18} /> : <Moon size={18} />}
        {dark ? "Light Mode" : "Dark Mode"}
      </button>

      <div className="grid gap-4 md:grid-cols-3">
        {/* Input */}
        <div className="md:col-span-2">
          <label className="text-sm font-medium">Input</label>

          <textarea
            value={text}
            onChange={(e) => setText(e.target.value)}
            rows={8}
            className="w-full mt-2 p-3 border rounded-lg resize-none 
             focus:outline-none focus:ring-2 focus:ring-indigo-200 bg-white text-black"
            placeholder="Speak or type text…"
          />

          <div className="flex gap-2 mt-2 items-center">
            <select
              value={from}
              onChange={(e) => setFrom(e.target.value)}
              className="p-2 border rounded bg-white text-black"
            >
              {langs.map((l) => (
                <option key={l.code} value={l.code}>
                  {l.name}
                </option>
              ))}
            </select>

            {/* Speech-to-text */}
            <button
              onClick={startSpeechToText}
              className="px-3 py-2 bg-slate-200 rounded hover:bg-slate-300 flex items-center gap-1"
            >
              <Mic size={16} />
              Speak
            </button>

            <button
              onClick={() => navigator.clipboard.writeText(text)}
              className="px-3 py-2 bg-slate-200 rounded hover:bg-slate-300"
            >
              Copy
            </button>
          </div>
        </div>

        {/* Translate To */}
        <div className="flex flex-col gap-2">
          <label className="text-sm font-medium">Translate To</label>

          <select
            value={to}
            onChange={(e) => setTo(e.target.value)}
            className="p-3 border rounded bg-white text-black"
          >
            {langs.map((l) => (
              <option key={l.code} value={l.code}>
                {l.name}
              </option>
            ))}
          </select>

          <div className="mt-2 flex gap-2">
            <button
              onClick={handleTranslate}
              disabled={loading}
              className="flex-1 px-3 py-2 bg-indigo-600 text-white rounded hover:bg-indigo-700 disabled:opacity-60"
            >
              Translate
            </button>

            <button
              onClick={handleSwap}
              className="px-3 py-2 bg-slate-200 rounded hover:bg-slate-300"
            >
              Swap
            </button>
          </div>
        </div>
      </div>

      {/* Output */}
      <div className="mt-4">
        <label className="text-sm font-medium">Output</label>

        <textarea
          readOnly
          value={result}
          rows={6}
          className="w-full mt-2 p-3 border rounded-lg bg-slate-100 resize-none text-black"
          placeholder="Translation will appear here..."
        />

        <div className="flex gap-2 mt-2">
          {/* Text-to-speech */}
          <button
            onClick={speakText}
            className="px-3 py-2 bg-slate-200 rounded hover:bg-slate-300 flex items-center gap-1"
          >
            <Volume2 size={16} /> Listen
          </button>

          <button
            onClick={() => navigator.clipboard.writeText(result)}
            className="px-3 py-2 bg-slate-200 rounded hover:bg-slate-300"
          >
            Copy Output
          </button>

          <button
            onClick={() => {
              setText("");
              setResult("");
            }}
            className="px-3 py-2 bg-slate-200 rounded hover:bg-slate-300"
          >
            Clear
          </button>
        </div>
      </div>

      {/* History */}
      <div className="mt-6">
        <h2 className="text-lg font-semibold mb-2">Translation History</h2>

        <button
          onClick={() => setHistory([])}
          className="mb-3 p-2 bg-red-100 text-red-600 rounded flex items-center gap-1"
        >
          <Trash2 size={16} /> Clear History
        </button>

        <div className="flex flex-col gap-2">
          {history.map((item, i) => (
            <div key={i} className="p-3 border rounded bg-white text-black">
              <p>
                <strong>Input:</strong> {item.text}
              </p>
              <p>
                <strong>Output:</strong> {item.result}
              </p>
              <p className="text-sm text-gray-500">
                {item.from} → {item.to}
              </p>
            </div>
          ))}
        </div>
      </div>

      {error && <div className="mt-3 text-sm text-red-600">{error}</div>}
      {loading && <div className="mt-3 text-sm text-slate-500">Working...</div>}
    </div>
  );
}
