import React, { useState } from "react";
import Modal from "./Modal";
import { saveData } from "../utils/localStorage";

export default function Workouts({ workouts, setWorkouts }) {
  const [open, setOpen] = useState(false);
  const [form, setForm] = useState({ name: "", duration: "", date: "" });

  // 🔒 Limit date input up to today
  const today = new Date().toISOString().split("T")[0];

  const add = () => {
    if (!form.name || !form.duration) return alert("Enter name + duration");

    // 🧭 Prevent saving future dates
    if (form.date && new Date(form.date) > new Date()) {
      return alert("Date cannot be in the future!");
    }

    const item = { id: Date.now(), ...form, duration: Number(form.duration) };
    const next = [item, ...workouts];
    setWorkouts(next);
    saveData("workouts", next);
    setForm({ name: "", duration: "", date: "" });
    setOpen(false);
  };

  const del = (id) => {
    const next = workouts.filter((w) => w.id !== id);
    setWorkouts(next);
    saveData("workouts", next);
  };

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold">Workouts</h2>
        <button
          onClick={() => setOpen(true)}
          className="bg-purple-600 text-white px-4 py-2 rounded-lg"
        >
          + Add
        </button>
      </div>

      {workouts.length === 0 ? (
        <p className="text-gray-500">No workouts added.</p>
      ) : (
        <ul className="space-y-3">
          {workouts.map((w) => (
            <li
              key={w.id}
              className="bg-[color:var(--card)] p-4 rounded-xl flex justify-between items-center shadow"
            >
              <div>
                <div className="font-medium">{w.name}</div>
                <div className="text-sm text-gray-500">{w.duration} mins</div>
              </div>
              <div className="flex items-center gap-2">
                <div className="text-sm text-gray-500">{w.date || ""}</div>
                <button onClick={() => del(w.id)} className="text-red-500">
                  ✕
                </button>
              </div>
            </li>
          ))}
        </ul>
      )}

      {open && (
        <Modal title="Add Workout" onClose={() => setOpen(false)}>
          <div className="flex flex-col gap-3">
            <input
              value={form.name}
              onChange={(e) => setForm({ ...form, name: e.target.value })}
              placeholder="Name"
              className="border p-2 rounded"
            />
            <input
              value={form.duration}
              onChange={(e) => setForm({ ...form, duration: e.target.value })}
              placeholder="Duration (min)"
              type="number"
              className="border p-2 rounded"
            />
            <input
              value={form.date}
              onChange={(e) => setForm({ ...form, date: e.target.value })}
              type="date"
              max={today} // ✅ Prevent selecting future dates
              className="border p-2 rounded"
            />
            <button
              onClick={add}
              className="bg-purple-600 text-white py-2 rounded"
            >
              Add
            </button>
          </div>
        </Modal>
      )}
    </div>
  );
}
