import React, { useEffect, useState } from 'react'
import { Routes, Route, useNavigate } from 'react-router-dom'
import Navbar from './components/Navbar'
import Dashboard from './components/Dashboard'
import Workouts from './components/Workouts'
import Meals from './components/Meals'
import Progress from './components/Progress'
import { loadData, saveData } from './utils/localStorage'

export default function App() {
  const [workouts, setWorkouts] = useState(() => loadData('workouts', []))
  const [meals, setMeals] = useState(() => loadData('meals', []))
  const [dark, setDark] = useState(() => loadData('themeDark', false))
  const navigate = useNavigate()

  useEffect(() => saveData('workouts', workouts), [workouts])
  useEffect(() => saveData('meals', meals), [meals])
  useEffect(() => saveData('themeDark', dark), [dark])

  useEffect(() => {
    if (dark) document.documentElement.classList.add('dark')
    else document.documentElement.classList.remove('dark')
  }, [dark])

  const resetAll = () => {
    if (!confirm('Delete ALL saved workouts & meals? This cannot be undone.')) return
    setWorkouts([])
    setMeals([])
    localStorage.removeItem('workouts')
    localStorage.removeItem('meals')
    navigate('/')
    alert('All data cleared.')
  }

  return (
    <div className="min-h-screen bg-[color:var(--bg)] transition-colors">
      <div className="max-w-6xl mx-auto p-4 md:flex md:gap-6">
        <Navbar dark={dark} setDark={setDark} resetAll={resetAll} />
        <main className="flex-1 mt-4 md:mt-0">
          <Routes>
            <Route index element={<Dashboard workouts={workouts} meals={meals} />} />
            <Route path='/workouts' element={<Workouts workouts={workouts} setWorkouts={setWorkouts} />} />
            <Route path='/meals' element={<Meals meals={meals} setMeals={setMeals} />} />
            <Route path='/progress' element={<Progress workouts={workouts} meals={meals} />} />
            <Route path='*' element={<Dashboard workouts={workouts} meals={meals} />} />
          </Routes>
        </main>
      </div>
    </div>
  )
}