import React, { useEffect, useState } from "react";
import axios from "axios";
import TodoForm from "../components/TodoForm";
import TodoItem from "../components/TodoItem";

const API_URL = "http://localhost:5000/api/todos";

export default function TodoList() {
  const [todos, setTodos] = useState([]);
  const [filter, setFilter] = useState("all");
  const [loading, setLoading] = useState(false);

  const fetchTodos = async () => {
    setLoading(true);
    try {
      const res = await axios.get(API_URL, {
        params: { status: filter === "all" ? "" : filter },
      });
      setTodos(res.data);
    } catch (error) {
      console.error("Error fetching todos:", error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchTodos();
  }, [filter]);

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-blue-100 py-12 px-6 flex flex-col items-center">
      <div className="w-full max-w-2xl bg-white/70 backdrop-blur-md shadow-sm rounded-sm p-8 border border-blue-100">
        <h1 className="text-4xl font-extrabold text-center mb-8 text-blue-600 tracking-tight">
          My Daily Tasks
        </h1>

        {/* Todo Form */}
        <TodoForm onAdd={fetchTodos} />

        {/* Filter Buttons */}
        <div className="flex justify-center gap-3 mt-6 mb-8">
          {["all", "completed", "pending"].map((status) => (
            <button
              key={status}
              onClick={() => setFilter(status)}
              className={`px-5 py-2 rounded-full text-sm font-semibold capitalize transition-all duration-300 ${
                filter === status
                  ? "bg-blue-600 text-white shadow-md scale-105"
                  : "border border-blue-100 text-blue-600 hover:bg-blue-200"
              }`}
            >
              {status}
            </button>
          ))}
        </div>

        {/* Todo List */}
        {loading ? (
          <div className="text-center py-10 text-blue-500 font-medium animate-pulse">
            Loading tasks...
          </div>
        ) : todos.length === 0 ? (
          <div className="text-center py-10 text-gray-500 font-medium">
            No tasks found — try adding one!
          </div>
        ) : (
          <ul className="space-y-3">
            {todos.map((todo) => (
              <TodoItem key={todo._id} todo={todo} onChange={fetchTodos} />
            ))}
          </ul>
        )}
      </div>
    </div>
  );
}
