import React, { useState } from "react";
import axios from "axios";

const API_URL = "http://localhost:5000/api/todos";

export default function TodoItem({ todo, onChange }) {
  const [editing, setEditing] = useState(false);
  const [title, setTitle] = useState(todo.title);
  const [loading, setLoading] = useState(false);

  const toggleComplete = async () => {
    setLoading(true);
    await axios.put(`${API_URL}/${todo._id}`, { completed: !todo.completed });
    setLoading(false);
    onChange();
  };

  const handleDelete = async () => {
    setLoading(true);
    await axios.delete(`${API_URL}/${todo._id}`);
    setLoading(false);
    onChange();
  };

  const handleEdit = async () => {
    if (!title.trim()) return;
    setLoading(true);
    await axios.put(`${API_URL}/${todo._id}`, { title });
    setEditing(false);
    setLoading(false);
    onChange();
  };

  return (
    <li
      className={`flex items-center justify-between bg-white/80 backdrop-blur-md border border-blue-100 rounded-md px-4 py-3 shadow-sm hover:shadow-md transition-all duration-300 ${
        todo.completed ? "opacity-80" : ""
      }`}
    >
      <div className="flex items-center gap-3 flex-1">
        <input
          type="checkbox"
          checked={todo.completed}
          onChange={toggleComplete}
          disabled={loading}
          className="w-5 h-5 accent-blue-600 cursor-pointer"
        />

        {editing ? (
          <input
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            onBlur={handleEdit}
            autoFocus
            className="flex-1 border border-blue-300 rounded-lg px-2 py-1 text-gray-700 focus:ring-2 focus:ring-blue-500 outline-none"
          />
        ) : (
          <span
            className={`flex-1 text-base cursor-text ${
              todo.completed
                ? "line-through text-gray-400"
                : "text-gray-800 hover:text-blue-600"
            } transition-all duration-300`}
            onDoubleClick={() => setEditing(true)}
          >
            {todo.title}
          </span>
        )}
      </div>

      <button
        onClick={handleDelete}
        disabled={loading}
        className="text-red-500 hover:text-red-600 text-sm font-semibold px-3 py-1 rounded-lg hover:bg-red-50 transition-all duration-200"
      >
        Delete
      </button>
    </li>
  );
}
