import React, { useEffect, useState } from "react";
import api from "../api";
import { Link } from "react-router-dom";
import { toast } from "react-toastify";
import { ShoppingBag, Edit, Trash2 } from "lucide-react";

export default function ProductList() {
  const [products, setProducts] = useState([]);

  useEffect(() => {
    fetchProducts();
  }, []);

  async function fetchProducts() {
    const res = await api.get("/products");
    setProducts(res.data);
  }

  async function remove(id) {
    if (!confirm("Delete this product?")) return;
    try {
      await api.delete("/products/" + id);
      toast.info("Product deleted!");
      fetchProducts();
    } catch (err) {
      console.error(err);
      toast.error("Failed to delete product.");
    }
  }

  return (
    <div>
      {/* Header */}
      <div className="flex justify-between items-center mb-8">
        <h2 className="text-2xl font-bold flex items-center gap-2 text-gray-800">
          <ShoppingBag className="w-6 h-6 text-sky-600" />
          Product Collection
        </h2>
        <Link
          to="/add"
          className="hidden sm:inline-block bg-sky-600 hover:bg-sky-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
        >
          + Add Product
        </Link>
      </div>

      {/* Empty State */}
      {products.length === 0 ? (
        <div className="text-center py-16 bg-white rounded-2xl shadow-sm border border-gray-100">
          <p className="text-gray-500">
            No products available. Start by adding a new one!
          </p>
          <Link
            to="/add"
            className="mt-4 inline-block bg-sky-600 text-white px-4 py-2 rounded-lg text-sm hover:bg-sky-700 transition-colors"
          >
            Add Product
          </Link>
        </div>
      ) : (
        <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4">
          {products.map((p) => (
            <div
              key={p._id}
              className="group bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden hover:shadow-lg hover:-translate-y-1 transition-all duration-200"
            >
              {/* Image */}
              {p.image ? (
                <img
                  src={`http://localhost:5000${p.image}`}
                  alt={p.name}
                  className="w-full h-48 object-cover transition-transform duration-300 group-hover:scale-105"
                />
              ) : (
                <div className="w-full h-48 flex items-center justify-center bg-gray-100 text-gray-400">
                  No Image
                </div>
              )}

              {/* Info */}
              <div className="p-4">
                <h3 className="font-semibold text-gray-800 truncate">
                  {p.name}
                </h3>
                <p className="text-sm text-gray-500 mt-1 line-clamp-2">
                  {p.description || "No description provided."}
                </p>

                <div className="mt-4 flex items-center justify-between">
                  <span className="text-sky-600 font-bold text-lg">
                    ₹{p.price}
                  </span>
                  <div className="flex items-center gap-3">
                    <Link
                      to={`/edit/${p._id}`}
                      title="Edit"
                      className="text-sky-600 hover:text-sky-700 transition-colors"
                    >
                      <Edit size={18} />
                    </Link>
                    <button
                      onClick={() => remove(p._id)}
                      title="Delete"
                      className="text-red-500 hover:text-red-600 transition-colors"
                    >
                      <Trash2 size={18} />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
