import React, { useEffect, useState } from "react";
import api from "../api";
import { useNavigate, useParams } from "react-router-dom";
import { toast } from "react-toastify";
import { Upload, Image as ImageIcon, Save, X } from "lucide-react";

export default function AddEditProduct() {
  const [form, setForm] = useState({
    name: "",
    price: "",
    description: "",
    image: null,
  });
  const [preview, setPreview] = useState(null);
  const navigate = useNavigate();
  const { id } = useParams();

  useEffect(() => {
    if (id) load();
  }, [id]);

  async function load() {
    const res = await api.get("/products");
    const found = res.data.find((p) => p._id === id);
    if (found) {
      setForm({ ...found, image: null });
      setPreview(`http://localhost:5000${found.image}`);
    }
  }

  async function submit(e) {
    e.preventDefault();
    const data = new FormData();
    Object.entries(form).forEach(([k, v]) => {
      if (v) data.append(k, v);
    });

    try {
      if (id) {
        await api.put(`/products/${id}`, data);
        toast.success("Product updated successfully!");
      } else {
        await api.post("/products", data);
        toast.success("Product added successfully!");
      }
      navigate("/");
    } catch (err) {
      console.error(err);
      toast.error("Something went wrong. Please try again!");
    }
  }

  function handleImageChange(e) {
    const file = e.target.files[0];
    if (file) {
      setForm((f) => ({ ...f, image: file }));
      setPreview(URL.createObjectURL(file));
    }
  }

  return (
    <div className="max-w-5xl mx-auto">
      <h2 className="text-3xl font-bold text-gray-800 mb-8">
        {id ? "Edit Product" : "Add New Product"}
      </h2>

      <form
        onSubmit={submit}
        className="grid md:grid-cols-2 gap-8 bg-white rounded-2xl p-8 shadow-sm border border-gray-100"
      >
        {/* Left Section - Form Fields */}
        <div className="space-y-5">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Product Name
            </label>
            <input
              required
              placeholder="Enter product name"
              value={form.name}
              onChange={(e) => setForm((f) => ({ ...f, name: e.target.value }))}
              className="w-full border border-gray-300 rounded-lg p-2.5 focus:ring-2 focus:ring-sky-500 focus:border-sky-500 text-sm"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Price
            </label>
            <input
              required
              type="number"
              placeholder="Enter price"
              value={form.price}
              onChange={(e) =>
                setForm((f) => ({ ...f, price: e.target.value }))
              }
              className="w-full border border-gray-300 rounded-lg p-2.5 focus:ring-2 focus:ring-sky-500 focus:border-sky-500 text-sm"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Description
            </label>
            <textarea
              placeholder="Add a short product description..."
              value={form.description}
              onChange={(e) =>
                setForm((f) => ({ ...f, description: e.target.value }))
              }
              className="w-full border border-gray-300 rounded-lg p-2.5 focus:ring-2 focus:ring-sky-500 focus:border-sky-500 text-sm"
              rows="4"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Upload Image
            </label>
            <div className="flex items-center justify-center w-full">
              <label className="flex flex-col items-center justify-center w-full h-32 border-2 border-dashed border-gray-300 rounded-lg cursor-pointer bg-gray-50 hover:bg-gray-100 transition-colors">
                <div className="flex flex-col items-center justify-center pt-4">
                  <Upload className="w-6 h-6 mb-1 text-gray-500" />
                  <p className="text-sm text-gray-500">Click to upload image</p>
                </div>
                <input
                  type="file"
                  accept="image/*"
                  className="hidden"
                  onChange={handleImageChange}
                />
              </label>
            </div>
          </div>

          <div className="flex gap-3 pt-2">
            <button
              type="submit"
              className="flex items-center gap-2 bg-sky-600 hover:bg-sky-700 text-white px-5 py-2.5 rounded-lg text-sm font-medium transition-colors"
            >
              <Save size={16} />
              Save
            </button>
            <button
              type="button"
              onClick={() => navigate(-1)}
              className="flex items-center gap-2 bg-gray-200 hover:bg-gray-300 text-gray-700 px-5 py-2.5 rounded-lg text-sm font-medium transition-colors"
            >
              <X size={16} />
              Cancel
            </button>
          </div>
        </div>

        {/* Right Section - Image Preview */}
        <div className="flex flex-col items-center justify-center bg-gray-50 rounded-lg border border-gray-200 p-4">
          {preview ? (
            <img
              src={preview}
              alt="Preview"
              className="rounded-lg w-full max-h-80 object-contain shadow-sm"
            />
          ) : (
            <div className="flex flex-col items-center text-gray-400">
              <ImageIcon className="w-12 h-12 mb-2" />
              <p className="text-sm">No image selected</p>
            </div>
          )}
        </div>
      </form>
    </div>
  );
}
