import React, { useEffect, useState } from "react";
import api from "../api";
import { Link } from "react-router-dom";

export default function ExpenseList() {
  const [expenses, setExpenses] = useState([]);
  const [filters, setFilters] = useState({
    category: "",
    from: "",
    to: "",
    q: "",
  });

  useEffect(() => {
    fetchExpenses();
  }, []);

  async function fetchExpenses() {
    const res = await api.get("/expenses", { params: filters });
    setExpenses(res.data);
  }

  async function remove(id) {
    if (!confirm("Delete this expense?")) return;
    await api.delete("/expenses/" + id);
    fetchExpenses();
  }

  return (
    <div className="max-w-5xl mx-auto">
      {/* Header */}
      <div className="flex justify-between items-center mb-6">
        <h2 className="text-2xl font-bold text-gray-800">Expenses</h2>
        <Link
          to="/add"
          className="bg-sky-500 hover:bg-sky-600 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
        >
          + Add Expense
        </Link>
      </div>

      {/* Filter Toolbar */}
      <div className="bg-white p-4 rounded-xl shadow-sm border border-gray-100 mb-8">
        <div className="grid grid-cols-1 md:grid-cols-5 gap-3">
          <input
            placeholder="Search title..."
            value={filters.q}
            onChange={(e) => setFilters((f) => ({ ...f, q: e.target.value }))}
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
          <input
            type="date"
            value={filters.from}
            onChange={(e) =>
              setFilters((f) => ({ ...f, from: e.target.value }))
            }
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
          <input
            type="date"
            value={filters.to}
            onChange={(e) => setFilters((f) => ({ ...f, to: e.target.value }))}
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
          <input
            placeholder="Category..."
            value={filters.category}
            onChange={(e) =>
              setFilters((f) => ({ ...f, category: e.target.value }))
            }
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
          <button
            onClick={fetchExpenses}
            className="bg-gray-800 text-white rounded-lg px-4 py-2 text-sm font-medium hover:bg-gray-900 transition-colors"
          >
            Filter
          </button>
        </div>
      </div>

      {/* Expense List */}
      {expenses.length === 0 ? (
        <div className="text-center py-10 text-gray-500">
          No expenses found. Try adjusting your filters.
        </div>
      ) : (
        <div className="space-y-3">
          {expenses.map((exp) => (
            <div
              key={exp._id}
              className="bg-white p-4 rounded-xl border border-gray-100 shadow-sm hover:shadow-md transition-all flex justify-between items-center"
            >
              <div>
                <div className="font-semibold text-gray-800">{exp.title}</div>
                <div className="text-sm text-gray-500">
                  {new Date(exp.date).toLocaleDateString()} • {exp.category}
                </div>
              </div>

              <div className="flex items-center gap-4">
                <div className="text-lg font-bold text-gray-800">
                  ₹{exp.amount.toLocaleString()}
                </div>
                <Link
                  to={"/edit/" + exp._id}
                  className="text-sm text-sky-600 hover:text-sky-700 bg-sky-200 rounded-md px-4 py-1.5 font-medium"
                >
                  Edit
                </Link>
                <button
                  onClick={() => remove(exp._id)}
                  className="text-sm text-red-600 hover:text-red-700  bg-red-200 rounded-md px-4 py-1.5 font-medium"
                >
                  Delete
                </button>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
