import React, { useEffect, useState } from "react";
import api from "../api";
import {
  PieChart,
  Pie,
  Cell,
  Tooltip,
  Legend,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  ResponsiveContainer,
  CartesianGrid,
} from "recharts";

const COLORS = [
  "#3b82f6",
  "#22c55e",
  "#facc15",
  "#f97316",
  "#a855f7",
  "#ef4444",
];

export default function Dashboard() {
  const [data, setData] = useState([]);
  const [byCategory, setByCategory] = useState([]);
  const [monthly, setMonthly] = useState([]);
  const [total, setTotal] = useState(0);

  useEffect(() => {
    fetchData();
  }, []);

  async function fetchData() {
    const res = await api.get("/expenses");
    setData(res.data);
    setTotal(res.data.reduce((sum, e) => sum + e.amount, 0));

    // category aggregation
    const catMap = {};
    res.data.forEach((e) => {
      catMap[e.category] = (catMap[e.category] || 0) + e.amount;
    });
    setByCategory(
      Object.keys(catMap).map((k) => ({ name: k, value: catMap[k] }))
    );

    // monthly aggregation
    const monthMap = {};
    res.data.forEach((e) => {
      const m = new Date(e.date).toISOString().slice(0, 7);
      monthMap[m] = (monthMap[m] || 0) + e.amount;
    });
    setMonthly(
      Object.keys(monthMap)
        .sort()
        .map((k) => ({ month: k, amount: monthMap[k] }))
    );
  }

  return (
    <div className="max-w-6xl mx-auto space-y-8">
      {/* Summary Card */}
      <div className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 text-center">
        <h2 className="text-2xl font-bold text-gray-800 mb-2">
          Dashboard Overview
        </h2>
        <p className="text-gray-500 text-sm">
          A quick insight into your spending
        </p>
        <div className="mt-4">
          <span className="text-4xl font-bold text-sky-600">
            ₹{total.toLocaleString()}
          </span>
          <p className="text-gray-500 text-sm mt-1">Total Expenses</p>
        </div>
      </div>

      {/* Charts */}
      <div className="grid md:grid-cols-2 gap-6">
        {/* Pie Chart Card */}
        <div className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 hover:shadow-md transition-all">
          <h3 className="text-lg font-semibold text-gray-800 mb-4 flex items-center gap-2">
            Expenses by Category
          </h3>
          {byCategory.length === 0 ? (
            <p className="text-sm text-gray-500 text-center py-8">
              No data available
            </p>
          ) : (
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={byCategory}
                  dataKey="value"
                  nameKey="name"
                  outerRadius={110}
                  label={({ name, percent }) =>
                    `${name} ${(percent * 100).toFixed(0)}%`
                  }
                >
                  {byCategory.map((_, i) => (
                    <Cell key={i} fill={COLORS[i % COLORS.length]} />
                  ))}
                </Pie>
                <Tooltip />
                <Legend />
              </PieChart>
            </ResponsiveContainer>
          )}
        </div>

        {/* Bar Chart Card */}
        <div className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 hover:shadow-md transition-all">
          <h3 className="text-lg font-semibold text-gray-800 mb-4 flex items-center gap-2">
            Monthly Expenses
          </h3>
          {monthly.length === 0 ? (
            <p className="text-sm text-gray-500 text-center py-8">
              No data available
            </p>
          ) : (
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={monthly}>
                <CartesianGrid strokeDasharray="3 3" stroke="#e5e7eb" />
                <XAxis dataKey="month" stroke="#6b7280" />
                <YAxis stroke="#6b7280" />
                <Tooltip />
                <Bar dataKey="amount" radius={[8, 8, 0, 0]}>
                  {monthly.map((_, i) => (
                    <Cell key={i} fill={COLORS[i % COLORS.length]} />
                  ))}
                </Bar>
              </BarChart>
            </ResponsiveContainer>
          )}
        </div>
      </div>
    </div>
  );
}
