import React, { useState, useEffect } from "react";
import api from "../api";
import { useNavigate, useParams } from "react-router-dom";

export default function ExpenseForm() {
  const [form, setForm] = useState({
    title: "",
    amount: "",
    category: "",
    date: new Date().toISOString().slice(0, 10),
  });
  const navigate = useNavigate();
  const { id } = useParams();

  useEffect(() => {
    if (id) load();
  }, [id]);

  async function load() {
    const res = await api.get("/expenses");
    const found = res.data.find((e) => e._id === id);
    if (found)
      setForm({
        title: found.title,
        amount: found.amount,
        category: found.category,
        date: new Date(found.date).toISOString().slice(0, 10),
      });
  }

  async function submit(e) {
    e.preventDefault();
    const payload = { ...form, amount: Number(form.amount) };
    if (id) {
      await api.put("/expenses/" + id, payload);
    } else {
      await api.post("/expenses", payload);
    }
    navigate("/expenses");
  }

  return (
    <div className="max-w-2xl mx-auto">
      <h2 className="text-2xl font-bold text-sky-500 mb-6">
        {id ? "Edit Expense" : "Add New Expense"}
      </h2>

      <form
        onSubmit={submit}
        className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 space-y-5"
      >
        {/* Title */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Title
          </label>
          <input
            required
            placeholder="e.g. Grocery shopping"
            value={form.title}
            onChange={(e) => setForm((f) => ({ ...f, title: e.target.value }))}
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
        </div>

        {/* Amount */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Amount
          </label>
          <input
            required
            type="number"
            placeholder="e.g. 1200"
            value={form.amount}
            onChange={(e) => setForm((f) => ({ ...f, amount: e.target.value }))}
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
        </div>

        {/* Category */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Category
          </label>
          <input
            required
            placeholder="e.g. Food, Rent, Bills"
            value={form.category}
            onChange={(e) =>
              setForm((f) => ({ ...f, category: e.target.value }))
            }
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
        </div>

        {/* Date */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Date
          </label>
          <input
            required
            type="date"
            value={form.date}
            onChange={(e) => setForm((f) => ({ ...f, date: e.target.value }))}
            className="w-full rounded-lg border border-gray-300 focus:outline-0 focus:border-sky-300 p-2.5 text-sm"
          />
        </div>

        {/* Buttons */}
        <div className="flex gap-3 pt-2">
          <button
            type="submit"
            className="px-5 py-2.5 bg-sky-500 text-white rounded-lg text-sm font-medium hover:bg-sky-600 transition-colors"
          >
            {id ? "Update Expense" : "Save Expense"}
          </button>
          <button
            type="button"
            onClick={() => navigate(-1)}
            className="px-5 py-2.5 bg-gray-100 text-gray-700 rounded-lg text-sm font-medium hover:bg-gray-200 transition-colors"
          >
            Cancel
          </button>
        </div>
      </form>
    </div>
  );
}
