import React, { useState, useEffect, useCallback } from "react";
import { Search, RotateCcw, Trophy, Zap, Lightbulb } from "lucide-react";

const THEMES = {
  animals: [
    "ELEPHANT",
    "GIRAFFE",
    "TIGER",
    "PENGUIN",
    "DOLPHIN",
    "KANGAROO",
    "CHEETAH",
    "ZEBRA",
  ],
  fruits: [
    "BANANA",
    "ORANGE",
    "STRAWBERRY",
    "MANGO",
    "PINEAPPLE",
    "GRAPE",
    "WATERMELON",
    "APPLE",
  ],
  countries: [
    "FRANCE",
    "JAPAN",
    "BRAZIL",
    "CANADA",
    "EGYPT",
    "INDIA",
    "NORWAY",
    "MEXICO",
  ],
  space: [
    "PLANET",
    "GALAXY",
    "METEOR",
    "COMET",
    "ASTEROID",
    "NEBULA",
    "ROCKET",
    "ORBIT",
  ],
};

const GRID_SIZE = 12;
const DIRECTIONS = [
  [0, 1],
  [1, 0],
  [1, 1],
  [-1, 1],
  [0, -1],
  [-1, 0],
  [-1, -1],
  [1, -1],
];

const WordSearchGame = () => {
  const [theme, setTheme] = useState("animals");
  const [grid, setGrid] = useState([]);
  const [words, setWords] = useState([]);
  const [foundWords, setFoundWords] = useState([]);
  const [selecting, setSelecting] = useState(false);
  const [selectedCells, setSelectedCells] = useState([]);
  const [highlightedCells, setHighlightedCells] = useState(new Set());
  const [score, setScore] = useState(0);
  const [timer, setTimer] = useState(0);
  const [gameActive, setGameActive] = useState(false);
  const [hints, setHints] = useState(3);
  const [hintCells, setHintCells] = useState(new Set());

  const generateGrid = useCallback(() => {
    const newGrid = Array(GRID_SIZE)
      .fill(null)
      .map(() => Array(GRID_SIZE).fill(""));
    const themeWords = [...THEMES[theme]];
    const placedWords = [];
    const usedCells = new Set();

    const canPlaceWord = (word, row, col, dir) => {
      for (let i = 0; i < word.length; i++) {
        const newRow = row + dir[0] * i;
        const newCol = col + dir[1] * i;
        if (
          newRow < 0 ||
          newRow >= GRID_SIZE ||
          newCol < 0 ||
          newCol >= GRID_SIZE
        ) {
          return false;
        }
        const cell = `${newRow},${newCol}`;
        if (
          newGrid[newRow][newCol] !== "" &&
          newGrid[newRow][newCol] !== word[i]
        ) {
          return false;
        }
      }
      return true;
    };

    const placeWord = (word) => {
      for (let attempt = 0; attempt < 100; attempt++) {
        const dir = DIRECTIONS[Math.floor(Math.random() * DIRECTIONS.length)];
        const row = Math.floor(Math.random() * GRID_SIZE);
        const col = Math.floor(Math.random() * GRID_SIZE);

        if (canPlaceWord(word, row, col, dir)) {
          const cells = [];
          for (let i = 0; i < word.length; i++) {
            const newRow = row + dir[0] * i;
            const newCol = col + dir[1] * i;
            newGrid[newRow][newCol] = word[i];
            cells.push(`${newRow},${newCol}`);
            usedCells.add(`${newRow},${newCol}`);
          }
          placedWords.push({ word, cells });
          return true;
        }
      }
      return false;
    };

    themeWords.forEach((word) => placeWord(word));

    for (let i = 0; i < GRID_SIZE; i++) {
      for (let j = 0; j < GRID_SIZE; j++) {
        if (newGrid[i][j] === "") {
          newGrid[i][j] = String.fromCharCode(
            65 + Math.floor(Math.random() * 26)
          );
        }
      }
    }

    setGrid(newGrid);
    setWords(placedWords);
    setFoundWords([]);
    setHighlightedCells(new Set());
    setScore(0);
    setTimer(0);
    setGameActive(true);
    setHints(3);
    setHintCells(new Set());
  }, [theme]);

  useEffect(() => {
    generateGrid();
  }, [generateGrid]);

  useEffect(() => {
    let interval;
    if (gameActive && foundWords.length < words.length) {
      interval = setInterval(() => {
        setTimer((t) => t + 1);
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [gameActive, foundWords.length, words.length]);

  const handleMouseDown = (row, col) => {
    setSelecting(true);
    setSelectedCells([`${row},${col}`]);
  };

  const handleMouseEnter = (row, col) => {
    if (selecting) {
      const cell = `${row},${col}`;
      setSelectedCells((prev) => {
        if (prev.includes(cell)) return prev;
        const [startRow, startCol] = prev[0].split(",").map(Number);
        const rowDiff = row - startRow;
        const colDiff = col - startCol;
        const isValidDirection =
          rowDiff === 0 ||
          colDiff === 0 ||
          Math.abs(rowDiff) === Math.abs(colDiff);

        if (isValidDirection) {
          return [...prev, cell];
        }
        return prev;
      });
    }
  };

  const handleMouseUp = () => {
    if (selectedCells.length > 0) {
      const selectedWord = selectedCells
        .map((cell) => {
          const [row, col] = cell.split(",").map(Number);
          return grid[row][col];
        })
        .join("");

      const reverseWord = selectedWord.split("").reverse().join("");

      const foundWord = words.find(
        (w) =>
          !foundWords.includes(w.word) &&
          (w.word === selectedWord || w.word === reverseWord)
      );

      if (foundWord) {
        setFoundWords((prev) => [...prev, foundWord.word]);
        setHighlightedCells((prev) => new Set([...prev, ...foundWord.cells]));
        setScore((s) => s + foundWord.word.length * 10);
      }
    }
    setSelecting(false);
    setSelectedCells([]);
  };

  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, "0")}`;
  };

  const useHint = () => {
    if (hints <= 0) return;

    const unFoundWords = words.filter((w) => !foundWords.includes(w.word));
    if (unFoundWords.length === 0) return;

    const randomWord =
      unFoundWords[Math.floor(Math.random() * unFoundWords.length)];
    const firstCell = randomWord.cells[0];

    setHintCells((prev) => new Set([...prev, firstCell]));
    setHints((h) => h - 1);

    setTimeout(() => {
      setHintCells((prev) => {
        const newSet = new Set(prev);
        newSet.delete(firstCell);
        return newSet;
      });
    }, 3000);
  };

  const isGameComplete = foundWords.length === words.length;

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-900 via-blue-900 to-indigo-900 p-4 md:p-8">
      <div className="max-w-6xl mx-auto">
        {/* Header */}
        <div className="text-center mb-8">
          <h1 className="text-4xl md:text-5xl font-bold text-white mb-2 flex items-center justify-center gap-3">
            <Search className="w-10 h-10" />
            Word Search Quest
          </h1>
          <p className="text-blue-200 text-lg">Find all the hidden words!</p>
        </div>

        {/* Stats Bar */}
        <div className="bg-white/10 backdrop-blur-md rounded-xl p-4 mb-6 flex flex-wrap gap-4 justify-between items-center">
          <div className="flex items-center gap-2 text-white">
            <Trophy className="w-6 h-6 text-yellow-400" />
            <span className="text-xl font-bold">{score}</span>
            <span className="text-sm text-blue-200">points</span>
          </div>
          <div className="flex items-center gap-2 text-white">
            <Zap className="w-6 h-6 text-green-400" />
            <span className="text-xl font-bold">{formatTime(timer)}</span>
          </div>
          <div className="flex items-center gap-2 text-white">
            <span className="text-lg">
              {foundWords.length}/{words.length}
            </span>
            <span className="text-sm text-blue-200">words</span>
          </div>
          <button
            onClick={useHint}
            disabled={hints <= 0}
            className={`flex items-center gap-2 px-4 py-2 rounded-lg font-semibold transition-all ${
              hints > 0
                ? "bg-yellow-500 text-white hover:bg-yellow-600"
                : "bg-gray-500/50 text-gray-300 cursor-not-allowed"
            }`}
          >
            <Lightbulb className="w-5 h-5" />
            Hints: {hints}
          </button>
        </div>

        <div className="grid md:grid-cols-[1fr_300px] gap-6">
          {/* Game Grid */}
          <div className="bg-white/10 backdrop-blur-md rounded-xl p-6">
            <div className="mb-4 flex gap-2 flex-wrap">
              {Object.keys(THEMES).map((t) => (
                <button
                  key={t}
                  onClick={() => setTheme(t)}
                  className={`px-4 py-2 rounded-lg font-semibold capitalize transition-all ${
                    theme === t
                      ? "bg-blue-500 text-white"
                      : "bg-white/20 text-white hover:bg-white/30"
                  }`}
                >
                  {t}
                </button>
              ))}
              <button
                onClick={generateGrid}
                className="ml-auto px-4 py-2 bg-green-500 text-white rounded-lg font-semibold hover:bg-green-600 transition-all flex items-center gap-2"
              >
                <RotateCcw className="w-4 h-4" />
                New Game
              </button>
            </div>

            <div
              className="inline-block bg-white/20 rounded-lg p-2"
              onMouseLeave={() => {
                if (selecting) handleMouseUp();
              }}
            >
              {grid.map((row, i) => (
                <div key={i} className="flex">
                  {row.map((cell, j) => {
                    const cellKey = `${i},${j}`;
                    const isHighlighted = highlightedCells.has(cellKey);
                    const isSelected = selectedCells.includes(cellKey);
                    const isHint = hintCells.has(cellKey);
                    return (
                      <div
                        key={j}
                        className={`w-8 h-8 md:w-10 md:h-10 flex items-center justify-center m-0.5 rounded font-bold text-sm md:text-base cursor-pointer select-none transition-all ${
                          isHighlighted
                            ? "bg-green-500 text-white"
                            : isHint
                            ? "bg-yellow-400 text-white animate-pulse"
                            : isSelected
                            ? "bg-blue-400 text-white"
                            : "bg-white/80 text-gray-800 hover:bg-white"
                        }`}
                        onMouseDown={() => handleMouseDown(i, j)}
                        onMouseEnter={() => handleMouseEnter(i, j)}
                        onMouseUp={handleMouseUp}
                      >
                        {cell}
                      </div>
                    );
                  })}
                </div>
              ))}
            </div>
          </div>

          {/* Word List */}
          <div className="bg-white/10 backdrop-blur-md rounded-xl p-6">
            <h2 className="text-2xl font-bold text-white mb-4">
              Find These Words
            </h2>
            <div className="space-y-2">
              {words.map(({ word }) => (
                <div
                  key={word}
                  className={`p-3 rounded-lg font-semibold transition-all ${
                    foundWords.includes(word)
                      ? "bg-green-500/50 text-white line-through"
                      : "bg-white/20 text-white"
                  }`}
                >
                  {word}
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Victory Modal */}
        {isGameComplete && (
          <div className="fixed inset-0 bg-black/60 flex items-center justify-center p-4 z-50">
            <div className="bg-white rounded-2xl p-8 max-w-md w-full text-center">
              <Trophy className="w-20 h-20 text-yellow-500 mx-auto mb-4" />
              <h2 className="text-3xl font-bold text-gray-800 mb-2">
                Congratulations!
              </h2>
              <p className="text-gray-600 mb-4">You found all the words!</p>
              <div className="bg-blue-50 rounded-lg p-4 mb-6">
                <p className="text-2xl font-bold text-blue-600">
                  {score} points
                </p>
                <p className="text-gray-600">Time: {formatTime(timer)}</p>
              </div>
              <button
                onClick={generateGrid}
                className="px-6 py-3 bg-blue-500 text-white rounded-lg font-semibold hover:bg-blue-600 transition-all"
              >
                Play Again
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default WordSearchGame;
