import { useState, useEffect } from "react";
import { motion } from "framer-motion";
import LinkCard from "./LinkCard";

export default function Shortener() {
  const [url, setUrl] = useState("");
  const [shortLinks, setShortLinks] = useState(() => {
    const saved = localStorage.getItem("shortLinks");
    return saved ? JSON.parse(saved) : [];
  });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");

  useEffect(() => {
    localStorage.setItem("shortLinks", JSON.stringify(shortLinks));
  }, [shortLinks]);

  const shortenUrl = async () => {
    if (!url.trim()) {
      setError("Please enter a valid URL");
      return;
    }
    setError("");
    setLoading(true);

    try {
      // ✅ TinyURL API (works directly, CORS-safe)
      const res = await fetch(
        "https://tinyurl.com/api-create.php?url=" + encodeURIComponent(url)
      );
      const shortUrl = await res.text();

      // TinyURL returns the short URL as plain text
      if (!shortUrl.startsWith("http")) throw new Error("TinyURL failed");

      const newLink = {
        original: url,
        short: shortUrl,
        createdAt: new Date().toLocaleString(),
      };

      setShortLinks([newLink, ...shortLinks]);
      setUrl("");
    } catch (err) {
      try {
        // 🔁 Backup API if TinyURL fails
        const res2 = await fetch(
          `https://api.shrtco.de/v2/shorten?url=${encodeURIComponent(url)}`
        );
        const data2 = await res2.json();
        if (!data2.ok) throw new Error("Backup API failed");

        const newLink = {
          original: url,
          short: data2.result.full_short_link,
          createdAt: new Date().toLocaleString(),
        };
        setShortLinks([newLink, ...shortLinks]);
        setUrl("");
      } catch (err2) {
        setError("Failed to shorten URL. Please try another link.");
      }
    } finally {
      setLoading(false);
    }
  };

  const copyToClipboard = (text) => {
    navigator.clipboard.writeText(text);
  };

  const deleteLink = (short) => {
    setShortLinks(shortLinks.filter((l) => l.short !== short));
  };

  const clearAll = () => {
    if (window.confirm("Are you sure you want to delete all saved links?")) {
      setShortLinks([]);
    }
  };

  return (
    <div className="bg-white/10 dark:bg-gray-800/50 backdrop-blur-md rounded-2xl p-6 shadow-xl text-center">
      <input
        type="url"
        placeholder="Enter your long URL..."
        value={url}
        onChange={(e) => setUrl(e.target.value)}
        className="w-full px-4 py-2 rounded-lg text-gray-800 focus:outline-none"
      />

      <div className="flex justify-center gap-4 mt-4">
        <button
          onClick={shortenUrl}
          className="bg-pink-500 hover:bg-pink-600 px-6 py-2 rounded-lg font-semibold transition disabled:opacity-50"
          disabled={loading}
        >
          {loading ? "Shortening..." : "Shorten URL"}
        </button>

        {shortLinks.length > 0 && (
          <button
            onClick={clearAll}
            className="bg-red-400 hover:bg-red-500 px-6 py-2 rounded-lg font-semibold transition"
          >
            Delete All
          </button>
        )}
      </div>

      {error && <p className="text-red-300 mt-3">{error}</p>}

      <motion.div
        className="mt-8 space-y-4 text-left"
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
      >
        {shortLinks.length === 0 ? (
          <p className="text-gray-400 text-center">No shortened links yet.</p>
        ) : (
          shortLinks.map((link, i) => (
            <LinkCard
              key={i}
              link={link}
              onCopy={copyToClipboard}
              onDelete={deleteLink}
            />
          ))
        )}
      </motion.div>
    </div>
  );
}
