import { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Copy, Trash, Link as LinkIcon } from "lucide-react";

export default function LinkCard({ link, onCopy, onDelete }) {
  const [copied, setCopied] = useState(false);

  useEffect(() => {
    if (copied) {
      const timer = setTimeout(() => setCopied(false), 1800);
      return () => clearTimeout(timer);
    }
  }, [copied]);

  const handleCopy = async () => {
    const textToCopy =
      typeof link.short === "string" && link.short.startsWith("http")
        ? link.short
        : "this is dummy link";

    try {
      await navigator.clipboard.writeText(textToCopy);
      setCopied(true);
      if (onCopy) onCopy(textToCopy);
    } catch (err) {
      console.error("Failed to copy");
    }
  };

  return (
    <motion.div
      layout
      whileHover={{ scale: 1.02 }}
      className="relative p-5 rounded-2xl bg-gradient-to-br from-pink-500/10 via-purple-500/10 to-blue-500/10 border border-pink-300/30 shadow-lg hover:shadow-pink-300/40 transition-all backdrop-blur-md"
    >
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
        <div className="flex items-start gap-3">
          <div className="bg-pink-500/20 p-3 rounded-full">
            <LinkIcon className="text-pink-500" size={22} />
          </div>
          <div>
            <p className="text-sm text-gray-600 truncate max-w-xs">
              {link.original}
            </p>
            <a
              href={link.short}
              target="_blank"
              rel="noreferrer"
              className="text-pink-600 font-semibold hover:text-pink-500 transition"
            >
              {link.short}
            </a>
            <p className="text-xs text-gray-400 mt-1">{link.createdAt}</p>
          </div>
        </div>

        <div className="flex gap-2">
          <motion.button
            whileTap={{ scale: 0.9 }}
            onClick={handleCopy}
            className="p-3 rounded-xl bg-green-500 hover:bg-green-600 text-white shadow-md transition"
          >
            <Copy size={16} />
          </motion.button>

          <motion.button
            whileTap={{ scale: 0.9 }}
            onClick={() => onDelete(link.short)}
            className="p-3 rounded-xl bg-red-500 hover:bg-red-600 text-white shadow-md transition"
          >
            <Trash size={16} />
          </motion.button>
        </div>
      </div>

      <AnimatePresence>
        {copied && (
          <motion.div
            initial={{ opacity: 0, y: -10 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -10 }}
            className="absolute top-3 right-4 bg-pink-500 text-white text-xs px-3 py-1 rounded-full shadow-lg"
          >
            Copied!
          </motion.div>
        )}
      </AnimatePresence>
    </motion.div>
  );
}
