import React, { useState, useEffect } from "react";
import { motion } from "framer-motion";
import Confetti from "react-confetti";
import "./App.css";

const emojis = ["🍕", "🦄", "🎈", "💎", "🌈", "🐱", "🔥", "🍔"];

function App() {
  const [cards, setCards] = useState([]);
  const [flipped, setFlipped] = useState([]);
  const [matched, setMatched] = useState([]);
  const [turn, setTurn] = useState(1);
  const [scores, setScores] = useState({ p1: 0, p2: 0 });
  const [moves, setMoves] = useState(0);
  const [winner, setWinner] = useState("");

  // Initialize cards
  useEffect(() => {
    resetGame();
  }, []);

  const resetGame = () => {
    const shuffled = [...emojis, ...emojis]
      .sort(() => Math.random() - 0.5)
      .map((emoji, index) => ({ id: index, emoji }));

    setCards(shuffled);
    setFlipped([]);
    setMatched([]);
    setTurn(1);
    setScores({ p1: 0, p2: 0 });
    setMoves(0);
    setWinner("");
  };

  const handleFlip = (id) => {
    if (flipped.length === 2 || flipped.includes(id) || matched.includes(id))
      return;
    setFlipped([...flipped, id]);
  };

  useEffect(() => {
    if (flipped.length === 2) {
      setMoves((prev) => prev + 1);
      const [first, second] = flipped;
      if (cards[first].emoji === cards[second].emoji) {
        setMatched([...matched, cards[first].id, cards[second].id]);
        setScores((prev) => ({
          ...prev,
          [turn === 1 ? "p1" : "p2"]: prev[turn === 1 ? "p1" : "p2"] + 1,
        }));
      } else {
        setTimeout(() => setTurn(turn === 1 ? 2 : 1), 800);
      }
      setTimeout(() => setFlipped([]), 900);
    }
  }, [flipped]);

  useEffect(() => {
    if (matched.length === emojis.length * 2) {
      if (scores.p1 > scores.p2) setWinner("🏆 Player 1 Wins!");
      else if (scores.p2 > scores.p1) setWinner("🏆 Player 2 Wins!");
      else setWinner("🤝 It’s a Tie!");
    }
  }, [matched]);

  return (
    <div className="game-container">
      {winner && <Confetti />}
      <h1 className="game-title">🧩 Emoji Match – 2 Player</h1>

      {/* Scoreboard */}
      <div className="scoreboard">
        <div className={`player ${turn === 1 ? "active" : ""}`}>
          <h2>Player 1</h2>
          <p>{scores.p1}</p>
        </div>
        <div className={`player ${turn === 2 ? "active" : ""}`}>
          <h2>Player 2</h2>
          <p>{scores.p2}</p>
        </div>
      </div>

      {/* Cards Grid */}
      <div className="game-grid">
        {cards.map((card, index) => {
          const isFlipped =
            flipped.includes(index) || matched.includes(card.id);
          return (
            <motion.div
              key={card.id}
              className="card-wrapper"
              onClick={() => handleFlip(index)}
            >
              <motion.div
                className={`card ${isFlipped ? "flipped" : ""}`}
                animate={{ rotateY: isFlipped ? 180 : 0 }}
                transition={{ duration: 0.5, ease: "easeInOut" }}
              >
                <div className="card-front">❓</div>
                <div className="card-back">{card.emoji}</div>
              </motion.div>
            </motion.div>
          );
        })}
      </div>

      {/* Info + Controls */}
      <div className="game-info">
        <p>Moves: {moves}</p>
        {winner && <p className="win-text">{winner}</p>}
        <button className="restart-btn" onClick={resetGame}>
          🔁 Restart Game
        </button>
      </div>
    </div>
  );
}

export default App;
