import React, { useEffect, useMemo, useState } from "react";
import { motion } from "framer-motion";

export default function Insights({ items, goal, setGoal, total }) {
  const [currentMonth, setCurrentMonth] = useState("");
  const [goalLocked, setGoalLocked] = useState(false);
  const [goalExists, setGoalExists] = useState(false);

  // ===== Initialize on load =====
  useEffect(() => {
    const now = new Date();
    const monthKey = `${now.getFullYear()}-${now.getMonth() + 1}`;
    setCurrentMonth(monthKey);

    const savedGoalData = JSON.parse(localStorage.getItem("monthlyGoalData"));

    // ✅ First time open: input 0, only Save visible
    if (
      !savedGoalData ||
      savedGoalData.month !== monthKey ||
      !savedGoalData.goal ||
      savedGoalData.goal <= 0
    ) {
      setGoal(0);
      setGoalLocked(false);
      setGoalExists(false);
    } else {
      // ✅ Existing goal for current month
      setGoal(savedGoalData.goal);
      setGoalLocked(true);
      setGoalExists(true);
    }
  }, [setGoal]);

  // ===== Save Goal =====
  const handleSaveGoal = () => {
    if (goal > 0) {
      const goalData = { month: currentMonth, goal };
      localStorage.setItem("monthlyGoalData", JSON.stringify(goalData));
      setGoalLocked(true);
      setGoalExists(true);
    } else {
      alert("Please enter a valid goal greater than 0.");
    }
  };

  // ===== Edit Goal =====
  const handleEditGoal = () => {
    setGoalLocked(false);
  };

  // ===== Reset Goal =====
  const handleReset = () => {
    if (window.confirm("Reset this month’s goal?")) {
      localStorage.removeItem("monthlyGoalData");
      setGoal(0);
      setGoalLocked(false);
      setGoalExists(false);
    }
  };

  // ===== Spending summary =====
  const spentByCategory = useMemo(() => {
    const map = {};
    items.forEach((i) => {
      const c = i.category || "General";
      map[c] = (map[c] || 0) + Math.abs(i.amount);
    });
    return Object.entries(map)
      .map(([name, value]) => ({ name, value }))
      .sort((a, b) => b.value - a.value);
  }, [items]);

  const top = spentByCategory[0] || { name: "—", value: 0 };

  // ===== Progress calculation =====
  const progress =
    goal > 0 ? Math.min(100, Math.round((total / goal) * 100)) : 0;

  const progressColor =
    progress >= 100
      ? "bg-green-500"
      : progress >= 70
      ? "bg-yellow-500"
      : "bg-indigo-500";

  const goalStatus =
    goal === 0
      ? "Set your goal to start tracking!"
      : progress >= 100
      ? " Goal achieved!"
      : total <= 0
      ? "No balance left 😞"
      : `Saved ₹${Math.max(total, 0)} of ₹${goal}`;

  return (
    <motion.div
      className="space-y-6"
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.4 }}
    >
      {/* ===== Monthly Goal Section ===== */}
      <div className="p-4 rounded-xl shadow-md dark:shadow-none bg-white dark:bg-[#04172a] transition">
        <div className="flex justify-between items-center">
          <h3 className="text-base font-semibold text-gray-800 dark:text-gray-200">
            💰 Monthly Goal
          </h3>

          <div className="flex gap-2">
            {/* ===== Button Logic ===== */}
            {!goalExists ? (
              <button
                onClick={handleSaveGoal}
                className="text-xs px-3 py-1 rounded bg-indigo-500 text-white hover:bg-indigo-600 transition"
              >
                Save
              </button>
            ) : goalLocked ? (
              <>
                <button
                  onClick={handleEditGoal}
                  className="text-xs px-3 py-1 rounded bg-green-500 text-white hover:bg-green-600 transition"
                >
                  Edit
                </button>
                <button
                  onClick={handleReset}
                  className="text-xs px-3 py-1 rounded bg-red-500 text-white hover:bg-red-600 transition"
                >
                  Reset
                </button>
              </>
            ) : (
              <button
                onClick={handleSaveGoal}
                className="text-xs px-3 py-1 rounded bg-indigo-500 text-white hover:bg-indigo-600 transition"
              >
                Save
              </button>
            )}
          </div>
        </div>

        {/* ===== Goal Input ===== */}
        <div className="mt-3 flex items-center gap-3">
          <input
            type="number"
            value={goal}
            onChange={(e) => {
              const val = Number(e.target.value);
              if (val < 0) return;
              setGoal(val);
            }}
            disabled={goalLocked}
            className={`p-2 rounded border w-full transition ${
              goalLocked
                ? "bg-gray-100 dark:bg-gray-800 text-gray-500 cursor-not-allowed"
                : "bg-white text-gray-800 dark:bg-black dark:text-white dark:border-gray-700"
            }`}
            placeholder="Set your monthly goal"
          />
        </div>

        {/* ===== Progress Bar ===== */}
        <div className="mt-3 bg-gray-200 dark:bg-gray-700 rounded-full h-3 overflow-hidden">
          <div
            className={`h-3 rounded-full transition-all duration-500 ${progressColor}`}
            style={{ width: `${progress}%` }}
          />
        </div>

        <div className="text-sm mt-2 text-gray-600 dark:text-gray-400 font-medium">
          Progress: {progress}% — {goalStatus}
        </div>
      </div>

      {/* ===== Quick Insights ===== */}
      <div className="p-4 rounded-xl shadow-md bg-white dark:bg-[#04172a] transition">
        <h3 className="text-base font-semibold text-gray-800 dark:text-gray-200 mb-2">
          ⚡ Quick Insights
        </h3>
        <ul className="space-y-2 text-sm text-gray-700 dark:text-gray-300">
          <li>• Total transactions: {items.length}</li>
          <li>• Highest spending category: {top.name}</li>
          <li>• Current balance: ₹{total}</li>
        </ul>
      </div>
    </motion.div>
  );
}
