import React, { useEffect, useMemo, useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import SummaryCard from "./SummaryCard";
import AddTransactionModal from "./AddTransactionModal";
import TransactionList from "./TransactionList";
import Analytics from "./Analytics";
import Insights from "./Insights";

const STORAGE_KEY = "finflow_txns_v1";

export default function Dashboard() {
  const [items, setItems] = useState(() => {
    try {
      return JSON.parse(localStorage.getItem(STORAGE_KEY)) || [];
    } catch {
      return [];
    }
  });
  const [isOpen, setIsOpen] = useState(false);
  const [goal, setGoal] = useState(
    () => parseFloat(localStorage.getItem("finflow_goal")) || 10000
  );

  useEffect(() => {
    localStorage.setItem(STORAGE_KEY, JSON.stringify(items));
  }, [items]);

  useEffect(() => {
    localStorage.setItem("finflow_goal", String(goal));
  }, [goal]);

  const add = (t) => setItems((prev) => [t, ...prev]);
  const del = (id) => setItems((prev) => prev.filter((p) => p.id !== id));

  const total = useMemo(() => items.reduce((s, i) => s + i.amount, 0), [items]);
  const income = useMemo(
    () => items.filter((i) => i.amount > 0).reduce((s, i) => s + i.amount, 0),
    [items]
  );
  const expense = useMemo(
    () => items.filter((i) => i.amount < 0).reduce((s, i) => s + i.amount, 0),
    [items]
  );

  return (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
      <div className="lg:col-span-2 space-y-6">
        <div className="flex items-center justify-between gap-4">
          <div className="flex-1 grid grid-cols-1 sm:grid-cols-3 gap-4">
            <SummaryCard title="Balance" value={total} />
            <SummaryCard title="Income" value={income} positive />
            <SummaryCard title="Expense" value={Math.abs(expense)} negative />
          </div>
          <div className="w-36">
            <button
              onClick={() => setIsOpen(true)}
              className="w-full bg-gradient-to-br from-primary to-indigo-500 text-white px-4 py-2 rounded shadow-lg"
            >
              + Add
            </button>
          </div>
        </div>

        <div className="glass p-4 rounded-lg shadow-sm">
          <Analytics items={items} />
        </div>

        <div className="glass p-4 rounded-lg shadow-sm">
          <TransactionList items={items} onDelete={del} />
        </div>
      </div>

      <aside className="space-y-6">
        <div className="glass p-4 rounded-lg shadow-sm">
          <Insights items={items} goal={goal} setGoal={setGoal} total={total} />
        </div>
      </aside>

      <AnimatePresence>
        {isOpen && (
          <AddTransactionModal onClose={() => setIsOpen(false)} onAdd={add} />
        )}
      </AnimatePresence>
    </div>
  );
}
